package ytstorage

import (
	"context"
	"fmt"
	"reflect"
	"sync"
	"sync/atomic"

	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
)

func save(ctx context.Context, items interface{}, path string, client yt.Client) error {
	iv := reflect.ValueOf(items)
	switch iv.Kind() {
	case reflect.Slice:
		return saveSlice(ctx, iv, path, client)
	case reflect.Chan:
		return saveChan(ctx, iv, path, client)
	default:
		return fmt.Errorf("non-iterable value passed")
	}
}

func saveSlice(ctx context.Context, value reflect.Value, path string, client yt.Client) error {
	itemsSlice := make([]interface{}, value.Len())
	for i := 0; i < value.Len(); i++ {
		itemsSlice[i] = value.Index(i).Interface()
	}
	wrt, err := createTableWriter(ctx, path, client)
	if err != nil {
		return err
	}
	for _, item := range itemsSlice {
		err := wrt.Write(item)
		if err != nil {
			_ = wrt.Rollback()
			return err
		}
	}
	return wrt.Commit()
}

func saveChan(ctx context.Context, value reflect.Value, path string, client yt.Client) error {
	wrt, err := createTableWriter(ctx, path, client)
	if err != nil {
		return err
	}
	for {
		item, ok := value.Recv()
		if ok {
			if item.CanInterface() {
				itemVal := item.Interface()
				if err, isErr := itemVal.(error); isErr {
					_ = wrt.Rollback()
					return err
				}
				err := wrt.Write(itemVal)
				if err != nil {
					_ = wrt.Rollback()
					return err
				}
			} else {
				_ = wrt.Rollback()
				return fmt.Errorf("invalid item obtained from channel")
			}
		} else {
			break
		}
	}
	return wrt.Commit()
}

func createTableWriter(ctx context.Context, path string, client yt.Client) (yt.TableWriter, error) {
	p := ypath.Path(path)
	exists, err := client.NodeExists(ctx, p, nil)
	if err != nil {
		return nil, err
	}
	var opts []yt.WriteTableOption
	if exists {
		opts = append(opts, yt.WithExistingTable())
	}
	return yt.WriteTable(ctx, client, p, opts...)
}

func Load(ctx context.Context, itemType reflect.Type, path string, token string, proxy string) ([]interface{}, error) {
	client, err := ythttp.NewClient(&yt.Config{
		Proxy: proxy,
		Token: token,
	})
	if err != nil {
		return nil, err
	}
	rdr, err := client.ReadTable(ctx, ypath.Path(path), nil)
	if err != nil {
		return nil, err
	}
	defer rdr.Close()
	var result []interface{}
	for rdr.Next() {
		item := reflect.New(itemType).Interface()
		err = rdr.Scan(item)
		if err != nil {
			return nil, err
		}
		result = append(result, item)
	}
	return result, rdr.Err()
}

func Link(ctx context.Context, path string, link string, token string, proxy string) error {
	client, err := ythttp.NewClient(&yt.Config{
		Proxy: proxy,
		Token: token,
	})
	if err != nil {
		return err
	}
	_, err = client.LinkNode(ctx, ypath.Path(path), ypath.Path(link), &yt.LinkNodeOptions{
		Force: true,
	})
	return err
}

func Save(ctx context.Context, items interface{}, path string, minWritesToSucceed int, token string, proxies ...string) error {
	var successCount int64 = 0
	wg := sync.WaitGroup{}
	errors := make([]error, len(proxies))
	for i, proxy := range proxies {
		client, err := ythttp.NewClient(&yt.Config{
			Proxy: proxy,
			Token: token,
		})
		if err != nil {
			errors[i] = err
			continue
		}
		wg.Add(1)
		go func(errIndex int) {
			defer wg.Done()
			if err := save(ctx, items, path, client); err != nil {
				errors[errIndex] = err
			} else {
				atomic.AddInt64(&successCount, 1)
			}
			client.Stop()
		}(i)
	}
	wg.Wait()
	if int(successCount) < minWritesToSucceed {
		return fmt.Errorf("save failed: %v", errors)
	} else {
		return nil
	}
}
