package ru.yandex.travel.hotels.busbroker;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Objects;
import java.util.Set;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.hotels.proto.OfferInvalidationServiceV1Grpc;
import ru.yandex.travel.hotels.proto.TPingRpcReq;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.tvm.TvmWrapper;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(BusBrokerProperties.class)
public class BusBrokerConfiguration {
    private final BusBrokerProperties properties;

    @Bean("BusBrokerClient")
    @ConditionalOnProperty("busbroker.enabled")
    public BusBrokerClient getBusBroker(@Qualifier("BusBrokerGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory,
                                        Retry retryHelper) {
        return new DefaultBusBrokerClient(haGrpcChannelFactory, retryHelper);
    }

    @Bean("BusBrokerClient")
    @ConditionalOnMissingBean(name = "BusBrokerClient")
    public BusBrokerClient getBusBrokerNoop() {
        return new NoopBusBrokerClient();
    }


    @Bean("BusBrokerSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean("BusBrokerGrpcChannelFactory")
    @ConditionalOnProperty("busbroker.enabled")
    public HAGrpcChannelFactory haGrpcChannelFactory(@Qualifier("BusBrokerSupplier") ChannelSupplier channelSupplier, TvmWrapper tvm) {
        return HAGrpcChannelFactory.Builder.newBuilder().withPingProducer(
                channel -> buildCompletableFuture(OfferInvalidationServiceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getIsReady() ? ChannelState.READY : ChannelState.NOT_READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(target -> createChannel(target, new TvmHeaderClientHelper(tvm)))
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        var clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        var interceptors = new ArrayList<ClientInterceptor>();
        interceptors.add(new LoggingClientInterceptor(clientFqdn, target,
                Set.of(OfferInvalidationServiceV1Grpc.getPingMethod().getFullMethodName())));
        interceptors.add(new DefaultTimeoutClientInterceptor(properties.getTimeout()));
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvmDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }
}
