package ru.yandex.travel.hotels.busbroker;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.Timestamp;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.SpeculativeRetryStrategy;
import ru.yandex.travel.hotels.busbroker.model.DateRange;
import ru.yandex.travel.hotels.busbroker.model.OfferInvalidationRsp;
import ru.yandex.travel.hotels.busbroker.model.OfferInvalidatonReq;
import ru.yandex.travel.hotels.proto.EOfferInvalidationSource;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.OfferInvalidationServiceV1Grpc;
import ru.yandex.travel.hotels.proto.TCheckInCheckOutFilter;
import ru.yandex.travel.hotels.proto.TFilter;
import ru.yandex.travel.hotels.proto.THotelId;
import ru.yandex.travel.hotels.proto.TOfferInvalidationReq;
import ru.yandex.travel.hotels.proto.TOfferInvalidationResp;
import ru.yandex.travel.hotels.proto.TTargetIntervalFilter;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
public class DefaultBusBrokerClient implements BusBrokerClient {
    private final HAGrpcChannelFactory haGrpcChannelFactory;
    private final RetryRateLimiter rateLimiter;
    private final Retry retryHelper;

    public DefaultBusBrokerClient(HAGrpcChannelFactory haGrpcChannelFactory,
                                  Retry retryHelper) {
        this.haGrpcChannelFactory = haGrpcChannelFactory;
        this.rateLimiter = new RetryRateLimiter(0.3);
        this.retryHelper = retryHelper;
    }

    public CompletableFuture<OfferInvalidationRsp> invalidateOffersByTargetInterval(EPartnerId partnerId,
                                                                                    String originalId,
                                                                                    LocalDate fromInclusive,
                                                                                    LocalDate toInclusive,
                                                                                    EOfferInvalidationSource offerInvalidationSource) {
        var req = new OfferInvalidatonReq();
        req.setHotelId(THotelId.newBuilder().setPartnerId(partnerId).setOriginalId(originalId).build());
        req.setFilters(List.of(new OfferInvalidatonReq.Filter(new OfferInvalidatonReq.TargetIntervalFilter(fromInclusive, toInclusive))));
        req.setOfferInvalidationSource(offerInvalidationSource);
        return invalidateOffersRaw(req);
    }

    public CompletableFuture<OfferInvalidationRsp> invalidateOffersByCheckInOut(EPartnerId partnerId,
                                                                                String originalId,
                                                                                LocalDate checkIn,
                                                                                LocalDate checkOut,
                                                                                EOfferInvalidationSource offerInvalidationSource) {
        return invalidateOffersByCheckInOut(
                partnerId,
                originalId,
                Collections.singletonList(new DateRange(checkIn, checkOut)),
                offerInvalidationSource
        );
    }

    public CompletableFuture<OfferInvalidationRsp> invalidateOffersByCheckInOut(EPartnerId partnerId,
                                                                                String originalId,
                                                                                List<DateRange> dateRanges,
                                                                                EOfferInvalidationSource offerInvalidationSource) {
        var req = new OfferInvalidatonReq();
        req.setHotelId(THotelId.newBuilder()
                .setPartnerId(partnerId)
                .setOriginalId(originalId)
                .build()
        );
        List<OfferInvalidatonReq.Filter> filters = new ArrayList<>();
        for (DateRange dateRange : dateRanges) {
            filters.add(new OfferInvalidatonReq.Filter(
                    new OfferInvalidatonReq.CheckInCheckOutFilter(dateRange.getCheckInDate(), dateRange.getCheckOutDate())
            ));
        }
        req.setFilters(filters);
        req.setOfferInvalidationSource(offerInvalidationSource);
        return invalidateOffersRaw(req);
    }

    public CompletableFuture<OfferInvalidationRsp> invalidateOffersRaw(OfferInvalidatonReq offerInvalidatonReq) {
        return retryHelper.withSpeculativeRetry("OfferInvalidationService::InvalidateOffers",
                req -> buildCompletableFuture(OfferInvalidationServiceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel()).invalidateOffers(req)),
                prepareProtoReq(offerInvalidatonReq),
                SpeculativeRetryStrategy.<TOfferInvalidationResp>builder()
                        .shouldRetryOnException(e -> true)
                        .validateResult(resp -> {
                            if (resp.hasError()) {
                                throw new RuntimeException(String.format("Error in response: %s",
                                        resp.getError().getMessage()));
                            }
                            if (!resp.hasResult()) {
                                throw new RuntimeException("No result in response");
                            }
                        })
                        .timeout(Duration.ofSeconds(3))
                        .retryDelay(Duration.ofMillis(400))
                        .numRetries(5)
                        .build(),
                rateLimiter)
                .thenApply(protoRsp -> {
                    OfferInvalidationRsp rsp = new OfferInvalidationRsp();
                    if (protoRsp.hasError()) {
                        rsp.setError(protoRsp.getError().getMessage());
                    }
                    return rsp;
                });
    }

    private TOfferInvalidationReq prepareProtoReq(OfferInvalidatonReq offerInvalidatonReq) {
        var builder = TOfferInvalidationReq.newBuilder();

        Preconditions.checkArgument(offerInvalidatonReq.getHotelId() == null || offerInvalidatonReq.getPermalink() == null,
                "Invalidation request should contain only one of permalink and hotelId");
        if (offerInvalidatonReq.getHotelId() != null) {
            builder.setHotelId(offerInvalidatonReq.getHotelId());
        } else if (offerInvalidatonReq.getPermalink() != null) {
            builder.setPermalink(offerInvalidatonReq.getPermalink().asLong());
        } else {
            throw new IllegalArgumentException("Invalidation request should contain either permalink or hotelId");
        }

        builder.setCurrency(Objects.requireNonNullElse(offerInvalidatonReq.getCurrency(), ECurrency.C_RUB));
        builder.setTimestamp(Timestamp.newBuilder()
                .setSeconds(Objects.requireNonNullElse(offerInvalidatonReq.getTimestamp(), Instant.now()).getEpochSecond())
                .build());

        builder.addAllFilters(offerInvalidatonReq.getFilters().stream().map(filter -> {
            if (filter.getCheckInCheckOutFilter() != null) {
                var checkInCheckOutFilterBuilder = TCheckInCheckOutFilter.newBuilder();
                if (filter.getCheckInCheckOutFilter().getCheckInDate() != null) {
                    checkInCheckOutFilterBuilder.setCheckInDate(filter.getCheckInCheckOutFilter().getCheckInDate().toString());
                }
                if (filter.getCheckInCheckOutFilter().getCheckOutDate() != null) {
                    checkInCheckOutFilterBuilder.setCheckOutDate(filter.getCheckInCheckOutFilter().getCheckOutDate().toString());
                }
                return TFilter.newBuilder().setCheckInCheckOutFilter(checkInCheckOutFilterBuilder.build()).build();
            } else if (filter.getTargetIntervalFilter() != null) {
                var targetIntervalFilterBuilder = TTargetIntervalFilter.newBuilder();
                if (filter.getTargetIntervalFilter().getDateFromInclusive() != null) {
                    targetIntervalFilterBuilder.setDateFromInclusive(filter.getTargetIntervalFilter().getDateFromInclusive().toString());
                }
                if (filter.getTargetIntervalFilter().getDateToInclusive() != null) {
                    targetIntervalFilterBuilder.setDateToInclusive(filter.getTargetIntervalFilter().getDateToInclusive().toString());
                }
                return TFilter.newBuilder().setTargetIntervalFilter(targetIntervalFilterBuilder.build()).build();
            } else {
                throw new IllegalArgumentException("Offer invalidation filter should has either checkin/out filter or" +
                        " target interval filter");
            }
        }).collect(Collectors.toUnmodifiableList()));

        Preconditions.checkArgument(offerInvalidatonReq.getOfferInvalidationSource() != null, "Invalidation source is" +
                " required");
        builder.setOfferInvalidationSource(offerInvalidatonReq.getOfferInvalidationSource());

        return builder.build();
    }
}

