package ru.yandex.travel.hotels.cluster_permalinks;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;

@Slf4j
@Component
@EnableConfigurationProperties(ClusterPermalinkServiceProperties.class)
public class ClusterPermalinkService implements ClusterPermalinkDataProvider {
    private static final int INDEX_STRUCTURE_VERSION = 1;
    private static final String FIELD_CLUSTER_PERMALINK_INDEX = "cpi";
    private static final String FIELD_CLUSTER_PERMALINK_STORE = "cps";
    private static final String FIELD_PERMALINK_INDEX = "pi";
    private static final String FIELD_PERMALINK_STORE = "ps";
    private static final int BATCH_SIZE = 100;

    private final YtLuceneIndex permalinkLuceneIndex;

    private final Counter clusterPermalinkFoundCounter;
    private final Counter clusterPermalinkNotFoundCounter;
    private final Counter allPermalinksOfClusterFoundCounter;
    private final Counter allPermalinksOfClusterNotFoundCounter;

    public ClusterPermalinkService(ClusterPermalinkServiceProperties params) {
        if (params.isEnabled()) {
            permalinkLuceneIndex = new YsonYtLuceneIndex(params, "ClusterPermalinks", INDEX_STRUCTURE_VERSION, (row) -> {
                Document document = new Document();
                document.add(new LongPoint(FIELD_CLUSTER_PERMALINK_INDEX, row.getLong("cluster_permalink")));
                document.add(new StoredField(FIELD_CLUSTER_PERMALINK_STORE, row.getLong("cluster_permalink")));
                document.add(new LongPoint(FIELD_PERMALINK_INDEX, row.getLong("permalink")));
                document.add(new StoredField(FIELD_PERMALINK_STORE, row.getLong("permalink")));
                return Collections.singletonList(document);
            });
        } else {
            log.warn("Permalinks index disabled");
            permalinkLuceneIndex = null;
        }
        clusterPermalinkFoundCounter = Counter.builder("cluster_permalinks.permalink2cluster.found").register(Metrics.globalRegistry);
        clusterPermalinkNotFoundCounter = Counter.builder("cluster_permalinks.permalink2cluster.notFound").register(Metrics.globalRegistry);
        allPermalinksOfClusterFoundCounter = Counter.builder("cluster_permalinks.permalink2all.found").register(Metrics.globalRegistry);
        allPermalinksOfClusterNotFoundCounter = Counter.builder("cluster_permalinks.permalink2all.notFound").register(Metrics.globalRegistry);
    }

    @PostConstruct
    public void init() {
        if (permalinkLuceneIndex != null) {
            permalinkLuceneIndex.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (permalinkLuceneIndex != null) {
            permalinkLuceneIndex.stop();
        }
    }

    public boolean isReady() {
        return permalinkLuceneIndex == null || permalinkLuceneIndex.isReady();
    }

    public Permalink findClusterPermalink(Permalink permalink) {
        Permalink clusterPermalink;
        if (permalinkLuceneIndex != null) {
            clusterPermalink = permalinkLuceneIndex.search(searcher -> {
                Query query = LongPoint.newExactQuery(FIELD_PERMALINK_INDEX, permalink.asLong());
                TopDocs topDocs = searcher.search(query, 1);
                if (topDocs.totalHits > 0) {
                    Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                    return Permalink.of(document.getField(FIELD_CLUSTER_PERMALINK_STORE).numericValue().longValue());
                }
                return null;
            });
        } else {
            clusterPermalink = null;
        }
        if (clusterPermalink != null) {
            clusterPermalinkFoundCounter.increment();
            return clusterPermalink;
        }
        clusterPermalinkNotFoundCounter.increment();
        return permalink;
    }

    public List<Permalink> findAllPermalinksOfCluster(Permalink permalink) {
        List<Permalink> permalinksOfCluster;
        if (permalinkLuceneIndex != null) {
            var clusterPermalink = findClusterPermalink(permalink);
            permalinksOfCluster = permalinkLuceneIndex.search(searcher -> {
                Query query = LongPoint.newExactQuery(FIELD_CLUSTER_PERMALINK_INDEX, clusterPermalink.asLong());

                var result = new HashSet<Permalink>();
                result.add(permalink);

                var topDocs = searcher.search(query, BATCH_SIZE);
                while (true) {
                    for (var topDoc : topDocs.scoreDocs) {
                        var doc = searcher.doc(topDoc.doc);
                        result.add(Permalink.of(doc.getField(FIELD_PERMALINK_STORE).numericValue().longValue()));
                    }
                    if (topDocs.scoreDocs.length < BATCH_SIZE) {
                        break;
                    }
                    topDocs = searcher.searchAfter(topDocs.scoreDocs[topDocs.scoreDocs.length - 1], query, BATCH_SIZE);

                }
                return result.stream().collect(Collectors.toUnmodifiableList());
            });
        } else {
            permalinksOfCluster = null;
        }

        if (permalinksOfCluster != null) {
            allPermalinksOfClusterFoundCounter.increment();
            return permalinksOfCluster;
        }
        allPermalinksOfClusterNotFoundCounter.increment();
        return List.of(permalink);
    }
}
