package ru.yandex.travel.hotels.common;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDateTime;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.Setter;


@Getter
@AllArgsConstructor
@NoArgsConstructor(access = AccessLevel.PRIVATE)
@Setter(AccessLevel.PRIVATE)
@EqualsAndHashCode
public class CancellationPenalty {
    private LocalDateTime startsAt;
    private LocalDateTime endsAt;
    private Type type;
    @JsonSerialize(using = RoundingDecimalSerializer.class)
    private BigDecimal amount;
    private String currency;

    public static CancellationPenalty noPenalty(LocalDateTime till) {
        return noPenalty(null, till);
    }

    public static CancellationPenalty noPenalty(LocalDateTime from, LocalDateTime till) {
        CancellationPenalty res = new CancellationPenalty();
        res.setType(Type.NO_PENALTY);
        res.setStartsAt(from);
        res.setEndsAt(till);
        return res;
    }

    public static CancellationPenalty fullPenalty(LocalDateTime since) {
        CancellationPenalty res = new CancellationPenalty();
        res.setType(Type.FULL_PRICE);
        res.setStartsAt(since);
        return res;
    }

    public static CancellationPenalty somePenalty(LocalDateTime since, LocalDateTime till, BigDecimal amount,
                                                  String currency) {
        CancellationPenalty res = new CancellationPenalty();
        res.setType(Type.SOME_PENALTY);
        res.setStartsAt(since);
        res.setEndsAt(till);
        res.setAmount(amount);
        res.setCurrency(currency);
        return res;
    }

    public boolean matchesDateTime(LocalDateTime target) {
        if (startsAt == null && endsAt == null) {
            // For now this should never happen
            return true;
        }
        if (startsAt == null) {
            return endsAt.isAfter(target);
        }
        if (endsAt == null) {
            return startsAt.isBefore(target) || startsAt.isEqual(target);
        }
        // we need full coverage (without holes) of the timeline here: [null,t1) + [t1, t2) + [t2,null)
        return (startsAt.isBefore(target) || startsAt.isEqual(target)) && endsAt.isAfter(target);
    }

    @RequiredArgsConstructor(access = AccessLevel.PRIVATE)
    public enum Type {
        NO_PENALTY("no_penalty"),
        SOME_PENALTY("some_penalty"),
        FULL_PRICE("full_price");

        private final String value;

        public static CancellationPenalty.Type fromValue(String text) {
            for (CancellationPenalty.Type b : CancellationPenalty.Type.values()) {
                if (String.valueOf(b.value).equals(text)) {
                    return b;
                }
            }
            return null;
        }
    }

    public static class RoundingDecimalSerializer extends JsonSerializer<BigDecimal> {
        @Override
        public void serialize(BigDecimal value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            gen.writeString(value.setScale(2, RoundingMode.HALF_UP).toString());
        }
    }


}
