package ru.yandex.travel.hotels.common.promo.mir;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.stream.Stream;

import com.google.protobuf.DoubleValue;
import com.google.protobuf.UInt32Value;
import io.micrometer.core.instrument.Metrics;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.proto.EMirEligibility;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TMirPromoStatus;

public class MirEligibilityService {
    static final String EUROPE_MOSCOW_ZONE_ID = "Europe/Moscow";

    private final MirProperties properties;
    private final MirHotelList whitelist;
    private final String meterPrefix;

    public MirEligibilityService(MirProperties properties, MirHotelList whitelist, String meterPrefix) {
        this.properties = properties;
        this.meterPrefix = meterPrefix;
        this.whitelist = whitelist;
        validateStages();
    }

    public TMirPromoStatus checkEligibility(EPartnerId partnerId, String originalId,
                                            Instant bookingInstant, LocalDate checkin, LocalDate checkout,
                                            BigDecimal priceAfterPlusWithdraw) {
        TMirPromoStatus status = checkEligibilityImpl(partnerId, originalId, bookingInstant, checkin, checkout,
                priceAfterPlusWithdraw);
        if (StringUtils.isNotBlank(meterPrefix)) {
            Metrics.counter(meterPrefix, "partner", partnerId.toString(), "status",
                    status.getEligibility().toString()).increment();
        }
        return status;
    }

    public boolean isPromoActive(Instant bookingInstant) {
        return getActiveStage(bookingInstant) != null;
    }

    private MirProperties.Stage getActiveStage(Instant bookingInstant) {
        for (var stage : properties.getStages().values()) {
            if (bookingInstant.isAfter((stage.getStageStarts())) && bookingInstant.isBefore(stage.getStageEnds())) {
                return stage;
            }
        }
        return null;
    }

    private void validateStages() {
        if (properties.getStages() != null) {
            final var starts =
                    this.properties.getStages().values().stream().map(stage -> Tuple2.tuple(stage.getStageStarts(), 1));
            final var ends = this.properties.getStages().values().stream().map(stage -> Tuple2.tuple(stage.getStageEnds()
                    , -1));
            var ignored =
                    Stream.concat(starts, ends).sorted(Comparator.comparing(Tuple2::get1)).map(Tuple2::get2).reduce((prev, next) -> {
                        var v = prev + next;
                        if (v > 1) {
                            throw new IllegalArgumentException("Several mir promo stages are active at the same time");
                        }
                        if (v < 0) {
                            throw new IllegalArgumentException("Illegal mir configuration");
                        }
                        return v;
                    });
        }
    }

    private TMirPromoStatus checkEligibilityImpl(EPartnerId partnerId, String originalId, Instant bookingInstant,
                                                 LocalDate checkin, LocalDate checkout, BigDecimal priceAfterPlusWithdraw) {
        // first, check if the promo is active:
        var activeStage = getActiveStage(bookingInstant);
        if (activeStage == null) {
            return TMirPromoStatus.newBuilder().setEligibility(EMirEligibility.ME_WRONG_BOOKING_DATE).build();
        }
        if (!isPromoActive(bookingInstant)) {
            return TMirPromoStatus.newBuilder().setEligibility(EMirEligibility.ME_WRONG_BOOKING_DATE).build();
        }
        // then check if the hotel is whitelisted:
        String mirId = whitelist.getMirIdIfEnabled(partnerId, originalId);
        if (mirId == null) {
            return TMirPromoStatus.newBuilder().setEligibility(EMirEligibility.ME_BLACKLISTED).build();
        }

        // then check for LOS and max checkin/checkout
        if (ChronoUnit.DAYS.between(checkin, checkout) < activeStage.getMinLOS()) {
            return TMirPromoStatus.newBuilder()
                    .setEligibility(EMirEligibility.ME_WRONG_LOS)
                    .setExpiresAt(ProtoUtils.fromInstant(activeStage.getStageEnds()))
                    .build();
        }
        if (activeStage.getLastCheckout() != null && checkout.atTime(23, 59).toInstant(ZoneOffset.of("+3")).isAfter(activeStage.getLastCheckout())) {
            return TMirPromoStatus.newBuilder()
                    .setEligibility(EMirEligibility.ME_WRONG_STAY_DATES)
                    .setExpiresAt(ProtoUtils.fromInstant(activeStage.getStageEnds()))
                    .build();
        }
        if (activeStage.getFirstCheckin() != null && checkin.atStartOfDay(ZoneId.of(EUROPE_MOSCOW_ZONE_ID)).toInstant().isBefore(activeStage.getFirstCheckin())) {
            return TMirPromoStatus.newBuilder()
                    .setEligibility(EMirEligibility.ME_WRONG_STAY_DATES)
                    .setExpiresAt(ProtoUtils.fromInstant(activeStage.getStageEnds()))
                    .build();
        }
        int cb = activeStage.getMaxCashbackAmount().min(priceAfterPlusWithdraw.multiply(activeStage.getCashbackRate()))
                .setScale(0, RoundingMode.DOWN).intValue();

        return TMirPromoStatus.newBuilder()
                .setEligibility(EMirEligibility.ME_ELIGIBLE)
                .setCashbackAmount(UInt32Value.newBuilder().setValue(cb).build())
                .setCashbackRate(DoubleValue.newBuilder().setValue(activeStage.getCashbackRate().doubleValue()).build())
                .setExpiresAt(ProtoUtils.fromInstant(activeStage.getStageEnds()))
                .setMirId(mirId)
                .build();
    }
}
