package ru.yandex.travel.hotels.common.refunds;

import java.time.Instant;
import java.time.ZoneOffset;
import java.time.temporal.TemporalAmount;

import javax.annotation.Nonnull;

import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;
import lombok.Builder;
import lombok.Value;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.CancellationPenalty;

@Value
@Builder(toBuilder = true)
@JsonDeserialize(builder = RefundRule.RefundRuleBuilder.class)
public class RefundRule {
    private RefundType type;
    private Money penalty;
    private Instant startsAt;
    private Instant endsAt;

    @Deprecated
    public static RefundRule fromCancellationPenalty(CancellationPenalty penalty) {
        var builder = RefundRule.builder();
        if (penalty.getStartsAt() != null) {
            builder.startsAt(penalty.getStartsAt().toInstant(ZoneOffset.UTC));
        }
        if (penalty.getEndsAt() != null) {
            builder.endsAt(penalty.getEndsAt().toInstant(ZoneOffset.UTC));
        }
        if (penalty.getAmount() != null) {
            builder.penalty(Money.of(penalty.getAmount(), ProtoCurrencyUnit.fromCurrencyCode(penalty.getCurrency())));
        }
        switch (penalty.getType()) {
            case NO_PENALTY:
                builder.type(RefundType.FULLY_REFUNDABLE);
                break;
            case SOME_PENALTY:
                builder.type(RefundType.REFUNDABLE_WITH_PENALTY);
                break;
            case FULL_PRICE:
                builder.type(RefundType.NON_REFUNDABLE);
                break;
        }
        return builder.build();
    }

    public boolean matchesInstant(@Nonnull Instant instant) {
        return (startsAt == null || startsAt.equals(instant) || startsAt.isBefore(instant)) &&
                (endsAt == null || endsAt.isAfter(instant));
    }

    @Deprecated
    public CancellationPenalty toCancellationPenalty() {
        CancellationPenalty.Type cpType = null;
        switch (type) {
            case NON_REFUNDABLE:
                cpType = CancellationPenalty.Type.FULL_PRICE;
                break;
            case REFUNDABLE_WITH_PENALTY:
                cpType = CancellationPenalty.Type.SOME_PENALTY;
                break;
            case FULLY_REFUNDABLE:
                cpType = CancellationPenalty.Type.NO_PENALTY;
                break;
        }

        return new CancellationPenalty(
                startsAt == null ? null : startsAt.atZone(ZoneOffset.UTC).toLocalDateTime(),
                endsAt == null ? null : endsAt.atZone(ZoneOffset.UTC).toLocalDateTime(),
                cpType,
                penalty == null ? null : penalty.getNumberStripped(),
                penalty == null ? null : penalty.getCurrency().getCurrencyCode());
    }

    @JsonPOJOBuilder(withPrefix = "")
    public static class RefundRuleBuilder {
    }

    public RefundRule shift(TemporalAmount plus) {
        var builder = RefundRule.builder()
            .type(type)
            .penalty(penalty);
        if (startsAt != null) {
            builder.startsAt(startsAt.plus(plus));
        }
        if (endsAt != null) {
            builder.endsAt(endsAt.plus(plus));
        }
        return builder.build();
    }
}
