package ru.yandex.travel.hotels.common.refunds;

import java.math.RoundingMode;
import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;
import lombok.Builder;
import lombok.Singular;
import lombok.Value;
import org.javamoney.moneta.Money;


@Value
@Builder(toBuilder = true)
@JsonDeserialize(builder = RefundRules.RefundRulesBuilder.class)
public class RefundRules {
    @Singular
    private List<RefundRule> rules;

    @JsonIgnore
    public boolean isRefundable() {
        if (rules == null || rules.isEmpty()) {
            return false;
        }
        return rules.stream().anyMatch(r -> r.getType() != RefundType.NON_REFUNDABLE);
    }

    @JsonIgnore
    public boolean isRefundableAt(Instant instant) {
        var rule = getRuleAtInstant(instant);
        return (rule != null && rule.getType() != RefundType.NON_REFUNDABLE);
    }

    @JsonIgnore
    public boolean isFullyRefundableAt(Instant instant) {
        var rule = getRuleAtInstant(instant);
        return (rule != null && rule.getType() == RefundType.FULLY_REFUNDABLE);
    }

    @JsonIgnore
    public boolean isFullyRefundable() {
        return rules.stream().anyMatch(r -> r.getType() == RefundType.FULLY_REFUNDABLE);
    }

    @JsonIgnore
    public RefundRule getRuleAtInstant(Instant instant) {
        if (rules == null || rules.isEmpty()) {
            return RefundRule.builder().type(RefundType.NON_REFUNDABLE).build();
        }
        return rules.stream().filter(r -> r.matchesInstant(instant)).findFirst()
                .orElseGet(() -> RefundRule.builder().type(RefundType.NON_REFUNDABLE).build());
    }

    @JsonIgnore
    public Instant getFullyRefundableTill() {
        if (rules == null || rules.isEmpty()) {
            return null;
        }
        Optional<Instant> lastRefundableInstant =
                rules.stream().filter(r -> r.getType() == RefundType.FULLY_REFUNDABLE && r.getEndsAt() != null)
                        .map(RefundRule::getEndsAt).max(Comparator.naturalOrder());
        return lastRefundableInstant.orElse(null);
    }

    @JsonIgnore
    public int getRuleIndexAtInstant(Instant instant) {
        if (rules != null) {
            for (int i = 0; i < rules.size(); i++) {
                if (rules.get(i).matchesInstant(instant)) {
                    return i;
                }
            }
        }
        return -1;
    }

    public RefundRules actualize() {
        if (rules == null) {
            return this.toBuilder().clearRules().build();
        }
        Instant now = Instant.now();
        return this.toBuilder().clearRules()
                .rules(this.rules.stream()
                        .filter(r -> r.getEndsAt() == null || r.getEndsAt().isAfter(now) || r.getEndsAt().equals(now))
                        .collect(Collectors.toUnmodifiableList()))
                .build();
    }

    public RefundRules withNonRefundableExtra(Money extraAmount, Instant instant) {
        var builder = RefundRules.builder();
        for (var rule : this.getRules()) {
            if (rule.getEndsAt() != null && rule.getEndsAt().isBefore(instant)) {  // first copy the "old" rules
                builder.rule(rule);
                continue;
            }
            switch (rule.getType()) {
                case FULLY_REFUNDABLE:
                    builder.rule(rule.toBuilder()
                            .type(RefundType.REFUNDABLE_WITH_PENALTY)
                            .penalty(extraAmount)
                            .build());
                    continue;
                case REFUNDABLE_WITH_PENALTY:
                    builder.rule(rule.toBuilder()
                            .penalty(rule.getPenalty().add(extraAmount))
                            .build());
                    continue;
                case NON_REFUNDABLE:
                    builder.rule(rule);
            }
        }
        return builder.build();
    }

    public RefundRules withFullyRefundableExtra(Money extraAmount, Instant instant) {
        return this.toBuilder().build();
    }

    public RefundRules withProportionallyRefundableExtra(Money extraAmount, Money oldTotalAmount, Instant instant) {
        var builder = RefundRules.builder();
        for (var rule : this.getRules()) {
            if (rule.getEndsAt() != null && rule.getEndsAt().isBefore(instant)) {  // first copy the "old" rules
                builder.rule(rule);
                continue;
            }
            switch (rule.getType()) {
                case FULLY_REFUNDABLE:
                case NON_REFUNDABLE:
                    builder.rule(rule);
                    continue;
                case REFUNDABLE_WITH_PENALTY:
                    var penalty = rule.getPenalty().getNumberStripped();
                    var total = oldTotalAmount.getNumberStripped();
                    var share = penalty.divide(total, 4, RoundingMode.HALF_UP);
                    builder.rule(rule.toBuilder()
                            .penalty(rule.getPenalty().add(extraAmount.multiply(share))).build());
                    continue;
            }
        }
        return builder.build();
    }

    @JsonPOJOBuilder(withPrefix = "")
    public static class RefundRulesBuilder {
    }
}
