package ru.yandex.travel.hotels.common.schedule;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import org.javamoney.moneta.Money;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

public class PaymentScheduleBuilder {
    public static final BigDecimal STANDARD_RATE = BigDecimal.valueOf(0.1);

    public static OfferPaymentSchedule twoPaymentSchedule(Instant createdAt, Instant checkinDayBeginsAt,
                                                          Money totalAmount,
                                                          Money minFirstPayment, Money maxFirstPayment,
                                                          RefundRules refundRules,
                                                          Duration minPaymentInterval,
                                                          Duration refundSafetyWindow) {
        if (!refundRules.isRefundable()) {
            return null;
        }
        if (totalAmount.isLessThanOrEqualTo(minFirstPayment)) {
            return null;
        }
        Instant nonRefInstant = refundRules.getRules().stream()
                .filter(r -> r.getType() == RefundType.NON_REFUNDABLE)
                .min(Comparator.comparing(RefundRule::getStartsAt))
                .orElseThrow()
                .getStartsAt();

        checkRulesInstantsSequence(refundRules.getRules());

        List<Tuple2<Instant, Money>> points = refundRules.getRules().stream()
                .filter(r -> r.getEndsAt() == null ||
                        r.getEndsAt().minus(minPaymentInterval).isAfter(createdAt))
                .map(r -> {
                    Instant p = r.getStartsAt();
                    if (p == null) {
                        p = createdAt;
                    } else {
                        p = p.minus(refundSafetyWindow);
                    }
                    Money amount;
                    switch (r.getType()) {
                        case FULLY_REFUNDABLE:
                            amount = Money.zero(totalAmount.getCurrency());
                            break;
                        case NON_REFUNDABLE:
                            amount = totalAmount;
                            break;
                        case REFUNDABLE_WITH_PENALTY:
                            amount = r.getPenalty();
                            break;
                        default:
                            throw new IllegalStateException();
                    }
                    return Tuple2.tuple(p, amount);
                })
                .sorted(Comparator.comparing(Tuple2::get1))
                .collect(Collectors.toList());

        checkMonotonicity(points);

        if (points.size() == 0) {
            return null;
        }

        Money firstPayment = points.get(0).get2();
        if (firstPayment.isGreaterThan(maxFirstPayment)) { // first payment is too large: deferred payment is
            // not possible
            return null;
        }
        if (firstPayment.isLessThan(minFirstPayment)) {
            firstPayment = minFirstPayment;
        }
        var firstPaymentOptions = points.stream()
                .skip(1)
                .filter(p -> p.get2().isLessThanOrEqualTo(maxFirstPayment))
                .map(t -> {
                    Money v = t.get2();
                    if (v.isLessThan(minFirstPayment)) {
                        return minFirstPayment;
                    } else {
                        return v;
                    }
                })
                .collect(Collectors.toList());
        firstPaymentOptions.add(0, firstPayment);

        var secondsFromFirstToNonRef = Duration.between(createdAt, nonRefInstant).toSeconds();

        Money bestFirstPaymentAmount = null;
        Instant bestLastPaymentInstant = null;
        BigDecimal bestMetric = null;
        for (Money firstPaymentAmount : firstPaymentOptions) {
            var lastPaymentAmount = totalAmount.subtract(firstPaymentAmount);
            var lastPaymentInstant =
                    points.stream()
                            .filter(p -> p.get2().isGreaterThan(firstPaymentAmount))
                            .map(Tuple2::get1)
                            .findFirst().orElse(null);
            if (lastPaymentInstant == null) {
                continue;
            }
            var secondsFromLastToNonRef = Duration.between(lastPaymentInstant, nonRefInstant).toSeconds();
            BigDecimal metric =
                    firstPaymentAmount.getNumberStripped()
                            .multiply(BigDecimal.valueOf(secondsFromFirstToNonRef))
                            .add(lastPaymentAmount.getNumberStripped().multiply(BigDecimal.valueOf(secondsFromLastToNonRef)));
            if (bestMetric == null || bestMetric.compareTo(metric) > 0) {
                bestMetric = metric;
                bestFirstPaymentAmount = firstPaymentAmount;
                bestLastPaymentInstant = lastPaymentInstant;
            }
        }
        Preconditions.checkState(bestMetric != null, "No best schedule found");
        if (bestLastPaymentInstant.isAfter(checkinDayBeginsAt)) {
            bestLastPaymentInstant = checkinDayBeginsAt.minus(1, ChronoUnit.MINUTES);
        }

        BigDecimal firstRate = bestFirstPaymentAmount.divide(totalAmount.getNumber()).getNumberStripped();
        BigDecimal secondRate = BigDecimal.ONE.subtract(firstRate);
        RefundRule refundRule = refundRules.getRuleAtInstant(bestLastPaymentInstant);
        Money penaltyIfUnpaid = null;
        switch (refundRule.getType()) {
            case NON_REFUNDABLE:
                // This means "no deferred payments"
                return null;
            case FULLY_REFUNDABLE:
                penaltyIfUnpaid = Money.zero(firstPayment.getCurrency());
                break;
            case REFUNDABLE_WITH_PENALTY:
                penaltyIfUnpaid = refundRule.getPenalty();
                break;
        }

        return OfferPaymentSchedule.builder()
                .initialPayment(
                        OfferPaymentScheduleItem.builder()
                                .amount(bestFirstPaymentAmount)
                                .ratio(firstRate)
                                .ratioIsStandard(firstRate.compareTo(STANDARD_RATE) == 0)
                                .name("Первоначальный платеж")
                                .build())
                .deferredPayment(OfferPaymentScheduleItem.builder()
                        .name("Окончательный платеж")
                        .amount(totalAmount.subtract(bestFirstPaymentAmount))
                        .ratio(secondRate)
                        .penaltyIfUnpaid(penaltyIfUnpaid)
                        .paymentEndsAt(bestLastPaymentInstant)
                        .build())
                .build();
    }

    private static void checkRulesInstantsSequence(List<RefundRule> rules) {
        RefundRule prev = null;
        for (var next : rules) {
            if (prev != null) {
                Preconditions.checkArgument(prev.getEndsAt() != null && prev.getEndsAt().equals(next.getStartsAt()),
                        "Penalty instants sequence is not continuous");
            }
            prev = next;
        }
    }

    private static void checkMonotonicity(List<Tuple2<Instant, Money>> points) {
        Money prev = null;
        for (var next : points) {
            if (prev != null) {
                Preconditions.checkState(next.get2().compareTo(prev) >= 0,
                        "Point amounts sequence is not monotonous");
            }
            prev = next.get2();
        }
    }
}
