package ru.yandex.travel.hotels.common.schedule;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.health.HealthChecked;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.THotelId;
import ru.yandex.travel.hotels.proto.TPaymentScheduleHotel;
import ru.yandex.travel.yt_lucene_index.MapPersistentConfig;

@Slf4j
public class PaymentScheduleHotelListImpl extends MapPersistentConfig<THotelId, Boolean, TPaymentScheduleHotel>
        implements PaymentScheduleHotelList, HealthChecked {
    private final PaymentScheduleHotelListProperties params;

    public PaymentScheduleHotelListImpl(PaymentScheduleHotelListProperties params) {
        super(params, "payment_schedule_hotel_list", TPaymentScheduleHotel::newBuilder,
                h -> THotelId.newBuilder()
                        .setOriginalId(h.getOriginalId())
                        .setPartnerId(h.getPartnerId())
                        .build(),
                h -> h.hasPaymentScheduleEnabled() ? h.getPaymentScheduleEnabled() : null);
        this.params = params;
    }

    @Override
    public boolean checkPaymentScheduleIsAllowedForHotel(EPartnerId partnerId, String originalId) {
        Preconditions.checkState(params.isEnabled(), "PaymentScheduleHotelList is not enabled");
        Boolean val = getByKey(THotelId.newBuilder()
                .setPartnerId(partnerId)
                .setOriginalId(originalId)
                .build());
        if (val == null) {
            String pidString = PaymentScheduleProperties.partnerIdStringMap.get(partnerId);
            Preconditions.checkArgument(pidString != null, "Unknown partnerID " + partnerId.name());
            var mode = params.getPartnerModes().get(pidString);
            Preconditions.checkNotNull(mode, "List mode is not set for partner " + pidString);
            switch (mode) {
                case BLACK:
                    return true;  // hotel is not found in blacklist, so assuming it is enabled
                case WHITE:
                    return false; // hotel is not found in whitelist, so assuming it is disabled
                default:
                    throw new IllegalStateException("Unexpected list mode");

            }
        } else {
            return val;
        }
    }

    @Override
    public boolean isHealthy() {
        return this.isReady();
    }
}
