package ru.yandex.travel.hotels.common.schedule;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.Map;
import java.util.Set;

import javax.validation.constraints.Max;
import javax.validation.constraints.NotNull;
import javax.validation.constraints.Pattern;

import com.google.common.collect.ImmutableMap;
import lombok.Data;
import org.springframework.validation.annotation.Validated;

import ru.yandex.travel.hotels.common.validators.OneOfNotNull;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Data
@Validated
@OneOfNotNull(
        fieldNames = {"minFirstPayment", "minFirstPaymentRate"},
        message = "Exactly one of min-first-payment or min-first-payment-rate must be set",
        exactlyOne = true)
public class PaymentScheduleProperties {
    public final static Duration MIN_PAYMENT_INTERVAL = Duration.ZERO;
    public final static Duration MIN_SAFETY_WINDOW = Duration.ofSeconds(3);

    static Map<EPartnerId, String> partnerIdStringMap = ImmutableMap.of(
            EPartnerId.PI_BNOVO, "bnovo",
            EPartnerId.PI_EXPEDIA, "expedia",
            EPartnerId.PI_TRAVELLINE, "travelline",
            EPartnerId.PI_DOLPHIN, "dolphin",
            EPartnerId.PI_BRONEVIK, "bronevik");

    private boolean enabled;

    private Set<@Pattern(regexp = "bnovo|expedia|travelline|dolphin|bronevik") @NotNull String> enabledPartners;
    @NotNull
    private Duration minPaymentInterval;
    @NotNull
    private Duration refundSafetyWindow;

    @NotNull
    @Max(1)
    private BigDecimal maxFirstPaymentRate;

    private PaymentScheduleHotelListProperties hotelLists;

    private BigDecimal minFirstPayment;
    private BigDecimal minFirstPaymentRate;

    public boolean checkEnabledForPartner(EPartnerId partnerId) {
        return enabledPartners != null && enabledPartners.contains(partnerIdStringMap.get(partnerId));
    }
}
