package ru.yandex.travel.hotels.common.token;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Formatter;

import com.google.protobuf.BoolValue;
import com.google.protobuf.ByteString;

import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TTravelToken;

public class TravelToken {
    private final static LocalDateTime EPOCH_START = LocalDateTime.of(2019, 1, 1, 0, 0, 0);
    private final static int SECONDS_A_DAY = 86400;

    private final TTravelToken protoToken;

    private TravelToken(TTravelToken protoToken) {
        this.protoToken = protoToken;
    }

    public static TravelToken fromProto(TTravelToken protoToken) {
        return new TravelToken(protoToken);
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getTokenId() {
        Formatter formatter = new Formatter();
        for (byte b : protoToken.getTokenIdBytes()) {
            formatter.format("%02x", b);
        }
        return formatter.toString();
    }

    public String getOfferId() {
        return protoToken.getOfferId();
    }

    public LocalDateTime getGeneratedAt() {
        return EPOCH_START.plusSeconds(protoToken.getGeneratedAtSecondsSinceEpoch());
    }

    public LocalDate getCheckInDate() {
        return EPOCH_START.plusDays(protoToken.getCheckInDateDaysSinceEpoch()).toLocalDate();
    }

    public LocalDate getCheckOutDate() {
        return EPOCH_START.plusDays(protoToken.getCheckOutDateDaysSinceEpoch()).toLocalDate();
    }

    public Occupancy getOccupancy() {
        if (protoToken.getOccupancy().isEmpty()) {
            return null;
        } else {
            return Occupancy.fromString(protoToken.getOccupancy());
        }
    }

    public long getPermalink() {
        return protoToken.getPermalink();
    }

    public EPartnerId getPartnerId() {
        return protoToken.getPartner();
    }

    public String getOriginalId() {
        switch (protoToken.getOriginalIdCase()) {
            case STRINGID:
                return protoToken.getStringId();
            case NUMERICID:
                return Long.toUnsignedString(protoToken.getNumericId());
            default:
                return null;
        }
    }

    TTravelToken getProtoToken() {
        return protoToken;
    }

    public static class Builder {
        private final TTravelToken.Builder builder = TTravelToken.newBuilder();

        public Builder setTokenIdBytes(byte[] bytes) {
            builder.setTokenIdBytes(ByteString.copyFrom(bytes));
            return this;
        }

        public Builder setOfferId(String offerId) {
            builder.setOfferId(offerId);
            return this;
        }

        public Builder setGeneratedAt(LocalDateTime time) {
            long secondsSinceEpoch = Duration.between(EPOCH_START, time).get(ChronoUnit.SECONDS);
            builder.setGeneratedAtSecondsSinceEpoch(secondsSinceEpoch);
            return this;
        }

        public Builder setCheckInDate(LocalDate checkInDate) {
            int daysSinceEpoch =
                    (int) (Duration.between(EPOCH_START, checkInDate.atStartOfDay()).get(ChronoUnit.SECONDS) / SECONDS_A_DAY);
            builder.setCheckInDateDaysSinceEpoch(daysSinceEpoch);
            return this;
        }

        public Builder setCheckOutDate(LocalDate checkOutDate) {
            int daysSinceEpoch =
                    (int) (Duration.between(EPOCH_START, checkOutDate.atStartOfDay()).get(ChronoUnit.SECONDS) / SECONDS_A_DAY);
            builder.setCheckOutDateDaysSinceEpoch(daysSinceEpoch);
            return this;
        }

        public Builder setOccupancy(Occupancy occupancy) {
            builder.setOccupancy(occupancy.toString());
            return this;
        }

        public Builder setPermalink(long permalink) {
            builder.setPermalink(permalink);
            return this;
        }

        public Builder setPartnerId(EPartnerId partnerId) {
            builder.setPartner(partnerId);
            return this;
        }

        public Builder setOriginalId(String originalId) {
            try {
                builder.setNumericId(Long.parseUnsignedLong(originalId));
            } catch (NumberFormatException ex) {
                builder.setStringId(originalId);
            }
            return this;
        }

        public Builder setOriginalRoomId(String originalRoomId) {
            builder.setOriginalRoomId(originalRoomId);
            return this;
        }

        public Builder setFreeCancellation(Boolean freeCancellation) {
            if (freeCancellation != null) {
                builder.setFreeCancellation(BoolValue.of(freeCancellation));
            }
            return this;
        }

        public Builder setPansion(EPansionType pansionType) {
            builder.setPansion(pansionType);
            return this;
        }

        public TravelToken build() {
            return new TravelToken(builder.build());
        }
    }


}
