package ru.yandex.travel.hotels.geosearch;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.protobuf.ExtensionRegistry;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.TextFormat;
import com.google.protobuf.util.JsonFormat;
import lombok.extern.slf4j.Slf4j;
import org.xerial.snappy.Snappy;
import yandex.maps.proto.atom.Atom;
import yandex.maps.proto.common2.geo_object.GeoObjectOuterClass;
import yandex.maps.proto.common2.metadata.MetadataOuterClass;
import yandex.maps.proto.common2.response.ResponseOuterClass;
import yandex.maps.proto.photos2.Photos2;
import yandex.maps.proto.search.business.Business;
import yandex.maps.proto.search.business_rating_2x.BusinessRating2X;
import yandex.maps.proto.search.experimental.Experimental;
import yandex.maps.proto.search.masstransit_2x.Masstransit2X;
import yandex.maps.proto.search.photos_2x.Photos2X;
import yandex.maps.proto.search.related_places.RelatedPlaces;
import yandex.maps.proto.search.related_places_1x.RelatedPlaces1X;
import yandex.maps.proto.search.search.Search;
import yandex.maps.proto.search.search_internal.SearchInternal;

import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.model.GeoHotel;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelFeatureGroup;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelLegalInfo;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelPhoto;
import ru.yandex.travel.hotels.geosearch.model.GeoHotelUgcFeatures;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.geosearch.model.GeoSimilarHotel;
import ru.yandex.travel.hotels.geosearch.model.TravelSortStats;
import ru.yandex.travel.hotels.offercache.api.TReadResp;

@Slf4j
public class GeoSearchParser {
    private static final ObjectMapper upperCamelCaseMapper;

    static {
        ObjectMapper mapper = null;
        try {
            mapper = new ObjectMapper().setPropertyNamingStrategy(PropertyNamingStrategy.UPPER_CAMEL_CASE);
        } catch (Exception e) {
            log.error("Error while initializing upperCamelCaseMapper", e);
        }
        upperCamelCaseMapper = mapper;
    }

    private final ObjectMapper mapper;
    private final JsonFormat.Parser jsonProtoParser = JsonFormat.parser().ignoringUnknownFields();
    private final ExtensionRegistry pbExtensionRegistry;

    public GeoSearchParser(ObjectMapper mapper) {
        this.mapper = mapper;
        this.pbExtensionRegistry = ExtensionRegistry.newInstance();
        this.pbExtensionRegistry.add(Search.sEARCHRESPONSEMETADATA);
        this.pbExtensionRegistry.add(Experimental.rESPONSEMETADATA);
        this.pbExtensionRegistry.add(Experimental.gEOOBJECTMETADATA);
        this.pbExtensionRegistry.add(Business.gEOOBJECTMETADATA);
        this.pbExtensionRegistry.add(Business.rESPONSEMETADATA);
        this.pbExtensionRegistry.add(Photos2X.gEOOBJECTMETADATA);
        this.pbExtensionRegistry.add(BusinessRating2X.gEOOBJECTMETADATA);
        this.pbExtensionRegistry.add(SearchInternal.sEARCHRESPONSEINFO);
        this.pbExtensionRegistry.add(Photos2.aTOMENTRY);
        this.pbExtensionRegistry.add(RelatedPlaces1X.gEOOBJECTMETADATA);
        this.pbExtensionRegistry.add(Masstransit2X.gEOOBJECTMETADATA);
    }

    public GeoSearchRsp parseBinProtoResponse(byte[] binResponse) throws InvalidProtocolBufferException {
        return transformResponse(ResponseOuterClass.Response.parseFrom(binResponse, pbExtensionRegistry));
    }

    public GeoSearchRsp parseTextProtoResponse(String textResponse) throws TextFormat.ParseException {
        ResponseOuterClass.Response.Builder origResponse = ResponseOuterClass.Response.newBuilder();
        TextFormat.getParser().merge(textResponse, pbExtensionRegistry, origResponse);
        return transformResponse(origResponse.build());
    }

    private GeoSearchRsp transformResponse(ResponseOuterClass.Response origResponse) {
        GeoSearchRsp response = new GeoSearchRsp();
        Map<String, GeoSimilarHotel> similarHotelMap = null;
        // Parse protobuf extensions
        for (MetadataOuterClass.Metadata metadata : origResponse.getReply().getMetadataList()) {
            if (metadata.hasExtension(Search.sEARCHRESPONSEMETADATA)) {
                Search.SearchResponseMetadata responseMetadata = metadata.getExtension(Search.sEARCHRESPONSEMETADATA);
                response.setResponseMetadata(responseMetadata);
                for (MetadataOuterClass.Metadata subMetadata: responseMetadata.getSourceList()) {
                    if (subMetadata.hasExtension(Business.rESPONSEMETADATA)) {
                        response.setBusinessMetadata(subMetadata.getExtension(Business.rESPONSEMETADATA));
                    }
                }
                if (responseMetadata.hasExtension(SearchInternal.sEARCHRESPONSEINFO)) {
                    response.setResponseInfo(responseMetadata.getExtension(SearchInternal.sEARCHRESPONSEINFO));
                }
            }
            if (metadata.hasExtension(Experimental.rESPONSEMETADATA)) {
                Experimental.ExperimentalMetadata m = metadata.getExtension(Experimental.rESPONSEMETADATA);
                for (Experimental.ExperimentalStorage.Item item : m.getExperimentalStorage().getItemList()) {
                    switch (item.getKey()) {
                        case "yandex_travel_response_data":
                            try {
                                TReadResp.Builder builder = TReadResp.newBuilder();
                                jsonProtoParser.merge(item.getValue(), builder);
                                response.setOfferCacheResponseMetadata(builder.build());
                            } catch (InvalidProtocolBufferException exc) {
                                log.error("Failed to decode yandex_travel_response_data", exc);
                            }
                            break;
                        case "similar_orgs_travel_snippet":
                            similarHotelMap = parseSimilarOrgsTravelSnippet(item.getValue());
                            break;
                        case "yandex_travel_sort_stats":
                            try {
                                response.setTravelSortStats(upperCamelCaseMapper.readValue(item.getValue(), TravelSortStats.class));
                            } catch (IOException e) {
                                log.error("Unexpected error while parsing yandex_travel_sort_stats. Value: '{}'", item.getValue(), e);
                            }
                            break;
                        default:
                            break;
                    }
                }
            }
        }
        for (GeoObjectOuterClass.GeoObject geoObject : origResponse.getReply().getGeoObjectList()) {
            GeoHotel hotel = new GeoHotel();
            for (MetadataOuterClass.Metadata metadata : geoObject.getMetadataList()) {
                if (metadata.hasExtension(Experimental.gEOOBJECTMETADATA)) {
                    parseGeoObjectExperimentalStorage(hotel, metadata.getExtension(Experimental.gEOOBJECTMETADATA));
                }
                if (metadata.hasExtension(Business.gEOOBJECTMETADATA)) {
                    hotel.setGeoObjectMetadata(metadata.getExtension(Business.gEOOBJECTMETADATA));
                    hotel.setPermalink(Permalink.of(hotel.getGeoObjectMetadata().getId()));
                }
                if (metadata.hasExtension(Photos2X.gEOOBJECTMETADATA)) {
                    hotel.setPhotos(metadata.getExtension(Photos2X.gEOOBJECTMETADATA));
                }
                if (metadata.hasExtension(BusinessRating2X.gEOOBJECTMETADATA)) {
                    hotel.setRating(metadata.getExtension(BusinessRating2X.gEOOBJECTMETADATA));
                }
                if (metadata.hasExtension(Masstransit2X.gEOOBJECTMETADATA)) {
                    hotel.setNearByStops(metadata.getExtension(Masstransit2X.gEOOBJECTMETADATA));
                }
                if (metadata.hasExtension(RelatedPlaces1X.gEOOBJECTMETADATA)) {
                    RelatedPlaces1X.RelatedPlaces relatedPlaces = metadata.getExtension(RelatedPlaces1X.gEOOBJECTMETADATA);
                    hotel.setSimilarHotels(new ArrayList<>());
                    if (similarHotelMap != null) {
                        for (RelatedPlaces.PlaceInfo place : relatedPlaces.getSimilarPlacesList()) {
                            String permalink = place.getLogId();// Мещерин благословил
                            GeoSimilarHotel similarHotel = similarHotelMap.get(permalink);
                            if (similarHotel == null) {
                                continue;
                            }
                            similarHotel.setPermalink(Permalink.of(permalink));
                            similarHotel.setPlaceInfo(place);
                            hotel.getSimilarHotels().add(similarHotel);
                        }
                    }
                }
            }
            hotel.setGeoObject(geoObject.toBuilder().clearMetadata().build());
            response.getHotels().add(hotel);
        }
        return response;
    }

    private void parseGeoObjectExperimentalStorage(GeoHotel hotel, Experimental.ExperimentalMetadata m) {
        for (Experimental.ExperimentalStorage.Item item : m.getExperimentalStorage().getItemList()) {
            try {
                switch (item.getKey()) {
                    case "yandex_travel/1.x":
                        TReadResp.THotel.Builder builder = TReadResp.THotel.newBuilder();
                        jsonProtoParser.merge(item.getValue(), builder);
                        hotel.setOfferCacheResponse(builder.build());
                        break;
                    case "legal_info/1.x":
                        // LegalInfo comes as json. Corresponding .proto is located in sprav, but we do not
                        // want to depend on it, so deserialize JSON manually
                        hotel.setLegalInfo(mapper.readerFor(GeoHotelLegalInfo.class).readValue(item.getValue()));
                        break;
                    case "sprav_proto_photos":
                        // SpravProtoPhotos comes as base64-encoded proto
                        byte[] data = Snappy.uncompress(Base64.getDecoder().decode(item.getValue()));
                        Atom.Feed feed = Atom.Feed.parseFrom(data, pbExtensionRegistry);
                        hotel.setSpravPhotos(new ArrayList<>());
                        for (Atom.Entry entry : feed.getEntryList()) {
                            if (!entry.hasExtension(Photos2.aTOMENTRY)) {
                                continue;
                            }
                            GeoHotelPhoto p = new GeoHotelPhoto();
                            p.setBase(entry);
                            p.setPhoto(entry.getExtension(Photos2.aTOMENTRY));
                            hotel.getSpravPhotos().add(p);
                        }
                        break;
                    case "feature_ugc_answers/1.x":
                        // UGC Answers comes as JSON, generated here:
                        // https://a.yandex-team.ru/arc/trunk/arcadia/search/geo/tools/task_manager/generators/feature_ugc_answers.yql?rev=5642086
                        List<GeoHotelUgcFeatures.Feature> featureList = mapper.readValue(item.getValue(), new TypeReference<List<GeoHotelUgcFeatures.Feature>>() {});
                        GeoHotelUgcFeatures features = new GeoHotelUgcFeatures();
                        features.setFeatureList(featureList);
                        hotel.setUgcFeatures(features);
                        break;
                    case "feature_groups/1.x":
                        // Feature groups comes as JSON, generated here:
                        // https://a.yandex-team.ru/arc/trunk/arcadia/search/geo/tools/task_manager/generators/feature_groups.yql?rev=6205308
                        hotel.setFeatureGroups(mapper.readValue(item.getValue(), new TypeReference<List<GeoHotelFeatureGroup>>() {}));
                        break;
                    case "rubric_id":
                        hotel.setCategoryIds(Arrays.asList(item.getValue().split(",")));
                        break;
                }
            } catch (Exception exc) {
                log.error("Failed to decode snippet {}", item.getKey(), exc);
            }
        }
    }

    private Map<String, GeoSimilarHotel> parseSimilarOrgsTravelSnippet(String snippet) {
        // This snippet comes as JSON, containing OfferCache response (THotel)
        // and special key "Extension", composed by this YQL:
        // https://a.yandex-team.ru/arc/trunk/arcadia/search/geo/tools/task_manager/generators/similar_orgs_extension.yql?rev=5007636#L151
        JsonParser parser = new JsonParser();
        JsonObject parsedSnippet = parser.parse(snippet).getAsJsonObject();
        Map<String, GeoSimilarHotel> result = new HashMap<>();
        for (Map.Entry<String, JsonElement> entry: parsedSnippet.entrySet()) {
            JsonObject element = entry.getValue().getAsJsonObject();
            JsonElement jsonExtensionElement = element.remove("Extension");
            if (jsonExtensionElement == null) {
                // May be missing
                continue;
            }
            GeoSimilarHotel hotel = new GeoSimilarHotel();
            result.put(entry.getKey(), hotel);
            JsonObject jsonExtension = jsonExtensionElement.getAsJsonObject();
            GeoSimilarHotel.Extension extension = new GeoSimilarHotel.Extension();
            hotel.setExtension(extension);
            extension.setStars(jsonExtension.get("stars").getAsInt());
            extension.setReviewCount(jsonExtension.get("review_count").getAsInt());
            extension.setRatingScore(jsonExtension.get("rating_score").getAsDouble());
            extension.setRatingCount(jsonExtension.get("rating_count").getAsInt());
            List<GeoSimilarHotel.Feature> features = new ArrayList<>();
            extension.setFeatures(features);
            for (JsonElement jsonFeatureElement: jsonExtension.get("features").getAsJsonArray()) {
                JsonObject jsonFeature = jsonFeatureElement.getAsJsonObject();
                GeoSimilarHotel.Feature feature = new GeoSimilarHotel.Feature();
                feature.setId(jsonFeature.get("id").getAsString());
                feature.setMain(jsonFeature.get("is_main").getAsBoolean());
                feature.setName(jsonFeature.get("name").getAsString());
                String featureType = jsonFeature.get("type").getAsString();
                switch (featureType) {
                    case "bool":
                        feature.setBoolValue(jsonFeature.get("value").getAsString().equals("1"));
                        features.add(feature);
                        break;
                    case "text":
                        feature.setStringValue(jsonFeature.get("value").getAsString());
                        features.add(feature);
                        break;
                    case "enum":
                        JsonArray jsonValues = jsonFeature.get("values").getAsJsonArray();
                        feature.setEnumValues(new ArrayList<>());
                        for (JsonElement jsonValueEl: jsonValues) {
                            feature.getEnumValues().add(jsonValueEl.getAsString());
                        }
                        features.add(feature);
                        break;
                    default:
                        log.warn("Unknown Similar Feature type '{}' for hotel {}", featureType, entry.getKey());
                        break;
                }
            }
            // OfferCacheResp
            if (!element.keySet().isEmpty()) {
                try {
                    String rest = element.toString();
                    TReadResp.THotel.Builder builder = TReadResp.THotel.newBuilder();
                    jsonProtoParser.merge(rest, builder);
                    hotel.setOfferCacheResponse(builder.build());
                } catch (Exception exc) {
                    log.warn("Failed to parse OfferCache response for Similar hotel {}", entry.getKey(), exc);
                }
            }
        }
        return result;
    }
}
