package ru.yandex.travel.hotels.geosearch;

import java.time.Duration;
import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;

import ru.yandex.misc.io.http.HttpException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.AhcHttpRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.hotels.common.HotelNotFoundException;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchReq;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchRsp;
import ru.yandex.travel.hotels.geosearch.model.GeoSearchSingleHotelRsp;
import ru.yandex.travel.tvm.TvmWrapper;

@Slf4j
public class GeoSearchService {
    private final GeoSearchProperties config;
    private final AsyncHttpClientWrapper client;
    private final TvmWrapper tvm;
    private final GeoSearchParser parser;
    private final DateTimeFormatter shortDateFormatter;
    private final Retry retryHelper;
    private final RetryRateLimiter retryRateLimiter;

    public GeoSearchService(GeoSearchProperties config,
                            AsyncHttpClientWrapper client,
                            ObjectMapper mapper,
                            TvmWrapper tvm,
                            Retry retryHelper
    ) {
        this.config = config;
        this.client = client;
        this.tvm = tvm;
        this.retryHelper = retryHelper;
        this.parser = new GeoSearchParser(mapper);
        this.shortDateFormatter = DateTimeFormatter.ofPattern("yyyyMMdd");
        if (tvm != null && config.getTvmDestinationAlias() != null) {
            tvm.validateAlias(config.getTvmDestinationAlias());
        }
        this.retryRateLimiter = new RetryRateLimiter(config.getRetryRateLimit());
    }

    private String getOfferCacheAddress(boolean prod) {
        if (prod) {
            return config.getProdOfferCacheAddress();
        } else {
            return config.getOfferCacheAddress();
        }
    }

    private void checkStatus(String name, Response response) {
        if (response.getStatusCode() == HttpStatus.SC_200_OK || response.getStatusCode() == HttpStatus.SC_501_NOT_IMPLEMENTED) {
            return;
        }
        String msg = String.format("GeoSearch returned %s, message '%s'",
                response.getStatusCode(), response.getResponseBody());
        log.error("{}: Bad HTTP status: {}", name, msg);
        throw new HttpException(response.getStatusCode(), msg);
    }

    private String buildName(GeoSearchReq req) {
        String name = "GeoSearch";
        if (req.getParentReqId() != null) {
            name += "/" + req.getParentReqId();
        }
        if (req.getRequestLogId() != null && !req.getRequestLogId().isEmpty()) {
            name += "/" + req.getRequestLogId();
        }
        name += "(" + req.getSearchKey() + ")";
        return name;
    }

    public CompletableFuture<GeoSearchRsp> query(GeoSearchReq req) {
        return query(req, null);
    }

    public CompletableFuture<GeoSearchRsp> query(GeoSearchReq req, String baseUrlOverride) {
        RequestBuilder httpReq = prepareRequest(req, baseUrlOverride != null ? baseUrlOverride : config.getBaseUrl());
        String name = buildName(req);
        log.debug("{}: Sending, full url: '{}'", name, httpReq.build().getUrl());
        Instant geoStarted = Instant.now();
        return retryHelper.withRetry(name, this::runRequest, httpReq, new AhcHttpRetryStrategy(), retryRateLimiter)
            .thenApply(httpResponse -> {
                checkStatus(name, httpResponse);
                try {
                    Instant geoFinished = Instant.now();
                    GeoSearchRsp rsp;
                    if (httpResponse.getStatusCode() == HttpStatus.SC_501_NOT_IMPLEMENTED) {
                        rsp = new GeoSearchRsp();
                        rsp.setNotImplementedError(true);
                    } else {
                        rsp = parser.parseBinProtoResponse(httpResponse.getResponseBodyAsBytes());
                        rsp.setNotImplementedError(false);
                    }
                    Instant parseFinished = Instant.now();
                    rsp.setResponseTime(Duration.between(geoStarted, geoFinished));
                    rsp.setParseTime(Duration.between(geoFinished, parseFinished));
                    log.debug("{}: Finished", name);
                    return rsp;
                } catch (InvalidProtocolBufferException e) {
                    log.error("{}: Unable to deserialize geosearch proto response", name, e);
                    throw new RuntimeException(e);
                }
            });
    }

    public CompletableFuture<GeoSearchSingleHotelRsp> querySingleHotel(GeoSearchReq req) {
        Preconditions.checkArgument((req.getPermalinks() != null && req.getPermalinks().size() == 1)
                        || req.getOriginalId() != null,
                "search Type invalid for protoQuerySingleHotel call");
        return query(req).thenApply(rsp -> {
            if (rsp.getHotels().size() < 1) {
                throw new HotelNotFoundException(String.format("GeoSearch returned %s hotels instead of 1, by key %s",
                        rsp.getHotels().size(), req.getSearchKey()));
            }
            GeoSearchSingleHotelRsp result = new GeoSearchSingleHotelRsp();
            result.setOfferCacheResponseMetadata(rsp.getOfferCacheResponseMetadata());
            result.setHotel(rsp.getHotels().get(0));
            return result;
        });
    }

    private CompletableFuture<Response> runRequest(RequestBuilder request) {
        return client.executeRequest(request);
    }

    private RequestBuilder prepareRequest(GeoSearchReq req, String baseUrl) {
        RequestBuilder builder = new RequestBuilder()
                .setReadTimeout(Math.toIntExact(config.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(config.getHttpRequestTimeout().toMillis()))
                .setUrl(baseUrl)
                .addQueryParam("lang", req.getLanguage().getFullLocale())
                .addQueryParam("origin", req.getOrigin().getValue())
                .addQueryParam("client_id", "travel.portal") // HOTELS-4171
                .addQueryParam("mode", "oid")
                .addQueryParam("type", "biz")
                .addQueryParam("ms", "pb");
        List<String> snippets = new ArrayList<>();
        if (req.isIncludeLegalInfo()) {
            snippets.add("legal_info/1.x");
        }
        if (req.isIncludeOfferCache()) {
            builder.addQueryParam("gta", "yandex_travel_response_data");
            String ocAddr = getOfferCacheAddress(req.isUseProdOfferCache());
            if (ocAddr != null) {
                builder.addQueryParam("source", "YandexTravel:" + ocAddr);
            }
            snippets.add("yandex_travel/1.x");
        }
        if (req.isIncludeRating()) {
            snippets.add("businessrating/2.x");
        }
        if (req.isIncludePhotos()) {
            snippets.add("photos/2.x");
        }
        if (req.isIncludeSpravPhotos()) {
            snippets.add("sprav_proto_photos");
        }
        if (req.isIncludeUgcAnswers()) {
            snippets.add("feature_ugc_answers/1.x");
        }
        if (req.isIncludeSimilarHotels()) {
            snippets.add("related_places/1.x");
        }
        if (req.isIncludeSimilarHotels() && req.isIncludeOfferCache()) {
            builder.addQueryParam("gta", "similar_orgs_travel_snippet"); // Цены приходят в очень странном месте
            snippets.add("similar_orgs_travel/1.x");// Чтобы были цены
            snippets.add("similar_orgs_extension/1.x"); // Чтобы были фичи
        }
        if (req.isIncludeNearByStops()) {
            snippets.add("masstransit/2.x");
        }
        if (req.isIncludeFeatureGroups()) {
            snippets.add("feature_groups/1.x");
        }
        if (!snippets.isEmpty()) {
            builder.addQueryParam("snippets", String.join(",", snippets));
        }
        if (req.isIncludeCategoryIds()) {
            builder.addQueryParam("gta", "rubric_id");
        }
        if (req.isIncludeSortStats()) {
            builder.addQueryParam("gta", "yandex_travel_sort_stats");
        }
        if (tvm != null && config.getTvmDestinationAlias() != null) {
            String serviceTicket = tvm.getServiceTicket(config.getTvmDestinationAlias());
            builder.setHeader(CommonHttpHeaders.HeaderType.SERVICE_TICKET.getHeader(), serviceTicket);
        }
        if (req.isSupressTextCorrection()) {
            builder.addQueryParam("correct_misspell", "0");
        }
        if (req.getOffset() != null) {
            builder.addQueryParam("skip", String.valueOf(req.getOffset()));
        }
        if (req.getLimit() != null) {
            builder.addQueryParam("results", String.valueOf(req.getLimit()));
        }
        if (req.getOfferCacheRequestParams() != null) {
            req.getOfferCacheRequestParams().putToQueryParams(builder);
        }
        if (req.getLr() != null) {
            builder.addQueryParam("lr", req.getLr().toString());
        }
        if (req.isRspn()) {
            builder.addQueryParam("rspn", "1");
        }
        if (req.getAutoscale() != null) {
            builder.addQueryParam("autoscale", req.getAutoscale() ? "1" : "0");
        }
        if (req.getBoundingBox() != null) {
            builder.addQueryParam("bbox", req.getBoundingBox());
        }
        if (req.getSort() != null) {
            builder.addQueryParam("sort", req.getSort());
        }
        if (req.getUll() != null) {
            builder.addQueryParam("ull", req.getUll());
        }
        if (req.getSortOrigin() != null) {
            builder.addQueryParam("sort_origin", req.getSortOrigin());
        }
        if (req.getGeowhere() != null) {
            builder.addQueryParam("geowhere", req.getGeowhere());
        }
        if (req.getFixedTop() != null) {
            builder.addQueryParam("fixed_top", req.getFixedTop()
                    .stream()
                    .map(Permalink::toString)
                    .collect(Collectors.joining(",")));
        }
        if (req.getGeowhereKinds() != null) {
            builder.addQueryParam("relev_filter_gwkinds", req.getGeowhereKinds()
                    .stream()
                    .map(GeoSearchReq.GeowhereKind::getValue)
                    .distinct()
                    .collect(Collectors.joining(",")));
        }
        if (req.getRearr() != null) {
            for (String value : req.getRearr()) {
                builder.addQueryParam("rearr", value);
            }
        }
        if (req.isPruneEmptyHotels()) {
            builder.addQueryParam("middle_yandex_travel_prune_empty_hotels", "1");
        }
        if (req.getFilterStars() != null) {
            builder.addQueryParam("business_filter", "star:" + String.join(",", req.getFilterStars()));
        }
        if (req.getFilterHotelPriceCategory() != null) {
            builder.addQueryParam("business_filter", "hotel_price_category:" + String.join(",",
                    req.getFilterHotelPriceCategory()));
        }
        if (req.getFilterBoolFeatures() != null) {
            for (String feature : req.getFilterBoolFeatures()) {
                builder.addQueryParam("business_filter", feature + ":1");
            }
        }
        if (req.getFilterBoolOrEnumFeatures() != null) {
            for (GeoSearchReq.BoolOrEnumFilter feature : req.getFilterBoolOrEnumFeatures()) {
                builder.addQueryParam("business_filter", feature.getId() + ":" + String.join(",", feature.getValues()));
            }
        }
        if (req.getFilterNumericFeatures() != null) {
            for (GeoSearchReq.NumericFilter feature : req.getFilterNumericFeatures()) {
                if (feature.getFrom() != null || feature.getTo() != null) {
                    var borders = Stream.of(feature.getFrom(), feature.getTo())
                            .map(x -> x != null ? String.format(Locale.US, "%.2f", x) : "")
                            .collect(Collectors.toUnmodifiableList());
                    builder.addQueryParam("business_filter", feature.getId() + ":" + String.join("-", borders));
                }
            }
        }
        if (req.getContext() != null) {
            builder.addQueryParam("ctx", req.getContext());
        }
        if (req.getParentReqId() != null) {
            builder.addQueryParam("parent_reqid", req.getParentReqId());
        }
        if (req.getFilterPriceFrom() != null || req.getFilterPriceTo() != null) {
            StringBuilder priceFilter = new StringBuilder();
            if (req.getFilterPriceFrom() != null) {
                priceFilter.append(req.getFilterPriceFrom());
            }
            priceFilter.append('-');
            if (req.getFilterPriceTo() != null) {
                priceFilter.append(req.getFilterPriceTo());
            }
            builder.addQueryParam("business_filter", "hotel_price_range:" + priceFilter.toString());

        }
        if (req.getFilterDateFrom() != null || req.getFilterDateTo() != null) {
            StringBuilder dateFilter = new StringBuilder();
            if (req.getFilterDateFrom() != null) {
                dateFilter.append(req.getFilterDateFrom().format(shortDateFormatter));
            }
            dateFilter.append('-');
            if (req.getFilterDateTo() != null) {
                dateFilter.append(req.getFilterDateTo().format(shortDateFormatter));
            }
            builder.addQueryParam("business_filter", "hotel_date_range:" + dateFilter.toString());
        }
        if (req.getFilterHotelProviders() != null) {
            builder.addQueryParam("business_filter", "hotel_provider:" + String.join(",",
                    req.getFilterHotelProviders()));
        }
        if (req.getFilterCategories() != null) {
            builder.addQueryParam("business_filter", "category_id:" +
                    String.join(",", req.getFilterCategories()));
        }
        if (req.getSortType() != null) {
            builder.addQueryParam("rearr", "scheme_Local/Geo/Hotels/Sort/Type=" + req.getSortType().getValue());
        }
        if (req.getEnableSortPriceFromFactors() != null) {
            builder.addQueryParam("rearr",
                    "scheme_Local/Geo/Hotels/Sort/EnablePriceFromFactors=" + (req.getEnableSortPriceFromFactors() ?
                            "1" : "0"));
        }
        if (req.getEnableSortPriceFromSnippets() != null) {
            builder.addQueryParam("rearr",
                    "scheme_Local/Geo/Hotels/Sort/EnablePriceFromSnippets=" + (req.getEnableSortPriceFromSnippets() ?
                            "1" : "0"));
        }
        if (req.getSortPriceFallbackToFactors() != null) {
            builder.addQueryParam("rearr",
                    "scheme_Local/Geo/Hotels/Sort/FallbackToFactors=" + (req.getSortPriceFallbackToFactors() ? "1" :
                            "0"));
        }
        if (req.getPermalinks() != null) {
            builder.addQueryParam("relev_result_ids",
                    req.getPermalinks().stream().map(p -> "b:" + p.toString()).collect(Collectors.joining(",")));
        } else if (req.getOriginalId() != null) {
            builder.addQueryParam("business_reference", String.format("%s:%s", req.getOriginalId().getPartnerCode(),
                    req.getOriginalId().getOriginalId()));
        } else {
            builder.addQueryParam("text", req.getText());
        }
        if (req.isIncludeClosedHotels()) {
            builder.addQueryParam("business_show_closed", "1");
        }
        if (req.getAttribution() != null) {
            if (req.getAttribution().getYandexUid() != null) {
                builder.addQueryParam("yandex_uid", req.getAttribution().getYandexUid());
            }
            if (req.getAttribution().getPassportUid() != null) {
                builder.addQueryParam("passport_uid", req.getAttribution().getPassportUid());
            }
            if (req.getAttribution().getIcookieDecrypted() != null) {
                builder.addQueryParam("icookie_decrypted", req.getAttribution().getIcookieDecrypted());
            }
            if (req.getAttribution().getUserIp() != null) {
                builder.addQueryParam("remote_ip", req.getAttribution().getUserIp());
            }
        }
        if (req.getAdditionalParams() != null && !req.getAdditionalParams().isEmpty()) {
            for (var entry: req.getAdditionalParams().entrySet()) {
                builder.addQueryParam(entry.getKey(), entry.getValue());
            }
        }
        return builder;
    }
}
