package ru.yandex.travel.hotels.geosearch.model;

import java.time.LocalDate;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.LanguageType;
import ru.yandex.travel.hotels.common.Permalink;

@Data
@Slf4j
public class GeoSearchReq {
    @RequiredArgsConstructor
    public enum GeowhereKind {
        /*
        * Magic values from https://a.yandex-team.ru/arc/trunk/arcadia/extsearch/geo/meta/rearrs/common/factors_calculation/geo_query_factors.cpp?rev=6356285#L190
        * According to https://st.yandex-team.ru/HOTELS-4839#5e441280e1de8412ab388e55
        */

        HOUSE("house", "1.0"),
        KM("km", "1.0"),
        STREET("street", "0.9"),
        ROUTE("route", "0.9"),
        BRIDGE("bridge", "0.9"),
        DISTRICT("district", "0.8"),
        VEGETATION("vegetation", "0.8"),
        CEMETERY("cemetery", "0.8"),
        METRO("metro", "0.7"),
        RAILWAY("railway", "0.6"),
        AREA("area", "0.5"),
        PROVINCE("province", "0.45"),
        HYDRO("hydro", "0.4"),
        LOCALITY("locality", "0.35"),
        COUNTRY("country", "0.3"),
        UNKNOWN("unknown", "0.05"),
        NONE("none", "0.0");

        @Getter
        private final String name;

        @Getter
        private final String value;

        private final static Map<String, GeowhereKind> allKindsByName;

        private final static Map<String, GeowhereKind> allKindsByValue;

        public static GeowhereKind getByNameOrValue(String s) {
            if (allKindsByName.containsKey(s)) {
                return allKindsByName.get(s);
            }
            return allKindsByValue.getOrDefault(s, null);
        }

        static {
            try {
                allKindsByName = Arrays.stream(GeoSearchReq.GeowhereKind.values())
                    .collect(Collectors.toMap(GeoSearchReq.GeowhereKind::getName, x -> x));
                allKindsByValue = Arrays.stream(GeoSearchReq.GeowhereKind.values())
                    .collect(Collectors.toMap(GeoSearchReq.GeowhereKind::getValue, x -> x, (fst, snd) -> fst));
            } catch (Throwable t) {
                log.error("Failure during static initialization of GeowhereKind", t);
                throw t;
            }
        }
    }

    @RequiredArgsConstructor
    public enum SortType {
        CHEAP_FIRST("CheapFirst"),
        EXPENSIVE_FIRST("ExpensiveFirst"),
        HIGH_RATING_FIRST("HighRatingFirst");

        @Getter
        private final String value;
    }

    @Data
    @RequiredArgsConstructor
    public static class BoolOrEnumFilter {
        private final String id;
        private final List<String> values;
    }

    @Data
    @RequiredArgsConstructor
    public static class NumericFilter {
        private final String id;
        private final Double from;
        private final Double to;
    }

    // Snippets
    private boolean includeLegalInfo;
    private boolean includeOfferCache;
    private boolean includeRating;
    private boolean includePhotos;
    private boolean includeSpravPhotos;
    private boolean includeUgcAnswers;
    private boolean includeSimilarHotels;
    private boolean includeNearByStops;
    private boolean includeFeatureGroups;
    private boolean includeCategoryIds;

    private boolean includeSortStats;

    // What we are looking for: One of
    // 1. Search text. This is default option
    private String text = null;

    // 2. Permalink
    private List<Permalink> permalinks = null;

    // 3. PartnerId + hotelOriginalId
    private GeoHotelOriginalId originalId = null;

    private boolean supressTextCorrection = false;
    private OfferCacheRequestParams offerCacheRequestParams = null;
    private Integer offset = null;
    private Integer limit = null;
    private Integer lr = null;
    private boolean rspn = false;
    private Boolean autoscale = null;
    private String boundingBox = null;
    private boolean pruneEmptyHotels = false;
    private List<String> filterStars = null; // TODO: deprecated, remove it
    private List<String> filterHotelPriceCategory = null;
    private List<String> filterBoolFeatures = null; // TODO: deprecated, remove it
    private List<BoolOrEnumFilter> filterBoolOrEnumFeatures = null;
    private List<NumericFilter> filterNumericFeatures = null;
    private Integer filterPriceFrom = null;
    private Integer filterPriceTo = null;
    private LocalDate filterDateFrom = null;
    private LocalDate filterDateTo = null;
    private List<String> filterHotelProviders = null;
    private List<String> filterCategories = null;
    private GeoOriginEnum origin = null;
    private String context = null;
    private String requestLogId = "";
    private String parentReqId = null;
    private boolean includeClosedHotels = false;
    private LanguageType language = LanguageType.ru;
    private boolean useProdOfferCache = false;
    private String geowhere = null;
    private List<GeowhereKind> geowhereKinds = null; // для каких типов топонимов использовать строгую фильтрацию
    private Attribution attribution = null;
    private List<String> rearr = null;
    private SortType sortType = null;
    private Boolean enableSortPriceFromFactors = null;
    private Boolean enableSortPriceFromSnippets = null;
    private Boolean sortPriceFallbackToFactors = null;
    private String sort = null;
    private String ull = null;
    private String sortOrigin = null;
    private List<Permalink> fixedTop = null;
    private Map<String, String> additionalParams = new HashMap<>();

    private GeoSearchReq() {
    }

    public static GeoSearchReq byPermalink(GeoOriginEnum origin, Permalink permalink, CommonHttpHeaders headers) {
        GeoSearchReq req = new GeoSearchReq();
        req.setOrigin(origin);
        req.setPermalinks(List.of(permalink));
        if (headers != null) {
            req.setAttribution(Attribution.fromHeaders(headers));
        }
        return req;
    }

    public static GeoSearchReq byPermalinks(GeoOriginEnum origin, List<Permalink> permalinks, CommonHttpHeaders headers) {
        GeoSearchReq req = new GeoSearchReq();
        req.setOrigin(origin);
        req.setPermalinks(permalinks);
        if (headers != null) {
            req.setAttribution(Attribution.fromHeaders(headers));
        }
        return req;
    }

    public static GeoSearchReq byOriginalId(GeoOriginEnum origin, GeoHotelOriginalId origId, CommonHttpHeaders headers) {
        GeoSearchReq req = new GeoSearchReq();
        req.setOrigin(origin);
        req.setOriginalId(origId);
        if (headers != null) {
            req.setAttribution(Attribution.fromHeaders(headers));
        }
        return req;
    }

    public static GeoSearchReq byText(GeoOriginEnum origin, String text, CommonHttpHeaders headers) {
        Preconditions.checkArgument(text != null,
                "GeoSearch.byText cannot have text == null");
        GeoSearchReq req = new GeoSearchReq();
        req.setOrigin(origin);
        req.setText(text);
        if (headers != null) {
            req.setAttribution(Attribution.fromHeaders(headers));
        }
        return req;
    }

    public String getSearchKey() {
        if (getPermalinks() != null) {
            return "Permalinks " + getPermalinks().toString();
        }
        if (getOriginalId() != null) {
            return "OrigId " + getOriginalId().toString();
        }
        return "Text '" + getText() + "'";
    }
}
