package ru.yandex.travel.hotels.geosearch.util;

import com.google.gson.JsonObject;
import com.google.protobuf.Descriptors;
import com.google.protobuf.MessageOrBuilder;
import org.asynchttpclient.RequestBuilder;

import ru.yandex.travel.proto.BaseTypes;
import ru.yandex.travel.hotels.proto.ProtoExtensions;


public class ProtoUtils {

    private static String getProtoValue(Descriptors.FieldDescriptor fd, Object obj) {
        boolean cgiKeyValue = fd.getOptions().hasExtension(ProtoExtensions.cgiKeyValue)
                && fd.getOptions().getExtension(ProtoExtensions.cgiKeyValue);
        if (cgiKeyValue) {
            if (!(obj instanceof BaseTypes.TStringPair)) {
                throw new RuntimeException(String.format("Cannot handle message subtype %s, field %s", fd.getType(), fd.getFullName()));
            }
            BaseTypes.TStringPair pair = (BaseTypes.TStringPair) obj;
            return String.format("%s=%s", pair.getKey(), pair.getValue());
        }
        switch (fd.getType()) {
            case INT32:
            case INT64:
            case UINT32:
            case UINT64:
            case DOUBLE:
            case FLOAT:
                return obj.toString();
            case BOOL:
                return (Boolean)obj ? "true" : "false";
            case ENUM:
                return ((Descriptors.EnumValueDescriptor)obj).getName();
            case STRING:
                return (String)obj;
            default:
                throw new RuntimeException(String.format("Cannot handle message subtype %s, field %s", fd.getType(), fd.getFullName()));
        }
    }

    public static void protoToQueryParams(MessageOrBuilder proto, RequestBuilder builder, String httpPrefix) {
        Descriptors.Descriptor d = proto.getDescriptorForType();

        for (Descriptors.FieldDescriptor fd: d.getFields()) {
            String name = httpPrefix + fd.getName();
            if (fd.isRepeated()) {
                for (int idx = 0; idx < proto.getRepeatedFieldCount(fd); ++idx) {
                    builder.addQueryParam(name, getProtoValue(fd, proto.getRepeatedField(fd, idx)));
                }
            } else {
                if (proto.hasField(fd)) {
                    var parseAsJson = fd.getOptions().hasExtension(ProtoExtensions.parseMessageAsJson)
                            && fd.getOptions().getExtension(ProtoExtensions.parseMessageAsJson);
                    if (fd.getType() == Descriptors.FieldDescriptor.Type.MESSAGE && !parseAsJson) {
                        protoToQueryParams((MessageOrBuilder)proto.getField(fd), builder, httpPrefix);
                    } else {
                        builder.addQueryParam(name, getProtoValue(fd, proto.getField(fd)));
                    }
                }
            }
        }
    }

    private static void addJsonProperty(JsonObject jsonObject, String name, Descriptors.FieldDescriptor fd, Object field) {
        switch (fd.getType()) {
            case INT32:
            case UINT32:
            case INT64:
            case UINT64:
            case DOUBLE:
            case FLOAT:
                jsonObject.addProperty(name, (Number)field);
                break;
            case BOOL:
                jsonObject.addProperty(name, (Boolean)field);
                break;
            case ENUM:
                jsonObject.addProperty(name, ((Descriptors.EnumValueDescriptor)field).getName());
                break;
            case STRING:
                jsonObject.addProperty(name, (String)field);
                break;
            default:
                throw new RuntimeException(String.format("Cannot handle message subtype %s, field %s", fd.getType(), fd.getFullName()));
        }
    }

    public static String protoToJSONWithPrefix(MessageOrBuilder proto, String namePrefix) {
        JsonObject res = new JsonObject();
        Descriptors.Descriptor d = proto.getDescriptorForType();

        for (Descriptors.FieldDescriptor fd: d.getFields()) {
            String name = namePrefix + fd.getName();
            if (fd.isRepeated()) {
                if (proto.getRepeatedFieldCount(fd) > 1) {
                    throw new RuntimeException(String.format("Cannot convert proto to json with repeated size > 1, field %s", fd.getFullName()));
                }
                if (proto.getRepeatedFieldCount(fd) == 1) {
                    addJsonProperty(res, name, fd, proto.getRepeatedField(fd, 0));
                }
            } else {
                if (proto.hasField(fd)) {
                    addJsonProperty(res, name, fd, proto.getField(fd));
                }
            }
        }
        return res.toString();
    }
}
