package ru.yandex.travel.hotels.models.booking_flow;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.Builder;
import lombok.Value;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.jackson.MoneySerializersModule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.schedule.OfferPaymentSchedule;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;

@Value
@Builder(toBuilder = true)
@JsonDeserialize(builder = Offer.OfferBuilder.class)
public class Offer {
    private HotelInfo hotelInfo;
    private RateInfo rateInfo;
    private RoomInfo roomInfo;
    private MetaInfo metaInfo;
    private StayInfo stayInfo;
    private RefundRules refundRules;
    private LegalInfo legalInfo;
    private PartnerHotelInfo partnerHotelInfo;
    private PromoCampaignsInfo promoCampaignsInfo;
    private OfferPaymentSchedule deferredPaymentSchedule;
    private boolean allowPostPay;
    private DiscountInfo discountInfo;
    private boolean allGuestsRequired;

    private final static ObjectMapper mapper = new ObjectMapper()
            .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
            .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
            .setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE)
            .registerModule(new JavaTimeModule())
            .registerModule(new Jdk8Module())
            .registerModule(new MoneySerializersModule());

    public static Offer fromJsonNode(JsonNode node) {
        try {
            return mapper.treeToValue(node, Offer.class);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    public JsonNode toJsonNode() {
        return mapper.valueToTree(this);
    }

    public OfferState checkOfferState() {
        if (rateInfo != null && rateInfo.getStatus() != RateStatus.CONFIRMED) {
            return OfferState.PRICE_CONFLICT;
        } else if ((hotelInfo == null) || (partnerHotelInfo == null) || (rateInfo == null) || (roomInfo == null) ||
                (stayInfo == null) || (refundRules == null) || (legalInfo == null)) {
            return OfferState.MISSING_DATA;
        } else {
            return OfferState.READY;
        }
    }

    /**
     * @return discounted price or rate info's price if discounted is not set
     */
    @JsonIgnore
    public Money calculateActualPrice() {
        // todo(tlg-13,tivelkov): find out if we should check discountInfo.applied here
        return discountInfo != null ? discountInfo.getDiscountedPrice() :
                rateInfo != null ? rateInfo.getTotalMoney() : null;
    }

    @JsonPOJOBuilder(withPrefix = "")
    public static class OfferBuilder {
    }
}
