package ru.yandex.travel.hotels.common.orders;

import java.time.Instant;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;
import org.javamoney.moneta.Money;

import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;
import ru.yandex.travel.hotels.proto.EPartnerId;

@Data
@NoArgsConstructor
@AllArgsConstructor
@SuperBuilder
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonAutoDetect(fieldVisibility = JsonAutoDetect.Visibility.ANY, getterVisibility = JsonAutoDetect.Visibility.NONE)
public class HotelItinerary {
    private String serviceId;
    private String customerEmail;
    private String customerPhone;
    private String customerIp;
    private String customerUserAgent;
    private boolean allowsSubscription;
    private List<Guest> guests;
    private Instant generatedAtInstant;
    private Instant expiresAtInstant;
    /**
     * Full price for the user (without discount)
     */
    private Money fiscalPrice;
    private Money commission;
    private RefundRules refundRules;
    private String uiPayloadType;
    private JsonNode uiPayload;
    private OrderDetails orderDetails;
    private ConfirmationInfo confirmation;
    private RefundInfo refundInfo;
    private CancellationDetails orderCancellationDetails;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private AppliedPromoCampaigns appliedPromoCampaigns;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private PromoCampaignsInfo activePromoCampaigns;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private MealData mealData;

    @Nullable
    private Money discount;
    /**
     * Money the hotel charges. In most cases matches {@link #fiscalPrice}. But if the original price has kopeck value
     * or when dolphin changed the price, the value differs.
     *
     * Must be set for {@link DolphinHotelItinerary}, for others can be null.
     */
    @Nullable
    private Money actualPrice;
    private HotelItineraryOriginalInfo originalInfo;
    @Nullable
    private HotelUserInfo userInfo;

    public void addExtra(Money money) {
        fiscalPrice = fiscalPrice.add(money);
        if (actualPrice != null) {
            actualPrice = actualPrice.add(money);
        }
    }

    public void removeExtra(Money money) {
        fiscalPrice = fiscalPrice.subtract(money);
        if (actualPrice != null) {
            actualPrice = actualPrice.subtract(money);
        }
    }

    /**
     * If actual price is set, returns the actual price. Otherwise the fiscal price is the real one.
     */
    @Nonnull
    @JsonIgnore
    public Money getRealHotelPrice() {
        return Objects.requireNonNullElse(getActualPrice(), getFiscalPrice());
    }

    @JsonIgnore
    public Money getPriceAfterDiscount() {
        if (discount == null) {
            return fiscalPrice;
        } else {
            return fiscalPrice.subtract(discount);
        }
    }

    public EPartnerId getPartnerId() {
        // must be overwritten by descendants. The class can't be made abstract ATM because of usage in API.
        return EPartnerId.UNRECOGNIZED;
    }

    private PostPayInfo postPay = new PostPayInfo(false, false);

    @Data
    @AllArgsConstructor
    @NoArgsConstructor
    public static class PostPayInfo {
        private boolean eligible;
        private boolean used;

        public boolean applied() {
            return eligible && used;
        }
    }
}
