package ru.yandex.travel.hotels.common.pansions;

import com.google.common.collect.ImmutableMap;
import ru.yandex.travel.hotels.proto.EPansionType;

import java.util.HashMap;
import java.util.List;


public class PansionUnifier {

    static public class UnmatchedPansionException extends Exception {
    }

    private PansionUnifier() {
    }

    public static EPansionType combine(List<EPansionType> pansions) {
        boolean ai = false;
        boolean fb = false;
        boolean hb = false;
        boolean breakfast = false;
        boolean dinner = false;
        for (EPansionType pansionType: pansions) {
            switch (pansionType) {
                case PT_AI:
                case PT_UAI:
                case PT_LAI:
                    ai = true;
                    break;
                case PT_BB:
                    breakfast = true;
                    break;
                case PT_FB:
                    fb = true;
                    break;
                case PT_HB:
                    hb = true;
                    break;
                case PT_BD:
                    dinner = true;
                    break;
            }
        }
        return get(ai, fb, hb, breakfast, false, dinner);
    }

    public static EPansionType get(boolean ai, boolean fb, boolean hb, boolean breakfast, boolean lunch, boolean dinner) {
        EPansionType pansion;
        if (ai) pansion = EPansionType.PT_AI;
        else if (fb) pansion = EPansionType.PT_FB;
        else if (hb) pansion = EPansionType.PT_HB;
        else if (breakfast && lunch && dinner) pansion = EPansionType.PT_FB;
        else if (breakfast && !lunch && dinner) pansion = EPansionType.PT_HB;
        else if (breakfast) pansion = EPansionType.PT_BB;  // this is either "breakfast only" or "breakfast + lunch but no dinner", as of TRAVELBACK-1608
        else if (dinner) pansion = EPansionType.PT_BD;
        else pansion = EPansionType.PT_RO;
        return pansion;
    }

    public static EPansionType get(String key) throws PansionUnifier.UnmatchedPansionException {
        EPansionType pansion = pansions.get(normalize(key));
        if (pansion == null)
            throw new PansionUnifier.UnmatchedPansionException();
        return pansion;
    }

    public static String normalize(String raw) {
        return raw
                .replaceAll("\\+", " ")
                .trim()
                .replaceAll("[ ]+", "_")
                .toUpperCase();
    }

    private static final ImmutableMap<String, EPansionType> pansions;
    private static final ImmutableMap<String, EPansionType> rawPansions = new ImmutableMap.Builder<String, EPansionType>()
            .put("unknown", EPansionType.PT_UNKNOWN)

            .put("24a", EPansionType.PT_AI)
            .put("все включено элегансе", EPansionType.PT_AI)
            .put("все включено gourmet", EPansionType.PT_AI)
            .put("все включено light", EPansionType.PT_AI)
            .put("все включено select", EPansionType.PT_AI)
            .put("все включено soft", EPansionType.PT_AI)
            .put("все включено", EPansionType.PT_AI)
            .put("всё включено", EPansionType.PT_AI)
            .put("все включено+оздоровление", EPansionType.PT_AI)
            .put("лайт все включено", EPansionType.PT_AI)
            .put("питание по системе «все включено» входит в стоимость номера", EPansionType.PT_AI)
            .put("a'la carte all inclusive", EPansionType.PT_AI)
            .put("aea", EPansionType.PT_AI)
            .put("al + лечение", EPansionType.PT_AI)
            .put("al + оздоровит. путевка", EPansionType.PT_AI)
            .put("al basic", EPansionType.PT_AI)
            .put("al select", EPansionType.PT_AI)
            .put("al3", EPansionType.PT_AI)
            .put("ala carte all inclusive", EPansionType.PT_AI)
            .put("alc", EPansionType.PT_AI)
            .put("all + оздоровит. путевка", EPansionType.PT_AI)
            .put("all in 3 (legendary concept)", EPansionType.PT_AI)
            .put("all in 3 legendary concept", EPansionType.PT_AI)
            .put("all inclusive & spa", EPansionType.PT_AI)
            .put("all inclusive half moon plan", EPansionType.PT_AI)
            .put("all inclusive light", EPansionType.PT_AI)
            .put("all inclusive select", EPansionType.PT_AI)
            .put("all inclusive special", EPansionType.PT_AI)
            .put("all inclusive ultimate in luxury", EPansionType.PT_AI)
            .put("all inclusive", EPansionType.PT_AI)
            .put("all park use", EPansionType.PT_AI)
            .put("all", EPansionType.PT_AI)
            .put("all+t", EPansionType.PT_AI)
            .put("alp", EPansionType.PT_AI)
            .put("appetite for luxury", EPansionType.PT_AI)
            .put("basic all", EPansionType.PT_AI)
            .put("daa", EPansionType.PT_AI)
            .put("dah", EPansionType.PT_AI)
            .put("e-class", EPansionType.PT_AI)
            .put("eclass", EPansionType.PT_AI)
            .put("elements all inc da", EPansionType.PT_AI)
            .put("exal", EPansionType.PT_AI)
            .put("exclusive all inclusive", EPansionType.PT_AI)
            .put("family", EPansionType.PT_AI)
            .put("gourmet incl.", EPansionType.PT_AI)
            .put("hci", EPansionType.PT_AI)
            .put("hea", EPansionType.PT_AI)
            .put("includes breakfest, dinner & resort credit in 2 hotels and the theme park", EPansionType.PT_AI)
            .put("lai", EPansionType.PT_AI)
            .put("lux.me", EPansionType.PT_AI)
            .put("luxme", EPansionType.PT_AI)
            .put("occidental +", EPansionType.PT_AI)
            .put("palazzo all inclusive", EPansionType.PT_AI)
            .put("platinum package", EPansionType.PT_AI)
            .put("pre. all inclusive life style", EPansionType.PT_AI)
            .put("premium", EPansionType.PT_AI)
            .put("privileged life", EPansionType.PT_AI)
            .put("residence club all inclusive", EPansionType.PT_AI)
            .put("s-class", EPansionType.PT_AI)
            .put("sa", EPansionType.PT_AI)
            .put("select all inclusive", EPansionType.PT_AI)
            .put("superior", EPansionType.PT_AI)
            .put("t vip", EPansionType.PT_AI)
            .put("top vip concept", EPansionType.PT_AI)
            .put("ual", EPansionType.PT_AI)
            .put("ui", EPansionType.PT_AI)

            .put("американский завтрак включен", EPansionType.PT_BB)
            .put("американский завтрак входит в стоимость номера", EPansionType.PT_BB)
            .put("американский завтрак", EPansionType.PT_BB)
            .put("английский завтрак входит в стоимость", EPansionType.PT_BB)
            .put("английский завтрак", EPansionType.PT_BB)
            .put("бесплатный завтрак", EPansionType.PT_BB)
            .put("бесплатный обед", EPansionType.PT_BB)
            .put("завтрак «шведский стол» входит в стоимость", EPansionType.PT_BB)
            .put("завтрак «шведский стол»", EPansionType.PT_BB)
            .put("завтрак (шведский стол", EPansionType.PT_BB)
            .put("завтрак (bb)", EPansionType.PT_BB)
            .put("завтрак \"английский\"", EPansionType.PT_BB)
            .put("завтрак \"континентальный\"", EPansionType.PT_BB)
            .put("завтрак \"шведский стол\" входит в стоимость номера", EPansionType.PT_BB)
            .put("завтрак \"шведский стол\"", EPansionType.PT_BB)
            .put("завтрак + лечение relax", EPansionType.PT_BB)
            .put("завтрак буфет с лечением", EPansionType.PT_BB)
            .put("завтрак буфет", EPansionType.PT_BB)
            .put("завтрак включен ", EPansionType.PT_BB)
            .put("завтрак включен в стоимость номера", EPansionType.PT_BB)
            .put("завтрак включен", EPansionType.PT_BB)
            .put("завтрак для двоих входит в стоимость номера", EPansionType.PT_BB)
            .put("завтрак для двоих", EPansionType.PT_BB)
            .put("завтрак континентальный", EPansionType.PT_BB)
            .put("завтрак «континентальный»", EPansionType.PT_BB)
            .put("завтрак на двоих", EPansionType.PT_BB)
            .put("завтрак плюс", EPansionType.PT_BB)
            .put("завтрак шведский стол", EPansionType.PT_BB)
            .put("завтрак", EPansionType.PT_BB)
            .put("завтрак+оздоровление", EPansionType.PT_BB)
            .put("израильский завтрак", EPansionType.PT_BB)
            .put("континентальный завтрак включен", EPansionType.PT_BB)
            .put("континентальный завтрак входит в стоимость номера", EPansionType.PT_BB)
            .put("континентальный завтрак", EPansionType.PT_BB)
            .put("обед входит в стоимость номера", EPansionType.PT_BB)
            .put("обед входит в стоимость", EPansionType.PT_BB)
            .put("питание включено", EPansionType.PT_BB)
            .put("полный завтрак входит в стоимость номера", EPansionType.PT_BB)
            .put("сертификат на питание и/или напитки", EPansionType.PT_BB)
            .put("только завтрак", EPansionType.PT_BB)
            .put("только завтраки", EPansionType.PT_BB)
            .put("ab", EPansionType.PT_BB)
            .put("american breakfast included", EPansionType.PT_BB)
            .put("bb + sport", EPansionType.PT_BB)
            .put("bb", EPansionType.PT_BB)
            .put("bb+d", EPansionType.PT_BB)
            .put("bb+db", EPansionType.PT_BB)
            .put("bb+dcm", EPansionType.PT_BB)
            .put("bbb", EPansionType.PT_BB)
            .put("bbbt", EPansionType.PT_BB)
            .put("bbl", EPansionType.PT_BB)
            .put("bbr", EPansionType.PT_BB)
            .put("bbtl", EPansionType.PT_BB)
            .put("bed & breakfast with park access", EPansionType.PT_BB)
            .put("bed & breakfast", EPansionType.PT_BB)
            .put("bed and breakfast (bb)", EPansionType.PT_BB)
            .put("bed and breakfast beach", EPansionType.PT_BB)
            .put("bed and breakfast or dinner", EPansionType.PT_BB)
            .put("bed and breakfast", EPansionType.PT_BB)
            .put("bl", EPansionType.PT_BB)
            .put("breakfast for 1", EPansionType.PT_BB)
            .put("breakfast for 2", EPansionType.PT_BB)
            .put("breakfast included", EPansionType.PT_BB)
            .put("breakfast", EPansionType.PT_BB)
            .put("brunch buffe", EPansionType.PT_BB)
            .put("brunch buffet", EPansionType.PT_BB)
            .put("buffet breakfast included", EPansionType.PT_BB)
            .put("buffet breakfast", EPansionType.PT_BB)
            .put("buffet lunch", EPansionType.PT_BB)
            .put("cb", EPansionType.PT_BB)
            .put("continental breakfast for 2", EPansionType.PT_BB)
            .put("continental breakfast", EPansionType.PT_BB)
            .put("dine around", EPansionType.PT_BB)
            .put("dnr", EPansionType.PT_BB)
            .put("english breakfast", EPansionType.PT_BB)
            .put("full breakfast", EPansionType.PT_BB)
            .put("hld", EPansionType.PT_BB)
            .put("limited package", EPansionType.PT_BB)
            .put("lnc", EPansionType.PT_BB)
            .put("lunch", EPansionType.PT_BB)
            .put("room-bb", EPansionType.PT_BB)
            .put("soft", EPansionType.PT_BB)

            .put("ужин", EPansionType.PT_BD)
            .put("dinner included", EPansionType.PT_BD)
            .put("dinner only", EPansionType.PT_BD)
            .put("dinner", EPansionType.PT_BD)
            .put("free dinner", EPansionType.PT_BD)

            .put("3-х разовое", EPansionType.PT_FB)
            .put("3fbcm", EPansionType.PT_FB)
            .put("3fbcmt", EPansionType.PT_FB)
            .put("4fb+h", EPansionType.PT_FB)
            .put("4fbcm", EPansionType.PT_FB)
            .put("4fbcmt", EPansionType.PT_FB)
            .put("4fbt", EPansionType.PT_FB)
            .put("завтрак, обед и ужин входят в стоимость номера", EPansionType.PT_FB)
            .put("завтрак, обед и ужин входят в стоимость", EPansionType.PT_FB)
            .put("завтрак, обед и ужин", EPansionType.PT_FB)
            .put("завтрак, обед, ужин (plus)", EPansionType.PT_FB)
            .put("завтрак, обед, ужин", EPansionType.PT_FB)
            .put("заказное меню", EPansionType.PT_FB)
            .put("пансион плюс", EPansionType.PT_FB)
            .put("пансион", EPansionType.PT_FB)
            .put("полный пансион + лечение (гинекология)", EPansionType.PT_FB)
            .put("полный пансион + лечение (детокс)", EPansionType.PT_FB)
            .put("полный пансион + лечение (расширенное)", EPansionType.PT_FB)
            .put("полный пансион + лечение (стандарт)", EPansionType.PT_FB)
            .put("полный пансион + лечение (урология)", EPansionType.PT_FB)
            .put("полный пансион + лечение (эконом)", EPansionType.PT_FB)
            .put("полный пансион + лечение", EPansionType.PT_FB)
            .put("полный пансион +: завтрак, обед и ужин плюс привилегии (подробнее см. в описании отеля)", EPansionType.PT_FB)
            .put("полный пансион +: завтрак", EPansionType.PT_FB)
            .put("полный пансион +", EPansionType.PT_FB)
            .put("полный пансион", EPansionType.PT_FB)
            .put("полный пансион+оздоровление", EPansionType.PT_FB)
            .put("ai light", EPansionType.PT_FB)
            .put("dafb", EPansionType.PT_FB)
            .put("dine around fb", EPansionType.PT_FB)
            .put("fb (dine around)", EPansionType.PT_FB)
            .put("fb + лечение", EPansionType.PT_FB)
            .put("fb + оздоровит. путевка", EPansionType.PT_FB)
            .put("fb silver", EPansionType.PT_FB)
            .put("fb.", EPansionType.PT_FB)
            .put("fb+beach", EPansionType.PT_FB)
            .put("fb+dr", EPansionType.PT_FB)
            .put("fb+spa", EPansionType.PT_FB)
            .put("fb+t", EPansionType.PT_FB)
            .put("fb+trmnt", EPansionType.PT_FB)
            .put("fbal", EPansionType.PT_FB)
            .put("fbb", EPansionType.PT_FB)
            .put("fbbt", EPansionType.PT_FB)
            .put("fbcm", EPansionType.PT_FB)
            .put("fbda", EPansionType.PT_FB)
            .put("fbpp -premium full board", EPansionType.PT_FB)
            .put("fbr", EPansionType.PT_FB)
            .put("fbsilver", EPansionType.PT_FB)
            .put("fbt", EPansionType.PT_FB)
            .put("fbt4", EPansionType.PT_FB)
            .put("fbt5", EPansionType.PT_FB)
            .put("fbtl", EPansionType.PT_FB)
            .put("fbtr", EPansionType.PT_FB)
            .put("full board + all inclusive", EPansionType.PT_FB)
            .put("full board included", EPansionType.PT_FB)
            .put("full board plus", EPansionType.PT_FB)
            .put("full board premium", EPansionType.PT_FB)
            .put("full board silk way", EPansionType.PT_FB)
            .put("full board vip", EPansionType.PT_FB)
            .put("full board zimnee", EPansionType.PT_FB)
            .put("full board", EPansionType.PT_FB)
            .put("fullboard plus", EPansionType.PT_FB)
            .put("lunch and dinner included", EPansionType.PT_FB)
            .put("pfb", EPansionType.PT_FB)
            .put("premium fullboard", EPansionType.PT_FB)
            .put("vfb", EPansionType.PT_FB)
            .put("vip fb -vip full board", EPansionType.PT_FB)
            .put("vip fullboard", EPansionType.PT_FB)

            .put("2hb", EPansionType.PT_HB)
            .put("завтрак и обед входит в стоимость", EPansionType.PT_HB)
            .put("завтрак и обед входят в стоимость номера", EPansionType.PT_HB)
            .put("завтрак и обед входят в стоимость", EPansionType.PT_HB)
            .put("завтрак и ужин (или обед) входят в стоимость номера", EPansionType.PT_HB)
            .put("завтрак и ужин входит в стоимость", EPansionType.PT_HB)
            .put("завтрак и ужин входят в стоимость номера", EPansionType.PT_HB)
            .put("завтрак и ужин входят в стоимость", EPansionType.PT_HB)
            .put("завтрак и ужин", EPansionType.PT_HB)
            .put("завтрак, ужин (\"швед. стол\")", EPansionType.PT_HB)
            .put("завтрак, ужин (plus)", EPansionType.PT_HB)
            .put("завтрак, ужин", EPansionType.PT_HB)
            .put("завтрак+обед (шв.стол)", EPansionType.PT_HB)
            .put("завтраки,ужины", EPansionType.PT_HB)
            .put("нальзя менять ужин на обед", EPansionType.PT_HB)
            .put("обед и ужин входят в стоимость номера", EPansionType.PT_HB)
            .put("обед", EPansionType.PT_HB)
            .put("полупансион - завтрак и обед", EPansionType.PT_HB)
            .put("полупансион - завтрак и ужин", EPansionType.PT_HB)
            .put("полупансион (3автрак+обед)", EPansionType.PT_HB)
            .put("полупансион (завтрак + обед)", EPansionType.PT_HB)
            .put("полупансион (завтрак + ужин)", EPansionType.PT_HB)
            .put("полупансион (hb)", EPansionType.PT_HB)
            .put("полупансион +: завтрак и ужин плюс привилегии ", EPansionType.PT_HB)
            .put("полупансион +: завтрак и ужин плюс привилегии (подробнее см. в описании отеля)", EPansionType.PT_HB)
            .put("полупансион +: завтрак и ужин плюс привилегии", EPansionType.PT_HB)
            .put("полупансион плюс: завтрак, обед или ужин без напитков (подробнее см. в описании отеля)", EPansionType.PT_HB)
            .put("полупансион плюс: завтрак", EPansionType.PT_HB)
            .put("полупансион плюс", EPansionType.PT_HB)
            .put("полупансион по цене завтраков", EPansionType.PT_HB)
            .put("полупансион с лечением", EPansionType.PT_HB)
            .put("полупансион: завтрак и ужин плюс услуги пляжа плюс привилегии (подробнее см. в описании отеля)", EPansionType.PT_HB)
            .put("полупансион", EPansionType.PT_HB)
            .put("полупансион+оздоровление", EPansionType.PT_HB)
            .put("система dine around", EPansionType.PT_HB)
            .put("a'la carte hb", EPansionType.PT_HB)
            .put("alacarteda", EPansionType.PT_HB)
            .put("bb+lb", EPansionType.PT_HB)
            .put("bb=hb (spo)", EPansionType.PT_HB)
            .put("breakfast & dinner", EPansionType.PT_HB)
            .put("breakfast and lunch included", EPansionType.PT_HB)
            .put("dahb", EPansionType.PT_HB)
            .put("dine around halfboard", EPansionType.PT_HB)
            .put("dine around hb", EPansionType.PT_HB)
            .put("dine around on hb", EPansionType.PT_HB)
            .put("dining around", EPansionType.PT_HB)
            .put("elite package hb", EPansionType.PT_HB)
            .put("half board & spa", EPansionType.PT_HB)
            .put("half board a la cart", EPansionType.PT_HB)
            .put("half board plus", EPansionType.PT_HB)
            .put("half board premium plus", EPansionType.PT_HB)
            .put("half board", EPansionType.PT_HB)
            .put("halfboard + all inclusive", EPansionType.PT_HB)
            .put("halfboard extended", EPansionType.PT_HB)
            .put("hb (обед)", EPansionType.PT_HB)
            .put("hb (ужин)", EPansionType.PT_HB)
            .put("hb (dine around)", EPansionType.PT_HB)
            .put("hb (dinner)", EPansionType.PT_HB)
            .put("hb (lunch)", EPansionType.PT_HB)
            .put("hb + лечение", EPansionType.PT_HB)
            .put("hb + оздоровит. путевка", EPansionType.PT_HB)
            .put("hb alcarte", EPansionType.PT_HB)
            .put("hb all lgt", EPansionType.PT_HB)
            .put("hb dinner", EPansionType.PT_HB)
            .put("hb elite", EPansionType.PT_HB)
            .put("hb fontana restaurant", EPansionType.PT_HB)
            .put("hb lunch or dinner", EPansionType.PT_HB)
            .put("hb lunch", EPansionType.PT_HB)
            .put("hb silver", EPansionType.PT_HB)
            .put("hb st/apt", EPansionType.PT_HB)
            .put("hb.", EPansionType.PT_HB)
            .put("hb+bd", EPansionType.PT_HB)
            .put("hb+beach", EPansionType.PT_HB)
            .put("hb+spa", EPansionType.PT_HB)
            .put("hb1", EPansionType.PT_HB)
            .put("hb2", EPansionType.PT_HB)
            .put("hb3", EPansionType.PT_HB)
            .put("hba", EPansionType.PT_HB)
            .put("hbal", EPansionType.PT_HB)
            .put("hbbt", EPansionType.PT_HB)
            .put("hbd", EPansionType.PT_HB)
            .put("hbda", EPansionType.PT_HB)
            .put("hbl", EPansionType.PT_HB)
            .put("hbp", EPansionType.PT_HB)
            .put("hbr", EPansionType.PT_HB)
            .put("hbs", EPansionType.PT_HB)
            .put("hbsilver", EPansionType.PT_HB)
            .put("hbt", EPansionType.PT_HB)
            .put("hbtl", EPansionType.PT_HB)
            .put("hbtr", EPansionType.PT_HB)
            .put("hd", EPansionType.PT_HB)
            .put("lhb", EPansionType.PT_HB)
            .put("lnc+d", EPansionType.PT_HB)
            .put("phb", EPansionType.PT_HB)
            .put("premium hb", EPansionType.PT_HB)
            .put("snacks ai", EPansionType.PT_HB)

            .put("всё включено (light)", EPansionType.PT_LAI)
            .put("all inclusive soft", EPansionType.PT_LAI)
            .put("da", EPansionType.PT_LAI)
            .put("light all inclusive", EPansionType.PT_LAI)
            .put("light inclusive", EPansionType.PT_LAI)
            .put("mal", EPansionType.PT_LAI)
            .put("soa", EPansionType.PT_LAI)
            .put("soft all inclusive", EPansionType.PT_LAI)

            .put("-", EPansionType.PT_RO)
            .put("без питания", EPansionType.PT_RO)
            .put("напитки и закуски", EPansionType.PT_RO)
            .put("не предоставляется", EPansionType.PT_RO)
            .put("размещение без питания", EPansionType.PT_RO)
            .put("скидочный купон на питание", EPansionType.PT_RO)
            .put("bed only", EPansionType.PT_RO)
            .put("bo", EPansionType.PT_RO)
            .put("ep", EPansionType.PT_RO)
            .put("ext", EPansionType.PT_RO)
            .put("no", EPansionType.PT_RO)
            .put("ob", EPansionType.PT_RO)
            .put("only bed (ob)", EPansionType.PT_RO)
            .put("pp", EPansionType.PT_RO)
            .put("residence concept", EPansionType.PT_RO)
            .put("room only", EPansionType.PT_RO)
            .put("room rate", EPansionType.PT_RO)
            .put("rr", EPansionType.PT_RO)
            .put("sc", EPansionType.PT_RO)
            .put("self catering", EPansionType.PT_RO)
            .put("sl", EPansionType.PT_RO)

            .put("24 hours al", EPansionType.PT_UAI)
            .put("всё включено (exclus)", EPansionType.PT_UAI)
            .put("все включено +", EPansionType.PT_UAI)
            .put("все включено премиум", EPansionType.PT_UAI)
            .put("все включено gold", EPansionType.PT_UAI)
            .put("все включено ultra all", EPansionType.PT_UAI)
            .put("ультра все включено", EPansionType.PT_UAI)
            .put("ультра всё включено", EPansionType.PT_UAI)
            .put("all excl.all inclus", EPansionType.PT_UAI)
            .put("all gold", EPansionType.PT_UAI)
            .put("all inclusive gold", EPansionType.PT_UAI)
            .put("all inclusive premium", EPansionType.PT_UAI)
            .put("all inclusive with superior service", EPansionType.PT_UAI)
            .put("all inclusivepremium", EPansionType.PT_UAI)
            .put("all ultra premium", EPansionType.PT_UAI)
            .put("axi", EPansionType.PT_UAI)
            .put("club pharao all inclusive", EPansionType.PT_UAI)
            .put("cpa", EPansionType.PT_UAI)
            .put("dal", EPansionType.PT_UAI)
            .put("deluxe all incl", EPansionType.PT_UAI)
            .put("deluxe all inclusive", EPansionType.PT_UAI)
            .put("diamond all inclusive", EPansionType.PT_UAI)
            .put("eal", EPansionType.PT_UAI)
            .put("ela all inclusive", EPansionType.PT_UAI)
            .put("elegance all incl", EPansionType.PT_UAI)
            .put("elegance all inclusive", EPansionType.PT_UAI)
            .put("extensive all inclusive", EPansionType.PT_UAI)
            .put("gold inclusive", EPansionType.PT_UAI)
            .put("hard all inclusive", EPansionType.PT_UAI)
            .put("hca", EPansionType.PT_UAI)
            .put("high class all inc", EPansionType.PT_UAI)
            .put("high class all inclusive", EPansionType.PT_UAI)
            .put("high end all inclusive", EPansionType.PT_UAI)
            .put("lal", EPansionType.PT_UAI)
            .put("lua", EPansionType.PT_UAI)
            .put("luxury all", EPansionType.PT_UAI)
            .put("max all inclusive", EPansionType.PT_UAI)
            .put("pall", EPansionType.PT_UAI)
            .put("plual", EPansionType.PT_UAI)
            .put("premier ua", EPansionType.PT_UAI)
            .put("premierua", EPansionType.PT_UAI)
            .put("premium +", EPansionType.PT_UAI)
            .put("premium all inclusive", EPansionType.PT_UAI)
            .put("premium plus all inclusive", EPansionType.PT_UAI)
            .put("premium plus", EPansionType.PT_UAI)
            .put("pua", EPansionType.PT_UAI)
            .put("pual", EPansionType.PT_UAI)
            .put("royal all inclusive", EPansionType.PT_UAI)
            .put("sclassua", EPansionType.PT_UAI)
            .put("sua", EPansionType.PT_UAI)
            .put("super all inclusive", EPansionType.PT_UAI)
            .put("superior all inclusive", EPansionType.PT_UAI)
            .put("ua", EPansionType.PT_UAI)
            .put("ua+", EPansionType.PT_UAI)
            .put("uai", EPansionType.PT_UAI)
            .put("ultra all exclusive", EPansionType.PT_UAI)
            .put("ultra all inclusive", EPansionType.PT_UAI)
            .put("ultra all park use", EPansionType.PT_UAI)
            .put("ultra deluxe all inclusive", EPansionType.PT_UAI)
            .put("unlimited services", EPansionType.PT_UAI)
            .put("vc", EPansionType.PT_UAI)
            .put("vip concept", EPansionType.PT_UAI)
            .put("vip ultra all inclusive", EPansionType.PT_UAI)
            .put("vipc", EPansionType.PT_UAI)
            .put("vipconcept", EPansionType.PT_UAI)
            .build();

    static {
        HashMap<String, EPansionType> normalizedPansions = new HashMap<>();
        rawPansions.forEach((k, v) -> normalizedPansions.put(normalize(k), v));
        pansions = ImmutableMap.copyOf(normalizedPansions);
    }
}
