package ru.yandex.travel.hotels.common.partners.base;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeoutException;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import lombok.Getter;
import org.asynchttpclient.Response;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.travel.commons.logging.IAsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.masking.LogAwareBodyGenerator;
import ru.yandex.travel.commons.logging.masking.LogAwareRequestBuilder;
import ru.yandex.travel.commons.retry.BaseRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableHttpException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.RetryableIOException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnableToParseResponseException;
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnexpectedHttpStatusCodeException;

public abstract class BaseClient<T extends BaseClientProperties> {
    protected final static Set<HttpMethod> METHODS_WITH_BODY = Set.of(HttpMethod.POST, HttpMethod.PUT,
            HttpMethod.DELETE);

    protected final T properties;
    protected final IAsyncHttpClientWrapper clientWrapper;
    protected final ObjectMapper objectMapper;
    protected final Retry retryHelper;


    protected BaseClient(T properties, IAsyncHttpClientWrapper clientWrapper, ObjectMapper objectMapper,
                         Retry retryHelper) {
        this.properties = properties;
        this.clientWrapper = clientWrapper;
        this.objectMapper = objectMapper;
        this.retryHelper = retryHelper;
    }

    protected abstract ClientMethods getClientMethods();

    protected <R> R handleResult(R searchResponse, Throwable throwable) {
        if (throwable == null) {
            return searchResponse;
        } else {
            Exception actualException = Retry.unwrapExecutionException((Exception) throwable);
            if (actualException instanceof TimeoutException) {
                throw new RetryableHttpException((TimeoutException) actualException);
            } else if (actualException instanceof IOException) {
                throw new RetryableIOException(actualException);
            } else if (actualException instanceof RuntimeException) {
                throw (RuntimeException) actualException;
            } else {
                throw new PartnerException(actualException);
            }
        }
    }

    @SuppressWarnings("unchecked")
    protected <R> Class<R> getResponseClass(String method) {
        var endpoint = getClientMethods().get(method);
        Preconditions.checkNotNull(endpoint, "Unknown method " + method);
        return endpoint.get3();
    }

    protected String getEndpoint(String method, List<String> urlParams) {
        var endpoint = getClientMethods().get(method);
        Preconditions.checkNotNull(endpoint, "Unknown method " + method);
        String endpointTemplate = endpoint.get1();
        if (urlParams != null) {
            endpointTemplate = MessageFormat.format(endpointTemplate, urlParams.toArray());
        }
        return endpointTemplate;
    }

    protected HttpMethod getHttpMethod(String method) {
        var endpoint = getClientMethods().get(method);
        Preconditions.checkNotNull(endpoint, "Unknown method " + method);
        return endpoint.get2();
    }

    protected Set<Integer> getSupportedResponseCodes(String method) {
        var endpoint = getClientMethods().get(method);
        Preconditions.checkNotNull(endpoint, "Unknown method " + method);
        return endpoint.get4();
    }

    protected <R, B> CompletableFuture<R> call(String destinationMethod) {
        return call(destinationMethod, new CallArguments());
    }

    @NotNull
    protected Map<String, String> getCommonHeaders(String destinationMethod) {
        return Collections.emptyMap();
    }

    protected <R, B> CompletableFuture<R> call(String destinationMethod, CallArguments params) {
        CompletableFuture<R> responseFuture;
        String endpoint = getEndpoint(destinationMethod, params.getUrlParams());
        HttpMethod httpMethod = getHttpMethod(destinationMethod);
        Set<Integer> responseCodes = getSupportedResponseCodes(destinationMethod);
        var requestBuilder = prepareRequestBuilder(destinationMethod, endpoint, httpMethod, params);
        Class<R> responseClass = getResponseClass(destinationMethod);
        IAsyncHttpClientWrapper.ResponseParser<R> responseParser = (r) -> responseAs(r, responseClass,
                responseCodes);
        if (properties.isEnableRetries()) {
            responseFuture = retryHelper.withRetry("PartnerClient",
                    (tuple) -> executeRequestWrapped(tuple),
                    Tuple4.tuple(requestBuilder, destinationMethod, params.getRequestId(), responseParser),
                    new BaseRetryStrategy<>());
        } else {
            responseFuture = clientWrapper.executeRequest(requestBuilder, destinationMethod,
                    params.getRequestId(),
                    responseParser);
        }
        return responseFuture.handle(this::handleResult);
    }

    private <R> CompletableFuture<R> executeRequestWrapped(Tuple4<LogAwareRequestBuilder, String, String,
            IAsyncHttpClientWrapper.ResponseParser<R>> input) {
        return clientWrapper.executeRequest(input.get1(), input.get2(), input.get3(), input.get4());
    }

    protected <E extends Exception> E unwrapExceptionAs(Throwable t, Class<E> exceptionClass) {
        if (t instanceof Exception) {
            Exception e = Retry.unwrapExecutionException((Exception) t);
            if (exceptionClass.equals(e.getClass())) {
                return (E) e;
            } else {
                if (t instanceof CompletionException) {
                    throw (CompletionException) t;
                } else {
                    throw new CompletionException(t);
                }
            }
        } else {
            throw new CompletionException(t);
        }
    }

    protected <R> R responseAs(Response response, Class<R> responseType, Set<Integer> responseCodes) {
        if (!responseCodes.contains(response.getStatusCode())) {
            if (response.getStatusCode() / 100 == 5) {
                throw new RetryableHttpException(response.getStatusCode(), response.getResponseBody());
            } else {
                throw new UnexpectedHttpStatusCodeException(response.getStatusCode(), response.getResponseBody());
            }
        }
        if (responseType.equals(IgnoredBodyResponse.class)) {
            return (R) new IgnoredBodyResponse(response.getStatusCode());
        }
        if (responseType.equals(EmptyResponse.class)) {
            Preconditions.checkArgument(response.getResponseBody().length() == 0,
                    "Unexpected non-empty response body");
            return (R) new EmptyResponse(response.getStatusCode());
        }
        return bodyAs(response.getResponseBody(), responseType);
    }

    protected <R> R bodyAs(String body, Class<R> type) {
        try {
            return objectMapper.readerFor(type).readValue(body);
        } catch (IOException e) {
            throw new UnableToParseResponseException(body, e);
        }
    }

    protected String getBaseUrl(String destinationMethod) {
        return properties.getBaseUrl();
    }

    protected LogAwareRequestBuilder prepareRequestBuilder(String destinationMethod, String endpoint, HttpMethod httpMethod,
                                                   CallArguments params) {
        if (!endpoint.startsWith("/") && !properties.getBaseUrl().endsWith("/")) {
            endpoint = "/" + endpoint;
        }
        LogAwareRequestBuilder builder = (LogAwareRequestBuilder) new LogAwareRequestBuilder()
                .setReadTimeout(Math.toIntExact(properties.getHttpReadTimeout().toMillis()))
                .setRequestTimeout(Math.toIntExact(properties.getHttpRequestTimeout().toMillis()))
                .setUrl(getBaseUrl(destinationMethod) + endpoint)
                .setMethod(httpMethod.toString());
        if (METHODS_WITH_BODY.contains(httpMethod)) {
            if (params.getBody() != null) {
                Preconditions.checkArgument(params.getFormParams().isEmpty(),
                        "Method body cannot be combined with form params");
                builder.setBody(LogAwareBodyGenerator.of(params.getBody(), objectMapper));
                builder.setHeader("Content-Type", "application/json");
            }
            if (params.getFormParams().size() > 0) {
                Preconditions.checkArgument(params.getBody() == null,
                        "Form params cannot be combined with method body");
                params.getFormParams().forEach(p -> builder.addFormParam(p));
            }
        }
        getCommonHeaders(destinationMethod).forEach(builder::addHeader);
        params.getHeaders().forEach(builder::addHeader);
        params.getQueryParams().forEach(t -> builder.addQueryParam(t.get1(), t.get2()));
        return builder;
    }

    public enum HttpMethod {
        GET,
        POST,
        PUT,
        DELETE
    }

    @Getter
    public static class CallArguments {
        private final List<Tuple2<String, String>> queryParams;
        private final List<LogAwareRequestBuilder.FormParam> formParams;
        private final Map<String, String> headers;
        private List<String> urlParams;
        private Object body;
        private String requestId;

        public CallArguments() {
            queryParams = new ArrayList<>();
            formParams = new ArrayList<>();
            headers = new HashMap<>();
            urlParams = null;
        }

        public CallArguments withBody(Object body) {
            this.body = body;
            return this;
        }

        public CallArguments withRequestId(String requestId) {
            this.requestId = requestId;
            return this;
        }

        public CallArguments withHeader(String name, String value) {
            headers.put(name, value);
            return this;
        }

        public CallArguments withQueryParam(String name, String value) {
            queryParams.add(Tuple2.tuple(name, value));
            return this;
        }

        public CallArguments withFormParam(String name, String value) {
            return this.withFormParam(LogAwareRequestBuilder.FormParam.of(name, value));
        }

        public CallArguments withFormParam(LogAwareRequestBuilder.FormParam param) {
            formParams.add(param);
            return this;
        }

        public CallArguments withUrlParams(List<String> values) {
            urlParams = values;
            return this;
        }

        public CallArguments withUrlParams(String... values) {
            urlParams = List.of(values);
            return this;
        }
    }
}
