package ru.yandex.travel.hotels.common.partners.base;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.LocalDateTime;
import java.util.EnumMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.BiFunction;
import java.util.function.Function;

import javax.xml.ws.AsyncHandler;
import javax.xml.ws.http.HTTPException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableMap;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.Logger;
import org.slf4j.MDC;

import ru.yandex.travel.commons.logging.LogEventRequest;
import ru.yandex.travel.commons.logging.LogEventResponse;
import ru.yandex.travel.commons.logging.LogEventType;
import ru.yandex.travel.commons.logging.LoggingMarkers;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.logging.masking.LogMaskingConverter;
import ru.yandex.travel.commons.metrics.MetricsUtils;

@Slf4j
public abstract class BaseSOAPClient<T extends BaseSOAPClientProperties> {

    private final ConcurrentMap<String, Meters> metersForOperators;
    private final String localHostName;
    private final String destinationName;


    protected final ObjectMapper logObjectMapper;
    protected final T properties;
    protected Logger logger;

    public BaseSOAPClient(T properties, Logger logger, String destinationName) {
        this.properties = properties;
        this.localHostName = getLocalHostName();
        this.logger = logger;
        this.destinationName = destinationName;
        this.logObjectMapper = LogMaskingConverter.getObjectMapperForLogEvents();
        metersForOperators = new ConcurrentHashMap<>();
    }

    protected abstract SOAPOperations getOperations();


    <R> void logRequest(String operation, R request, String yaCallId, Map<String, String> mdc) {
        LogEventRequest logEventRequest = new LogEventRequest();
        logEventRequest.setTimestamp(System.currentTimeMillis());
        logEventRequest.setDateTime(LocalDateTime.now());
        logEventRequest.setDestinationName(destinationName);
        logEventRequest.setDestinationMethod(operation);

        logEventRequest.setCallId(yaCallId);
        logEventRequest.setFqdn(localHostName);
        logEventRequest.setRequestMethod("POST");
        logEventRequest.setRequestBody(request);
        logEventRequest.setMdc(mdc);
        try {
            logger.info(LoggingMarkers.HTTP_REQUEST_RESPONSE_MARKER, logObjectMapper.writeValueAsString(logEventRequest));
        } catch (JsonProcessingException e) {
            log.error("Unable to log request", e);
        }
    }

    public <R, A> CompletableFuture<A> call(
            String operation,
            R request,
            String requestId) {

        String yaCallId;
        if (requestId != null) {
            yaCallId = requestId;
        } else {
            yaCallId = UUID.randomUUID().toString();
        }

        var meters = metersForOperators.computeIfAbsent(operation, k -> new Meters(
                destinationName,
                operation
        ));
        Map<String, String> mdc = MDC.getCopyOfContextMap();

        logRequest(operation, request, yaCallId, mdc);

        long requestSentTime = System.currentTimeMillis();

        Function<R, CompletableFuture<A>> futureSupplier = getFutureSupplier(operation);
        CompletableFuture<A> completableFuture = futureSupplier.apply(request);

        completableFuture.whenComplete((r, t) -> {
            long responseTime = (System.currentTimeMillis() - requestSentTime);
            LogEventResponse logEventResponse = new LogEventResponse();
            logEventResponse.setTimestamp(System.currentTimeMillis());
            logEventResponse.setDateTime(LocalDateTime.now());
            logEventResponse.setDestinationName(destinationName);
            logEventResponse.setDestinationMethod(operation);
            logEventResponse.setCallId(yaCallId);
            logEventResponse.setFqdn(localHostName);
            logEventResponse.setResponseTime(responseTime);
            logEventResponse.setMdc(mdc);
            SOAPStatus status = SOAPStatus.OK;
            if (t != null) {
                logEventResponse.setEventKind(LogEventType.EXCEPTION.getValue());
                logEventResponse.setException(t);
                String exceptionClassName;
                if (t instanceof ExecutionException) {
                    exceptionClassName = t.getCause().getClass().getCanonicalName();
                } else {
                    exceptionClassName = t.getClass().getCanonicalName();
                }
                logEventResponse.setExceptionClass(exceptionClassName);
                meters.errorWithExceptionCounters.computeIfAbsent(exceptionClassName,
                        k -> createExceptionCounter(operation, exceptionClassName)).increment();
                meters.errorCounter.increment();

                if (t instanceof ExecutionException) {
                    var error = t.getCause();
                    if (error.getClass().equals(TimeoutException.class)) {
                        status = SOAPStatus.TIMEOUT;
                    } else if (error instanceof HTTPException) {
                        status = SOAPStatus.HTTP_ERROR;
                    } else {
                        status = SOAPStatus.ERROR;
                    }
                } else {
                    status = SOAPStatus.ERROR;
                }
            }
            if (r != null) {
                logEventResponse.setResponseBody(r);
                meters.timer.record(responseTime, TimeUnit.MILLISECONDS);
            }
            meters.callCounters.get(status).increment();
            meters.callCounter.increment();
            try (var ignored = NestedMdc.empty()) {
                if (mdc != null) {
                    MDC.setContextMap(mdc);
                }
                logger.info(LoggingMarkers.HTTP_REQUEST_RESPONSE_MARKER,
                        logObjectMapper.writeValueAsString(logEventResponse));
            } catch (JsonProcessingException e) {
                log.error("Unable to log response", e);
            }
        });
        return completableFuture;
    }

    private <R, A> Function<R, CompletableFuture<A>> getFutureSupplier(String operationName) {
        BiFunction <R, AsyncHandler<A>, Future<?>> executor = getExecutor(operationName);
        return (R request) -> {
            CompletableFuture<A> completableFuture = new CompletableFuture<>();
            executor.apply(request, res -> {
                try {
                    var ans = res.get();
                    completableFuture.complete(ans);
                } catch (ExecutionException | InterruptedException e) {
                    completableFuture.completeExceptionally(e);
                }
            });
            return completableFuture;
        };
    }

    @SuppressWarnings("unchecked")
    protected <R, A> BiFunction <R, AsyncHandler<A>, Future<?>> getExecutor(String operation) {
        return (BiFunction <R, AsyncHandler<A>, Future<?>>) getOperations().get(operation).get2();
    }

    private String getLocalHostName() {
        try {
            return InetAddress.getLocalHost().getCanonicalHostName();
        } catch (UnknownHostException e) {
            throw new RuntimeException("Unknown host", e);
        }
    }

    private Counter createExceptionCounter(String operation, String exceptionClassName) {
        return Counter.builder("soap.client.errors.exceptions")
                .tag("destination", destinationName)
                .tag("operation", operation)
                .tag("exception", exceptionClassName)
                .register(Metrics.globalRegistry);
    }

    private enum SOAPStatus {
        ERROR,
        HTTP_ERROR,
        OK,
        TIMEOUT,
    }

    private static class Meters {
        final Counter errorCounter;
        final ConcurrentMap<String, Counter> errorWithExceptionCounters;
        final Counter callCounter;
        final ImmutableMap<SOAPStatus, Counter> callCounters;
        final Timer timer;

        Meters(String destination, String operation) {
            errorCounter = Counter.builder("soap.client.count.errors")
                    .tag("destination", destination)
                    .tag("operation", operation)
                    .register(Metrics.globalRegistry);
            callCounter = Counter.builder("soap.client.count.calls")
                    .tag("destination", destination)
                    .tag("operation", operation)
                    .register(Metrics.globalRegistry);
            errorWithExceptionCounters = new ConcurrentHashMap<>();

            EnumMap<SOAPStatus, Counter> callCountersBuilder = new EnumMap<>(SOAPStatus.class);

            for (var status: SOAPStatus.values()) {
                String statusName = status.toString();
                Counter soapCallCounter = Counter.builder("soap.client.calls.count")
                        .tag("destination", destination)
                        .tag("operation", operation)
                        .tag("status", statusName)
                        .register(Metrics.globalRegistry);
                callCountersBuilder.put(status, soapCallCounter);
            }

            this.callCounters = ImmutableMap.copyOf(callCountersBuilder);

            this.timer = Timer.builder("soap.client.calls.time")
                    .tag("destination", destination)
                    .tag("operation", operation)
                    .serviceLevelObjectives(MetricsUtils.smallDurationSla())
                    .publishPercentiles(MetricsUtils.higherPercentiles())
                    .register(Metrics.globalRegistry);
        }
    }

}
