package ru.yandex.travel.hotels.common.partners.bnovo;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.CallContextBaseClient;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.DuplicateBookingException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AccommodationType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServiceTaxSystem;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServiceVat;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServiceVatBinding;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Booking;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingJson;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingList;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingStatusId;
import ru.yandex.travel.hotels.common.partners.bnovo.model.CancellationFineType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ConfirmationResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Error;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ErrorResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.GeoData;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Hotel;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelDetailsResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStatusChangedResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStayMap;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntitiesResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntity;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntityResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Offer;
import ru.yandex.travel.hotels.common.partners.bnovo.model.PriceLosRequest;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomPhoto;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Service;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServiceBoardType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServicePriceType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServiceType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Stay;
import ru.yandex.travel.hotels.common.partners.bnovo.model.StayMap;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EHotelOfferOutcome;
import ru.yandex.travel.hotels.proto.TBNovoOffer;

@Slf4j
public class CallContextBNovoClient extends CallContextBaseClient implements BNovoClient {
    private static final String CODE_PREFIX = "TEST_CONTEXT_GENERATED-";
    private final BNovoClient baseClient;
    // transient state, needed to cancelAndGet implementation,
    // which first calls get, then cancel, then get again - all with the same call phase
    private final Set<String> cancellingBookings;
    private final Set<String> confirmingBookings;


    public CallContextBNovoClient(BNovoClient baseClient, CallContext callContext) {
        super(callContext);
        this.baseClient = baseClient;
        confirmingBookings = ConcurrentHashMap.newKeySet();
        cancellingBookings = ConcurrentHashMap.newKeySet();
    }


    @Override
    public CompletableFuture<Map<Long, RatePlan>> getRatePlans(long accountId, String httpRequestId) {
        return wrap(() -> baseClient.getRatePlans(accountId, httpRequestId),
                () -> {
                    switch (getTestContext().getHotelDataLookupOutcome()) {
                        case HO_MOCKED:
                            log.info("Will return a completely mocked rate plans for hotel {}", accountId);
                            return CompletableFuture.completedFuture(Map.of(0L, mockRatePlan(accountId)));
                        case HO_REAL:
                            if (getTestContext().getForceAvailability()) {
                                log.info("Will return real rate plans with an added mock for hotel {}", accountId);
                                return baseClient.getRatePlans(accountId, httpRequestId)
                                        .thenApply(r -> {
                                            r.put(0L, mockRatePlan(accountId));
                                            return r;
                                        });
                            } else {
                                log.info("Will return real rate plans for hotel {}", accountId);
                                return baseClient.getRatePlans(accountId, httpRequestId);
                            }

                        default:
                            throw new IllegalStateException();
                    }
                });
    }

    @Override
    public CompletableFuture<Map<Long, RoomType>> getRoomTypes(long accountId, String httpRequestId) {
        return wrap(() -> baseClient.getRoomTypes(accountId, httpRequestId),
                () -> {
                    switch (getTestContext().getHotelDataLookupOutcome()) {
                        case HO_MOCKED:
                            log.info("Will return a completely mocked room types for hotel {}", accountId);
                            return CompletableFuture.completedFuture(Map.of(0L, mockRoomType(accountId)));
                        case HO_REAL:
                            if (getTestContext().getForceAvailability()) {
                                log.info("Will return real room types with an added mock for hotel {}", accountId);
                                return baseClient.getRoomTypes(accountId, httpRequestId).thenApply(r -> {
                                    r.put(0L, mockRoomType(accountId));
                                    return r;
                                });
                            } else {
                                log.info("Will return real room types with an added mock for hotel {}", accountId);
                                return baseClient.getRoomTypes(accountId, httpRequestId);
                            }
                        default:
                            throw new IllegalStateException();
                    }
                });
    }

    @Override
    public CompletableFuture<HotelStayMap> getPrices(PriceLosRequest request) {
        return wrap(() -> baseClient.getPrices(request),
                () -> {
                    switch (callContext.getPhase()) {
                        case OFFER_SEARCH:
                            if (getTestContext().getForceAvailability()) {
                                log.info("Will return a mocked price availability for request {}", request.toString());
                                HotelStayMap hsm = new HotelStayMap();
                                for (var accountId : request.getAccounts()) {
                                    StayMap sm = new StayMap();
                                    sm.put(request.getCheckinFrom(),
                                            mockStay(request.getCheckinFrom(),
                                                    request.getNights(),
                                                    BigDecimal.valueOf(getTestContext().getPriceAmount())));
                                    hsm.put(String.valueOf(accountId), sm);
                                }
                                return CompletableFuture.completedFuture(hsm);
                            } else {
                                log.info("Will return a real price availability for request {}", request.toString());
                                return baseClient.getPrices(request);
                            }
                        case OFFER_VALIDATION:
                        case ORDER_CREATION:
                            EHotelOfferOutcome outcome;
                            if (callContext.getPhase() == CallContext.CallPhase.OFFER_VALIDATION) {
                                outcome = callContext.getTestContext().getGetOfferOutcome();
                            } else {
                                outcome = callContext.getTestContext().getCreateOrderOutcome();
                            }
                            var stay = getStay();
                            switch (outcome) {
                                case OO_PRICE_MISMATCH:
                                    log.info("Will return a mismatching price stay for request {}", request.toString());
                                    BigDecimal multiplicand =
                                            BigDecimal.valueOf(callContext.getTestContext().getPriceMismatchRate());
                                    stay = stay.toBuilder()
                                            .clearRates()
                                            .rates(stay.getRates().stream().map(offer -> {
                                                        var price = offer.getPrice();
                                                        var newPrice = price.multiply(multiplicand);
                                                        return offer.toBuilder()
                                                                .price(newPrice)
                                                                .clearPricesByDates()
                                                                .pricesByDates(offer.getPricesByDates().entrySet().stream()
                                                                        .collect(Collectors.toMap(
                                                                                Map.Entry::getKey,
                                                                                e -> e.getValue().multiply(multiplicand))))
                                                                .build();
                                                    })
                                                    .collect(Collectors.toList()))
                                            .build();
                                    break;
                                case OO_SOLD_OUT:
                                    log.info("Will return a sold-out stay for request {}", request.toString());
                                    stay = stay.toBuilder()
                                            .clearRates()
                                            .rate(
                                                    stay.getRates().get(0).toBuilder().price(BigDecimal.valueOf(-1)).build())
                                            .build();
                                    break;
                                case OO_SUCCESS:
                                    log.info("Will return the same (matching the search) stay for request {}",
                                            request.toString());
                                    break;
                                default:
                                    throw new IllegalStateException();
                            }
                            HotelStayMap hsm = new HotelStayMap();
                            for (var accountId : request.getAccounts()) {
                                StayMap sm = new StayMap();
                                sm.put(request.getCheckinFrom(), stay);
                                hsm.put(String.valueOf(accountId), sm);
                            }
                            return CompletableFuture.completedFuture(hsm);
                        default:
                            throw new UnsupportedOperationException();
                    }
                });
    }

    @Override
    public CompletableFuture<Map<Long, Hotel>> getAllHotels() {
        return wrap(baseClient::getAllHotels,
                () -> {
                    throw new UnsupportedOperationException();
                });
    }

    @Override
    public CompletableFuture<Hotel> getHotelByAccountId(long accountId, String httpRequestId) {
        return wrap(() -> baseClient.getHotelByAccountId(accountId, httpRequestId),
                () -> {
                    switch (getTestContext().getHotelDataLookupOutcome()) {
                        case HO_REAL:
                            log.info("Will return a real hotel info for  account {}", accountId);
                            return baseClient.getHotelByAccountId(accountId, httpRequestId);
                        case HO_MOCKED:
                            log.info("Will return a mocked hotel info for  account {}", accountId);
                            return CompletableFuture.completedFuture(mockHotel(accountId));
                        default:
                            throw new IllegalStateException();
                    }
                });
    }

    @Override
    public CompletableFuture<Hotel> getHotelByUID(String uid) {
        return wrap(() -> baseClient.getHotelByUID(uid),
                () -> {
                    Preconditions.checkNotNull(callContext.getOfferData());
                    switch (getTestContext().getHotelDataLookupOutcome()) {
                        case HO_MOCKED:
                            if (((TBNovoOffer) callContext.getOfferData()).hasUIDMapping()) {
                                String originalId =
                                        ((TBNovoOffer) callContext.getOfferData()).getUIDMapping().getOriginalId();
                                log.info("Well return a mocked hotel info for uuid {}", uid);
                                var mock = mockHotel(Long.parseLong(originalId)).toBuilder().uid(uid).build();
                                return CompletableFuture.completedFuture(mock);
                            } else {
                                throw new IllegalStateException("Attempt to mock hotel by uid " +
                                        "without uid mapping in context");
                            }
                        case HO_REAL:
                            log.info("Well return a real hotel info for uuid {}", uid);
                            return baseClient.getHotelByUID(uid);
                        default:
                            throw new IllegalStateException();
                    }
                });
    }

    @Override
    public CompletableFuture<BookingList> createBooking(long accountId, BookingJson bookingJson) {
        return wrap(() -> baseClient.createBooking(accountId, bookingJson),
                () -> {
                    switch (getTestContext().getReservationOutcome()) {
                        case RO_PRICE_MISMATCH:
                        case RO_SUCCESS:
                            Booking booking = mockCreatedBooking(accountId, bookingJson);
                            return CompletableFuture.failedFuture(new DuplicateBookingException());
                        case RO_SOLD_OUT:
                            ErrorResponse error = ErrorResponse.builder()
                                    .errors(List.of(
                                            Error.builder()
                                                    .field("booking")
                                                    .type("isAvailable")
                                                    .message("Бронирование недоступно")
                                                    .build()))
                                    .build();
                            throw new CompletionException(new SoldOutException(error));
                        default:
                            throw new UnsupportedOperationException();
                    }

                });
    }

    @Override
    public CompletableFuture<ConfirmationResponse> confirmBooking(String id) {
        return wrap(() -> baseClient.confirmBooking(id),
                () -> {
                    switch (getTestContext().getConfirmationOutcome()) {
                        case CO_SUCCESS:
                        case CO_PRICE_MISMATCH:
                            String code = generateCode(id);
                            confirmingBookings.add(code);
                            return CompletableFuture.completedFuture(ConfirmationResponse.builder()
                                    .confirmedBooking(code)
                                    .build());
                        case CO_NOT_FOUND:
                            return CompletableFuture.completedFuture(ConfirmationResponse.builder()
                                    .alreadyCancelledBooking(generateCode(id))
                                    .build());
                        default:
                            throw new UnsupportedOperationException();
                    }
                });
    }

    @Override
    public CompletableFuture<Booking> getBooking(long accountId, String number) {
        return wrap(() -> baseClient.getBooking(accountId, number),
                () -> {
                    switch (callContext.getPhase()) {
                        case ORDER_CONFIRMATION:
                            switch (getTestContext().getConfirmationOutcome()) {
                                case CO_SUCCESS:
                                case CO_PRICE_MISMATCH:
                                    if (confirmingBookings.contains(number)) {
                                        confirmingBookings.remove(number);
                                        return CompletableFuture.completedFuture(Booking.builder()
                                                .number(number)
                                                .otaId("yandex")
                                                .accountId(accountId)
                                                .otaBookingId(number.substring(CODE_PREFIX.length()))
                                                .statusId(BookingStatusId.CONFIRMED)
                                                .paid(true)
                                                .onlineWarrantyDeadlineDate(null)
                                                .bookingGuaranteeAutoBookingCancel(false)
                                                .build());
                                    } else {
                                        return CompletableFuture.completedFuture(Booking.builder()
                                                .number(number)
                                                .otaId("yandex")
                                                .accountId(accountId)
                                                .otaBookingId(number.substring(CODE_PREFIX.length()))
                                                .statusId(BookingStatusId.CONFIRMED)
                                                .paid(false)
                                                .onlineWarrantyDeadlineDate(callContext.getItinerary().getExpiresAtInstant())
                                                .bookingGuaranteeAutoBookingCancel(true)
                                                .build());
                                    }
                                case CO_NOT_FOUND:
                                    return CompletableFuture.completedFuture(Booking.builder()
                                            .number(number)
                                            .otaId("yandex")
                                            .accountId(accountId)
                                            .otaBookingId(number.substring(CODE_PREFIX.length()))
                                            .statusId(BookingStatusId.CANCELLED)
                                            .paid(false)
                                            .onlineWarrantyDeadlineDate(null)
                                            .bookingGuaranteeAutoBookingCancel(false)
                                            .extraArray(Booking.ExtraArray.builder()
                                                    .cancelReasonClass(1)
                                                    .build())
                                            .build());
                                default:
                                    throw new UnsupportedOperationException();
                            }
                        case ORDER_CANCELLATION:
                        case ORDER_REFUND:
                            if (cancellingBookings.contains(number)) {
                                cancellingBookings.remove(number);
                                return CompletableFuture.completedFuture(Booking.builder()
                                        .number(number)
                                        .otaId("yandex")
                                        .accountId(accountId)
                                        .otaBookingId(number.substring(CODE_PREFIX.length()))
                                        .statusId(BookingStatusId.CANCELLED)
                                        .paid(callContext.getPhase() == CallContext.CallPhase.ORDER_REFUND)
                                        .onlineWarrantyDeadlineDate(null)
                                        .bookingGuaranteeAutoBookingCancel(false)
                                        .build());
                            } else {
                                return CompletableFuture.completedFuture(Booking.builder()
                                        .number(number)
                                        .otaId("yandex")
                                        .accountId(accountId)
                                        .otaBookingId(number.substring(CODE_PREFIX.length()))
                                        .statusId(BookingStatusId.CONFIRMED)
                                        .paid(callContext.getPhase() == CallContext.CallPhase.ORDER_REFUND)
                                        .onlineWarrantyDeadlineDate(callContext.getItinerary().getExpiresAtInstant())
                                        .bookingGuaranteeAutoBookingCancel(true)
                                        .build());
                            }
                        default:
                            throw new UnsupportedOperationException();
                    }
                });
    }

    @Override
    public CompletableFuture<Booking> cancelBooking(long accountId, String number, String email) {
        return wrap(() -> baseClient.cancelBooking(accountId, number, email),
                () -> {
                    cancellingBookings.add(number);
                    return CompletableFuture.completedFuture(Booking.builder()
                            .number(number)
                            .accountId(accountId)
                            .otaId("yandex")
                            .otaBookingId(number.substring(CODE_PREFIX.length()))
                            .statusId(BookingStatusId.CANCELLED)
                            .paid(callContext.getPhase() == CallContext.CallPhase.ORDER_REFUND)
                            .onlineWarrantyDeadlineDate(null)
                            .bookingGuaranteeAutoBookingCancel(false)
                            .build());
                });
    }

    @Override
    public CompletableFuture<HotelStatusChangedResponse> notifyHotelStatusChanged(String hotelCode) {
        return wrap(() -> baseClient.notifyHotelStatusChanged(hotelCode),
                () -> {
                    throw new UnsupportedOperationException();
                });
    }

    @Override
    public CompletableFuture<HotelDetailsResponse> getHotelDetails(String hotelCode) {
        return wrap(() -> baseClient.getHotelDetails(hotelCode),
                () -> {
                    throw new UnsupportedOperationException();
                });
    }

    @Override
    public CompletableFuture<BookingList> getBookings(long accountId) {
        return wrap(() -> baseClient.getBookings(accountId),
                () -> {
                    BNovoHotelItinerary itinerary = (BNovoHotelItinerary) callContext.getItinerary();
                    BookingJson bookingJson = BookingJson.build(itinerary.getYandexNumber(),
                            itinerary.getAccountId(), itinerary.getBNovoStay(),
                            itinerary.getGuests().get(0).getFirstName(),
                            itinerary.getGuests().get(0).getLastName(),
                            itinerary.getCustomerPhone(),
                            itinerary.getCustomerEmail(),
                            itinerary.getOccupancy() == null ? null : itinerary.getOccupancy().getAdults(),
                            itinerary.getOccupancy() == null ? null : itinerary.getOccupancy().getChildren());


                    return CompletableFuture.completedFuture(
                            BookingList.builder()
                                    .bookings(List.of(mockCreatedBooking(accountId, bookingJson)))
                                    .build());
                });
    }

    @Override
    public CompletableFuture<Map<Long, Service>> getServices(long accountId, String httpRequestId) {
        return wrap(() -> baseClient.getServices(accountId, httpRequestId),
                () -> CompletableFuture.completedFuture(mockMealServices(accountId)));
    }

    @Override
    public CompletableFuture<LegalEntitiesResponse> getLegalEntities(long accountId, String httpRequestId) {
        return wrap(() -> baseClient.getLegalEntities(accountId, httpRequestId),
                () -> CompletableFuture.completedFuture(LegalEntitiesResponse.builder()
                        .legalEntities(Collections.singletonList(mockLegalEntity(accountId, 42, false))).build()));
    }


    @Override
    public CompletableFuture<LegalEntityResponse> getLegalEntity(long accountId, long entityId, String httpRequestId) {
        return wrap(() -> baseClient.getLegalEntity(accountId, entityId, httpRequestId),
                () -> CompletableFuture.completedFuture(LegalEntityResponse.builder()
                        .legalEntity(mockLegalEntity(accountId, 42L, true))
                        .build()));
    }

    private LegalEntity mockLegalEntity(long accountId, long entityId, boolean mockServices) {
        var builder = LegalEntity.builder()
                .id(entityId)
                .accountId(accountId)
                .inn("7736207543");  // yandex inn, matches synthetic agreement
        if (mockServices) {
            builder.additionalServicesVat(
                    List.of(
                            AdditionalServiceVatBinding.builder()
                                    .id(1L)
                                    .legalEntityId(entityId)
                                    .additionalServiceId(1L)
                                    .vat(AdditionalServiceVat.VAT20)
                                    .taxSystem(AdditionalServiceTaxSystem.COMMON)
                                    .build(),
                            AdditionalServiceVatBinding.builder()
                                    .id(2L)
                                    .legalEntityId(entityId)
                                    .additionalServiceId(2L)
                                    .vat(AdditionalServiceVat.VAT20)
                                    .taxSystem(AdditionalServiceTaxSystem.COMMON)
                                    .build(),
                            AdditionalServiceVatBinding.builder()
                                    .id(3L)
                                    .legalEntityId(entityId)
                                    .additionalServiceId(3L)
                                    .vat(AdditionalServiceVat.VAT20)
                                    .taxSystem(AdditionalServiceTaxSystem.COMMON)
                                    .build(),
                            AdditionalServiceVatBinding.builder()
                                    .id(4L)
                                    .legalEntityId(entityId)
                                    .additionalServiceId(4L)
                                    .vat(AdditionalServiceVat.VAT20)
                                    .taxSystem(AdditionalServiceTaxSystem.COMMON)
                                    .build(),
                            AdditionalServiceVatBinding.builder()
                                    .id(5L)
                                    .legalEntityId(entityId)
                                    .additionalServiceId(5L)
                                    .vat(AdditionalServiceVat.VAT20)
                                    .taxSystem(AdditionalServiceTaxSystem.COMMON)
                                    .build()

                    )
            );
        }
        return builder.build();
    }

    private Map<Long, Service> mockMealServices(long accountId) {
        return Map.of(
                1L, Service.builder()
                        .id(1L)
                        .accountId(accountId)
                        .type(ServiceType.BOARD)
                        .boardType(ServiceBoardType.BREAKFAST)
                        .name("Завтрак")
                        .price(getTestContext().getMealPrice().getValue())
                        .priceType(ServicePriceType.PERSON)
                        .build(),
                2L, Service.builder()
                        .id(2L)
                        .accountId(accountId)
                        .type(ServiceType.BOARD)
                        .boardType(ServiceBoardType.LUNCH)
                        .name("Обед")
                        .price(getTestContext().getMealPrice().getValue())
                        .priceType(ServicePriceType.PERSON)
                        .build(),
                3L, Service.builder()
                        .id(3L)
                        .accountId(accountId)
                        .type(ServiceType.BOARD)
                        .boardType(ServiceBoardType.DINNER)
                        .name("Ужин")
                        .priceType(ServicePriceType.PERSON)
                        .price(getTestContext().getMealPrice().getValue())
                        .build(),
                4L, Service.builder()
                        .id(4L)
                        .accountId(accountId)
                        .type(ServiceType.BOARD)
                        .boardType(ServiceBoardType.PACKAGE)
                        .isPackage(true)
                        .name("Полупансион")
                        .price(getTestContext().getMealPrice().getValue())
                        .priceType(ServicePriceType.PERSON)
                        .packageAdditionalServicesIds(List.of(1L, 3L))
                        .build(),
                5L, Service.builder()
                        .id(5L)
                        .accountId(accountId)
                        .type(ServiceType.BOARD)
                        .boardType(ServiceBoardType.PACKAGE)
                        .isPackage(true)
                        .price(getTestContext().getMealPrice().getValue())
                        .priceType(ServicePriceType.PERSON)
                        .name("Полный пансион")
                        .packageAdditionalServicesIds(List.of(1L, 2L, 3L))
                        .build());
    }

    private String generateCode(String yandexId) {
        return CODE_PREFIX + yandexId;
    }

    private Booking mockCreatedBooking(long accountId, BookingJson bookingJson) {
        var prices = new Booking.PriceMap();
        bookingJson.getRoomTypes().values().iterator().next().getPrices().forEach(prices::put);
        return Booking.builder()
                .accountId(accountId)
                .otaId("yandex")
                .otaBookingId(bookingJson.getOtaBookingId())
                .statusId(BookingStatusId.CONFIRMED)
                .roomtypeId(Long.parseLong(bookingJson.getRoomTypes().keySet().iterator().next()))
                .planId(bookingJson.getPlanId())
                .parentRoomTypeId(0L)
                .number(generateCode(bookingJson.getOtaBookingId()))
                .arrival(bookingJson.getArrival().atTime(14, 0))
                .departure(bookingJson.getDeparture().atTime(12, 0))
                .name(bookingJson.getName())
                .surname(bookingJson.getSurname())
                .email(bookingJson.getEmail())
                .phone(bookingJson.getPhone())
                .adults(getOccupancy().getAdults())
                .children(getOccupancy().getChildren().size())
                .amount(getStay().getRates().get(0).getPrice())
                .prices(prices)
                .onlineWarrantyDeadlineDate(Instant.now().plusSeconds(900))
                .bookingGuaranteeAutoBookingCancel(true)
                .build();

    }

    private long getAccountId() {
        switch (callContext.getPhase()) {
            case OFFER_SEARCH:
                return Long.parseLong(callContext.getSearchOffersReq().getHotelId().getOriginalId());
            case OFFER_VALIDATION:
            case ORDER_CREATION:
                Preconditions.checkNotNull(callContext.getTravelToken());
                return Long.parseLong(callContext.getTravelToken().getOriginalId());
            case ORDER_RESERVATION:
            case ORDER_REFUND:
            case ORDER_CANCELLATION:
            case ORDER_CONFIRMATION:
                Preconditions.checkNotNull(callContext.getItinerary());
                return Long.parseLong(callContext.getItinerary().getOrderDetails().getOriginalId());
            default:
                throw new IllegalStateException();
        }
    }

    private Occupancy getOccupancy() {
        switch (callContext.getPhase()) {
            case OFFER_SEARCH:
                return Occupancy.fromString(callContext.getSearchOffersReq().getOccupancy());
            case OFFER_VALIDATION:
            case ORDER_CREATION:
                Preconditions.checkNotNull(callContext.getTravelToken());
                return callContext.getTravelToken().getOccupancy();
            case ORDER_RESERVATION:
            case ORDER_REFUND:
            case ORDER_CANCELLATION:
            case ORDER_CONFIRMATION:
                Preconditions.checkNotNull(callContext.getItinerary());
                int numAdults = (int) callContext.getItinerary().getGuests().stream().filter(g -> !g.isChild()).count();
                var childAges = callContext.getItinerary().getGuests().stream()
                        .filter(Guest::isChild)
                        .map(guest -> String.valueOf(guest.getAge()))
                        .collect(Collectors.joining(","));
                if (childAges.length() == 0) {
                    return Occupancy.fromString(String.valueOf(numAdults));
                } else {
                    return Occupancy.fromString(String.format("%s-%s", numAdults, childAges));
                }
            default:
                throw new IllegalStateException();
        }
    }


    private Stay getStay() {
        switch (callContext.getPhase()) {
            case ORDER_CREATION:
            case OFFER_VALIDATION:
                Preconditions.checkNotNull(callContext);
                Preconditions.checkNotNull(callContext.getOfferData());
                return ProtoUtils.fromTJson(((TBNovoOffer) callContext.getOfferData()).getBNovoStay(), Stay.class);
            case ORDER_RESERVATION:
            case ORDER_CONFIRMATION:
            case ORDER_CANCELLATION:
            case ORDER_REFUND:
                Preconditions.checkNotNull(callContext);
                Preconditions.checkNotNull(callContext.getItinerary());
                return ((BNovoHotelItinerary) callContext.getItinerary()).getBNovoStay();
            default:
                throw new UnsupportedOperationException();
        }

    }

    private Hotel mockHotel(long accountId) {
        return Hotel.builder()
                .id(accountId)
                .name("Тестовый сгенерированный отель")
                .phone("8 800 511-71-04")
                .email("orders@travel.yandex.ru")
                .address("Москва, ул Льва Толстого, 16")
                .enabled(true)
                .checkin("14:00")
                .checkout("12:00")
                .geoData(GeoData.builder()
                        .latitude(55.734399)
                        .longitude(37.587293)
                        .build())
                .timezone(TimeZone.getTimeZone("Europe/Moscow"))
                .uid(UUID.randomUUID().toString())
                .build();
    }


    private RatePlan mockRatePlan(long accountId) {
        Preconditions.checkNotNull(getTestContext());
        var builder = RatePlan.builder()
                .id(0)
                .accountId(accountId)
                .name("Сгенерированный тарифный план")
                .description("Тарифный план, доступный только для тестировщиков Яндекса")
                .isDefault(false)
                .enabled(true)
                .nutrition(new Boolean[]{false, false, false})
                .boardFromServices(true)
                .enabledOTA(true)
                .enabledYandex(true)
                .forPromoCode(false)
                .cancellationRules(callContext.getTestContext().getCancellation().toString());
        switch (callContext.getTestContext().getPansionType()) {
            case PT_BB:
                builder.additionalServicesIds(List.of(1L));
                break;
            case PT_HB:
                builder.additionalServicesIds(List.of(4L));
                break;
            case PT_BD:
                builder.additionalServicesIds(List.of(3L));
                break;
            case PT_FB:
            case PT_AI:
            case PT_UAI:
            case PT_LAI:
                builder.additionalServicesIds(List.of(5L));
                break;
            case PT_RO:
                builder.additionalServicesIds(Collections.emptyList());
                break;
            default:
                throw new IllegalStateException();
        }
        switch (callContext.getTestContext().getCancellation()) {
            case CR_FULLY_REFUNDABLE:
                builder.cancellationDeadline(0);
                break;
            case CR_NON_REFUNDABLE:
                builder.cancellationDeadline(100000)
                        .cancellationFineType(CancellationFineType.FULL_AMOUNT);
                break;
            case CR_PARTIALLY_REFUNDABLE:
                builder.cancellationDeadline(100000)
                        .cancellationFineType(CancellationFineType.PERCENTAGE)
                        .cancellationFineAmount(BigDecimal.valueOf(callContext.getTestContext().getPartiallyRefundableRate()));
                break;
            case CR_CUSTOM:
                Instant willBePartialRefundableAt =
                        Instant.now().plus(callContext.getTestContext().getPartiallyRefundableInMinutes(),
                                ChronoUnit.MINUTES);
                Instant checkinAt =
                        LocalDate.parse(callContext.getSearchOffersReq().getCheckInDate()).atStartOfDay().toInstant(ZoneOffset.UTC);
                int days = (int) ChronoUnit.DAYS.between(willBePartialRefundableAt, checkinAt);
                builder.cancellationDeadline(days)
                        .cancellationFineType(CancellationFineType.PERCENTAGE)
                        .cancellationFineAmount(BigDecimal.valueOf(callContext.getTestContext().getPartiallyRefundableRate()));
                break;
            default:
                throw new IllegalStateException();
        }
        return builder.build();
    }

    private RoomType mockRoomType(long accountId) {
        Preconditions.checkNotNull(getTestContext());
        return RoomType.builder()
                .id(0L)
                .accountId(accountId)
                .parentId(0L)
                .name(getTestContext().getOfferName())
                .description("Ненастоящий номер в ненастоящем отеле, цена на размещение в котором " +
                        "взимается в ненастоящих деньгах. Ну и вообще вам всё это кажется.")
                .adults(getOccupancy().getAdults())
                .children(getOccupancy().getChildren().size())
                .accommodationType(AccommodationType.HOTEL_ROOM)
                .photos(Collections.singletonList(
                        RoomPhoto.builder()
                                .id(0L)
                                .url("https://avatars.mds.yandex.net/get-altay/200322/" +
                                        "2a0000015b1717fa178796a0b1d5ec6b0689/XXXL")
                                .thumb("https://avatars.mds.yandex.net/get-altay/200322/" +
                                        "2a0000015b1717fa178796a0b1d5ec6b0689/M")
                                .build()
                ))
                .build();
    }

    private Stay mockStay(LocalDate checkin, int nights, BigDecimal expectedPrice) {
        Occupancy occupancy = getOccupancy();
        var offerBuilder = Offer.builder()
                .price(expectedPrice)
                .adults(occupancy.getAdults())
                .children(occupancy.getChildren().size())
                .planId(0)
                .roomtypeId(0);
        BigDecimal pricePerNight = expectedPrice.divide(BigDecimal.valueOf(nights), RoundingMode.HALF_UP);
        for (int i = 0; i < nights; i++) {
            offerBuilder.pricesByDate(checkin.plusDays(i), pricePerNight);
        }
        return Stay.builder()
                .currency("RUB")
                .checkin(checkin)
                .nights(nights)
                .rate(offerBuilder.build())
                .timestamp(Instant.now().atZone(ZoneId.systemDefault()))
                .build();
    }
}
