package ru.yandex.travel.hotels.common.partners.bnovo;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.slf4j.MDC;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.logging.masking.LogAwareRequestBuilder;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnexpectedHttpStatusCodeException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.AlreadyCancelledException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.BNovoErrorException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.CancelAfterDepartureException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.DuplicateBookingException;
import ru.yandex.travel.hotels.common.partners.bnovo.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AdditionalServicesResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AuthRequest;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AuthResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Booking;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingJson;
import ru.yandex.travel.hotels.common.partners.bnovo.model.BookingList;
import ru.yandex.travel.hotels.common.partners.bnovo.model.CancellationResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ConfirmationResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ErrorResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Hotel;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelDetailsResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelInfoResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelList;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStatusChangedResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStayMap;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntitiesResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.LegalEntityResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.model.PriceLosRequest;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlanList;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomTypeList;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Service;
import ru.yandex.travel.hotels.common.partners.bnovo.model.SingleBookingResponse;
import ru.yandex.travel.hotels.common.partners.bnovo.utils.AuthenticationRetryHelper;

import static java.util.function.Function.identity;

public class DefaultBNovoClient extends BaseClient<BNovoClientProperties> implements BNovoClient {

    private static final String BNOVO_DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private static final String PLANS = "Plans";
    private static final String ROOM_TYPES = "RoomTypes";
    private static final String PRICES_LOS = "PricesLos";
    private static final String ACCOUNTS = "Accounts";
    private static final String HOTEL_INFO = "HotelInfo";
    private static final String HOTEL_INFO_PUBLIC = "HotelInfoPublic";
    private static final String AUTHENTICATE = "Authenticate";
    private static final String CREATE_BOOKING = "CreateBooking";
    private static final String CONFIRM_PAYMENT = "ConfirmPayment";
    private static final String GET_BOOKING = "GetBooking";
    private static final String CANCEL_BOOKING = "CancelBooking";
    private static final String HOTEL_STATUS_CHANGED = "HotelStatusChanged";
    private static final String HOTEL_DETAILS = "HotelDetails";
    private static final String GET_BOOKINGS = "GetBookings";
    private static final String GET_SERVICES = "GetServices";
    private static final String GET_LEGAL_ENTITIES = "GetLegalEntities";
    private static final String GET_LEGAL_ENTITY = "GetLegalEntity";

    private static final String HOTEL_CODE_PARAM = "account_id";
    private static final String TOKEN_PARAM = "token";

    private static final ClientMethods METHODS;
    private static final Set<String> PRIVATE_METHODS;
    private static final Set<String> PRICES_LOS_METHODS;
    private static final ObjectMapper MAPPER;

    static {
        METHODS = new ClientMethods()
                .register(PLANS, "/plans", HttpMethod.GET, RatePlanList.class)
                .register(ROOM_TYPES, "/roomtypes", HttpMethod.GET, RoomTypeList.class)
                .register(PRICES_LOS, "/prices_los", HttpMethod.POST, HotelStayMap.class)
                .register(ACCOUNTS, "/accounts", HttpMethod.GET, HotelList.class)
                .register(HOTEL_INFO, "/accounts/{0}", HttpMethod.GET, HotelInfoResponse.class)
                .register(HOTEL_INFO_PUBLIC, "/accounts", HttpMethod.GET, HotelInfoResponse.class)
                .register(AUTHENTICATE, "/auth", HttpMethod.POST, AuthResponse.class)
                .register(CREATE_BOOKING, "/bookings", HttpMethod.POST, BookingList.class)
                .register(CONFIRM_PAYMENT, "/confirm_payment", HttpMethod.POST, ConfirmationResponse.class)
                .register(GET_BOOKING, "/bookings", HttpMethod.GET, SingleBookingResponse.class)
                .register(CANCEL_BOOKING, "/bookings", HttpMethod.DELETE, CancellationResponse.class)
                .register(HOTEL_STATUS_CHANGED, "/yandex_hotel_status_changed", HttpMethod.POST,
                        HotelStatusChangedResponse.class)
                .register(HOTEL_DETAILS, "/yandex_hotel_details", HttpMethod.GET, HotelDetailsResponse.class)
                .register(GET_BOOKINGS, "/bookings", HttpMethod.GET, BookingList.class)
                .register(GET_SERVICES, "/additional_services", HttpMethod.GET, AdditionalServicesResponse.class)
                .register(GET_LEGAL_ENTITIES, "/legal_entities", HttpMethod.GET, LegalEntitiesResponse.class)
                .register(GET_LEGAL_ENTITY, "/legal_entities/{0}", HttpMethod.GET, LegalEntityResponse.class);

        PRIVATE_METHODS = Set.of(ACCOUNTS, HOTEL_INFO, AUTHENTICATE, CREATE_BOOKING, CONFIRM_PAYMENT, GET_BOOKING,
                CANCEL_BOOKING, GET_BOOKINGS, GET_LEGAL_ENTITIES, GET_LEGAL_ENTITY);
        PRICES_LOS_METHODS = Set.of(PRICES_LOS, ROOM_TYPES, PLANS, GET_SERVICES);
        MAPPER = createObjectMapper();
    }

    private final AtomicReference<Token> token = new AtomicReference<>(null);
    private AuthenticationRetryHelper authRetryHelper;

    public DefaultBNovoClient(AsyncHttpClientWrapper clientWrapper, BNovoClientProperties properties,
                              Retry retryHelper) {
        super(properties, clientWrapper, createObjectMapper(), retryHelper);
        authRetryHelper = new AuthenticationRetryHelper(properties.getAuthenticationMaxRetryCount());
    }

    public static ClientMethods getMethods() {
        return DefaultBNovoClient.METHODS;
    }

    public static ObjectMapper getMapper() {
        return DefaultBNovoClient.MAPPER;
    }

    public static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(
                LocalDate.class,
                new LocalDateDeserializer(DateTimeFormatter.ISO_DATE));
        javaTimeModule.addSerializer(
                LocalDate.class,
                new LocalDateSerializer(DateTimeFormatter.ISO_DATE));

        javaTimeModule.addDeserializer(
                LocalDateTime.class,
                new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern(BNOVO_DATE_TIME_PATTERN)));
        javaTimeModule.addSerializer(
                LocalDateTime.class,
                new LocalDateTimeSerializer(DateTimeFormatter.ofPattern(BNOVO_DATE_TIME_PATTERN)));
        mapper.registerModule(javaTimeModule);
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    @Override
    public CompletableFuture<Map<Long, RatePlan>> getRatePlans(long accountId, String httpRequestId) {
        CompletableFuture<RatePlanList> ratePlanListFuture = call(PLANS, new CallArguments()
                .withQueryParam("account_id", String.valueOf(accountId))
                .withRequestId(httpRequestId)
        );
        return ratePlanListFuture.thenApply(rpl -> rpl.getPlans().stream().collect(
                Collectors.toMap(RatePlan::getId, identity())));
    }

    @Override
    public CompletableFuture<Map<Long, RoomType>> getRoomTypes(long accountId, String httpRequestId) {
        CompletableFuture<RoomTypeList> roomTypeListFuture = call(ROOM_TYPES, new CallArguments()
                .withQueryParam("account_id", String.valueOf(accountId))
                .withRequestId(httpRequestId)
        );
        return roomTypeListFuture.thenApply(rtl -> rtl.getRooms().stream().collect(
                Collectors.toMap(RoomType::getId, identity())));
    }

    @Override
    public CompletableFuture<HotelStayMap> getPrices(PriceLosRequest request) {
        return call(PRICES_LOS, new CallArguments()
                .withRequestId(request.getRequestId())
                .withBody(request));
    }

    @Override
    public CompletableFuture<Map<Long, Hotel>> getAllHotels() {
        CompletableFuture<HotelList> future = withAuthentication(t ->
                call(ACCOUNTS, new CallArguments()
                        .withQueryParam(TOKEN_PARAM, t.getToken())));
        return future.thenApply(hl -> hl.getAccounts().stream().collect(Collectors.toMap(Hotel::getId, hi -> hi)));
    }

    @Override
    public CompletableFuture<Hotel> getHotelByAccountId(long accountId, String httpRequestId) {
        CompletableFuture<HotelInfoResponse> resp = withAuthentication(t ->
                call(HOTEL_INFO, new CallArguments()
                        .withUrlParams(String.valueOf(accountId))
                        .withQueryParam(TOKEN_PARAM, t.getToken())
                        .withRequestId(httpRequestId)
                ), httpRequestId);
        return resp.thenApply(HotelInfoResponse::getAccount);
    }

    @Override
    public CompletableFuture<Hotel> getHotelByUID(String uid) {
        CompletableFuture<HotelInfoResponse> resp = call(HOTEL_INFO_PUBLIC, new CallArguments()
                .withQueryParam("uid", uid));
        return resp.thenApply(HotelInfoResponse::getAccount);
    }

    @Override
    public CompletableFuture<BookingList> createBooking(long accountId, BookingJson bookingJson) {
        CompletableFuture<BookingList> resp = withAuthentication(t ->
                call(CREATE_BOOKING, new CallArguments()
                        .withFormParam(TOKEN_PARAM, t.getToken())
                        .withFormParam("account_id", String.valueOf(accountId))
                        .withFormParam(
                                LogAwareRequestBuilder.FormParam.forObject("booking_json", bookingJson, MAPPER))
                ));
        return FutureUtils.handleExceptionOfType(resp, BNovoErrorException.class, t -> {
            if (t.getErrorResponse().getErrors().stream().anyMatch(e -> e.getType().equals("isAvailable"))) {
                throw new CompletionException(new SoldOutException(t.getErrorResponse()));
            } else {
                throw new CompletionException(t);
            }
        });
    }

    @Override
    public CompletableFuture<ConfirmationResponse> confirmBooking(String id) {
        return withAuthentication(t ->
                call(CONFIRM_PAYMENT, new CallArguments()
                        .withFormParam(TOKEN_PARAM, t.getToken())
                        .withFormParam("ota_booking_id", id)));
    }

    @Override
    public CompletableFuture<Booking> getBooking(long accountId, String number) {
        CompletableFuture<SingleBookingResponse> resp = withAuthentication(t ->
                call(GET_BOOKING, new CallArguments()
                        .withQueryParam(TOKEN_PARAM, t.getToken())
                        .withQueryParam("account_id", String.valueOf(accountId))
                        .withQueryParam("booking_number", number)));
        return resp.thenApply(SingleBookingResponse::getBooking);
    }

    @Override
    public CompletableFuture<Booking> cancelBooking(long accountId, String number, String email) {
        CompletableFuture<CancellationResponse> resp = withAuthentication(t ->
                call(CANCEL_BOOKING, new CallArguments()
                        .withQueryParam(TOKEN_PARAM, t.getToken())
                        .withQueryParam("account_id", String.valueOf(accountId))
                        .withQueryParam("booking_number", number)
                        .withQueryParam("email", email)));
        return FutureUtils.handleExceptionOfType(resp, BNovoErrorException.class, t -> {
            if (t.getErrorResponse().getErrors().stream().anyMatch(e -> e.getType().equals("AlreadyCanceled"))) {
                throw new CompletionException(new AlreadyCancelledException(t.getErrorResponse()));
            }
            if (t.getErrorResponse().getErrors().stream().anyMatch(e -> e.getType().equals("CancelAfterDeparture"))) {
                throw new CompletionException(new CancelAfterDepartureException(t.getErrorResponse()));
            } else {
                throw new CompletionException(t);
            }
        }).thenApply(cr -> {
            Preconditions.checkState(cr.getDeletedBookings().size() == 1,
                    "Unexpected number of deleted bookings");
            return cr.getDeletedBookings().get(0);
        });
    }

    @Override
    public CompletableFuture<HotelStatusChangedResponse> notifyHotelStatusChanged(String hotelCode) {
        return withAuthentication(t -> call(HOTEL_STATUS_CHANGED, new CallArguments()
                .withQueryParam(TOKEN_PARAM, t.getToken())
                .withQueryParam(HOTEL_CODE_PARAM, hotelCode)));
    }

    @Override
    public CompletableFuture<HotelDetailsResponse> getHotelDetails(String hotelCode) {
        return withAuthentication(t -> call(HOTEL_DETAILS, new CallArguments()
                .withQueryParam(TOKEN_PARAM, t.getToken())
                .withQueryParam(HOTEL_CODE_PARAM, hotelCode)));
    }

    @Override
    public CompletableFuture<BookingList> getBookings(long accountId) {
        return withAuthentication(t -> call(GET_BOOKINGS, new CallArguments()
                .withQueryParam(TOKEN_PARAM, t.getToken())
                .withQueryParam("account_id", String.valueOf(accountId))));
    }

    @Override
    public CompletableFuture<Map<Long, Service>> getServices(long accountId, String httpRequestId) {
        CompletableFuture<AdditionalServicesResponse> servicesFuture = call(GET_SERVICES, new CallArguments()
                .withQueryParam("account_id", String.valueOf(accountId))
                .withRequestId(httpRequestId)
        );
        return servicesFuture.thenApply(asr -> asr.getAdditionalServices().stream().collect(
                Collectors.toMap(Service::getId, identity())));
    }

    @Override
    public CompletableFuture<LegalEntitiesResponse> getLegalEntities(long accountId, String httpRequestId) {
        return withAuthentication(t ->
                call(GET_LEGAL_ENTITIES, new CallArguments()
                        .withQueryParam(TOKEN_PARAM, t.getToken())
                        .withQueryParam(HOTEL_CODE_PARAM, String.valueOf(accountId))
                ), httpRequestId);
    }

    @Override
    public CompletableFuture<LegalEntityResponse> getLegalEntity(long accountId, long entityId, String httpRequestId) {
        return withAuthentication(t ->
                call(GET_LEGAL_ENTITY, new CallArguments()
                        .withUrlParams(String.valueOf(entityId))
                        .withQueryParam(TOKEN_PARAM, t.getToken())
                        .withQueryParam(HOTEL_CODE_PARAM, String.valueOf(accountId))
                ), httpRequestId);
    }

    @Override
    protected <R, B> CompletableFuture<R> call(String destinationMethod, CallArguments params) {
        CompletableFuture<R> res = super.call(destinationMethod, params);
        return FutureUtils.handleExceptionOfType(res, UnexpectedHttpStatusCodeException.class, t -> {
            if (t.getStatusCode() == 406) {
                try {
                    ErrorResponse errorResponse = MAPPER.readerFor(ErrorResponse.class).readValue(t.getResponseBody());
                    throw new CompletionException(new BNovoErrorException(errorResponse));
                } catch (IOException e) {
                    throw new CompletionException(e);
                }
            }
            if (t.getStatusCode() == 429 && CREATE_BOOKING.equals(destinationMethod)) {
                throw new CompletionException(new DuplicateBookingException());
            } else {
                throw t;
            }
        });
    }

    @Override
    protected ClientMethods getClientMethods() {
        return METHODS;
    }

    @Override
    protected String getBaseUrl(String destinationMethod) {
        if (PRIVATE_METHODS.contains(destinationMethod)) {
            return properties.getPrivateApiBaseUrl();
        } else if (PRICES_LOS_METHODS.contains(destinationMethod)) {
            return properties.getPricesLosApiBaseUrl();
        } else {
            return super.getBaseUrl(destinationMethod);
        }
    }

    protected CompletableFuture<Token> authenticate(String httpRequestId) {
        CompletableFuture<AuthResponse> future = call(AUTHENTICATE, new CallArguments()
                .withBody(AuthRequest.builder()
                        .username(properties.getUsername())
                        .password(properties.getPassword())
                        .build())
                .withRequestId(httpRequestId)
        );
        return future.thenApply(r -> {
            Token newToken = new Token(r.getToken(), Instant.now());
            while (true) {
                Token oldToken = token.get();
                if (oldToken != null && oldToken.getCreatedAt().isAfter(newToken.getCreatedAt())) {
                    return oldToken;
                }
                if (token.compareAndSet(oldToken, newToken)) {
                    return newToken;
                }
            }
        });
    }

    protected <K> CompletableFuture<K> withAuthentication(Function<Token, CompletableFuture<K>> function) {
        return withAuthentication(function, null);
    }

    protected <K> CompletableFuture<K> withAuthentication(Function<Token, CompletableFuture<K>> function,
                                                          String httpRequestId) {
        return authRetryHelper.withRetry(authenticateRequest(function, httpRequestId));
    }

    protected <K> CompletableFuture<K> authenticateRequest(Function<Token, CompletableFuture<K>> function,
                                                           String httpRequestId) {
        try {
            Token token = this.token.get();

            if (token != null && token.getCreatedAt().plus(properties.getTokenValidityDuration()).isAfter(Instant.now())) {
                return function.apply(token);
            } else {
                var currentMdc = MDC.getCopyOfContextMap();
                return authenticate(httpRequestId).thenCompose(newToken -> {
                    try (var ignored = NestedMdc.nestedMdc(currentMdc)) {
                        return function.apply(newToken);
                    }
                });
            }
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Getter
    @AllArgsConstructor
    protected static class Token {
        private final String token;
        private final Instant createdAt;
    }
}
