package ru.yandex.travel.hotels.common.partners.bnovo.model;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import com.google.common.base.Preconditions;
import lombok.Builder;
import lombok.Singular;
import lombok.Value;

import ru.yandex.travel.commons.logging.masking.PersonalData;

@Value
@Builder
public class BookingJson {
    private long accountId;
    private long planId;
    private String otaBookingId;
    private LocalDate arrival;
    private LocalDate departure;
    @PersonalData
    private String name;
    @PersonalData
    private String surname;
    @PersonalData
    private String email;
    @PersonalData
    private String phone;
    private String comment;
    private String lang;
    @Singular
    private Map<String, RoomType> roomTypes;
    @JsonSerialize(using = Serialization.JsonAsStringSerializer.class)
    private BookingExtra extra;


    public static BookingJson build(String id, long accountId, Stay stay, String firstName, String lastName,
                                    String phone, String email, Integer numAdults, List<Integer> childAges) {
        Preconditions.checkArgument(stay.getRates().size() == 1, "Unexpected amount of offers");
        Offer offer = stay.getRates().get(0);
        Preconditions.checkArgument(offer.getPricesByDates().size() == stay.getNights(),
                "Amount of nights does not match with price breakdown");

        BookingExtra extra = null;
        if (numAdults != null) {
            List<RequestedOccupancy> requestedOccupancies = new ArrayList<>();
            requestedOccupancies.add(RequestedOccupancy.builder().type("adult").count(numAdults).build());
            if (childAges == null) {
                childAges = Collections.emptyList();
            }
            var ageMap = childAges.stream().collect(Collectors.groupingBy(Function.identity(), Collectors.counting()));
            for (var age : ageMap.entrySet()) {
                requestedOccupancies.add(RequestedOccupancy.builder()
                        .type("child")
                        .age(age.getKey())
                        .count(age.getValue().intValue())
                        .build());
            }
            extra = BookingExtra.builder()
                    .guests(GuestInfo.builder()
                            .requestedOccupancies(requestedOccupancies)
                            .build())
                    .build();
        }

        return BookingJson.builder()
                .accountId(accountId)
                .planId(offer.getPlanId())
                .otaBookingId(id)
                .arrival(stay.getCheckin())
                .departure(stay.getCheckin().plusDays(stay.getNights()))
                .name(firstName)
                .surname(lastName)
                .email(email)
                .phone(phone)
                .lang("ru")
                .roomType(
                        String.valueOf(offer.getRoomtypeId()),
                        RoomType.builder()
                                .count(1)
                                .prices(offer.getPricesByDates())
                                .roomTypeService(ServiceList.builder().build())
                                .build())
                .extra(extra)
                .build();
    }

    @Value
    @Builder
    public static class RoomType {
        private int count;
        @Singular
        private Map<LocalDate, BigDecimal> prices;
        @Singular
        private List<ServiceList> roomTypeServices;
    }

    @Value
    @Builder
    public static class ServiceList {
        @Singular
        private List<Service> services;
    }

    @Value
    @Builder
    public static class Service {
        private long i;
        private long c;
        private BigDecimal totalAmount;
        private Map<LocalDate, CountPrice> dayPrices;
    }

    @Value
    @Builder
    public static class CountPrice {
        private int count;
        private BigDecimal price;
    }

    @Value
    @Builder
    private static class BookingExtra {
        @JsonProperty("Guests")
        private GuestInfo guests;
    }

    @Value
    @Builder
    public static class GuestInfo {
        @JsonProperty(value = "Requested occupancy")
        private List<RequestedOccupancy> requestedOccupancies;
    }

    @Value
    @Builder
    public static class RequestedOccupancy {
        private String type;
        private int count;
        private Integer age;
    }
}
