package ru.yandex.travel.hotels.common.partners.bnovo.model;

import java.io.IOException;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.deser.ContextualDeserializer;
import com.fasterxml.jackson.databind.exc.InvalidFormatException;

import ru.yandex.travel.hotels.common.partners.bnovo.DefaultBNovoClient;

public class Serialization {
    public static class IntegerToBooleanDeserializer extends JsonDeserializer<Boolean> {
        @Override
        public Boolean deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            switch (p.getText()) {
                case "0":
                    return false;
                case "1":
                    return true;
                default:
                    throw new InvalidFormatException(p, "Unable to deserialize as boolean flag", p.getText(),
                            Boolean.class);
            }
        }
    }

    public static class BitMaskDeserializer extends JsonDeserializer<Boolean[]> {

        @Override
        public Boolean[] deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            var value = p.getText();
            Boolean[] res = new Boolean[value.length()];
            for (int i = 0; i < value.length(); i++) {
                if (value.charAt(i) != '0' && value.charAt(i) != '1') {
                    throw new InvalidFormatException(p, "Unable to deserialize as boolean mask", p.getText(),
                            Array.class);
                }
                res[i] = (value.charAt(i) == '1');
            }
            return res;
        }
    }

    public static class StringToPriceMapDeserializer extends JsonDeserializer<Booking.PriceMap> {
        @Override
        public Booking.PriceMap deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            var value = p.getText();
            return DefaultBNovoClient.getMapper().readerFor(Booking.PriceMap.class).readValue(value);
        }
    }

    public static class GeoDataDeserializer extends JsonDeserializer<GeoData> {

        @Override
        public GeoData deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            var value = p.getText();
            if (value.charAt(0) != '(' || value.charAt(value.length() - 1) != ')') {
                throw new InvalidFormatException(p, "Unable to deserialize as geo coordinates", value,
                        GeoData.class);
            }
            value = value.substring(1, value.length() - 1);
            String[] parts = value.split(",");
            if (parts.length != 2) {
                throw new InvalidFormatException(p, "Unable to deserialize as geo coordinates", value,
                        GeoData.class);
            }
            return GeoData.builder()
                    .latitude(Double.parseDouble(parts[0]))
                    .longitude(Double.parseDouble(parts[1]))
                    .build();
        }
    }

    public static class MapFromListOrMapDeserializer extends JsonDeserializer<Map<?, ?>> implements ContextualDeserializer {

        @Override
        public JsonDeserializer<Map<?, ?>> createContextual(DeserializationContext outerContext,
                                                            BeanProperty property) throws JsonMappingException {
            JavaType contextualType = outerContext.getContextualType();
            return new JsonDeserializer<Map<?, ?>>() {
                @Override
                public Map<?, ?> deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                        JsonProcessingException {
                    if (JsonToken.START_ARRAY.equals(parser.getCurrentToken())) {
                        var listType = context.getTypeFactory().constructCollectionLikeType(ArrayList.class,
                                Object.class);
                        var deserialized =
                                (ArrayList<?>) context.findRootValueDeserializer(listType).deserialize(parser, context);
                        if (deserialized.size() == 0) {
                            return Collections.emptyMap();
                        } else {
                            throw new InvalidFormatException(parser, "Unable to deserialize non-empty list as map",
                                    deserialized, Map.class);
                        }
                    } else {
                        var mapType = context.getTypeFactory().constructMapType(HashMap.class,
                                context.getTypeFactory().constructType(Object.class), contextualType.getContentType());
                        var map = (Map<?, ?>) context.findRootValueDeserializer(mapType).deserialize(parser, context);
                        return map;
                    }
                }
            };
        }

        @Override
        public Map<?, ?> deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            return null;
        }
    }

    public static class JsonAsStringSerializer<T> extends JsonSerializer<T> {
        private static final ObjectMapper MAPPER = DefaultBNovoClient.createObjectMapper();

        @Override
        public void serialize(T value, JsonGenerator gen, SerializerProvider serializers) throws IOException {
            String s = MAPPER.writeValueAsString(value);
            gen.writeString(s);
        }
    }
}
