package ru.yandex.travel.hotels.common.partners.bnovo.utils

import org.slf4j.LoggerFactory
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnexpectedHttpStatusCodeException
import java.util.concurrent.CompletableFuture

class AuthenticationRetryHelper(
    private val maxAllowedRetry: Int
) {
    private val log = LoggerFactory.getLogger(this.javaClass)

    fun <T> withRetry(authSupplier: CompletableFuture<T>): CompletableFuture<T> {
        return withRetry("Authentication for request", 0, authSupplier)
    }

    private fun <T> withRetry(
        name: String,
        iteration: Int,
        authSupplier: CompletableFuture<T>
    ): CompletableFuture<T> {
        return if (iteration < maxAllowedRetry) {
            FutureUtils.handleExceptionOfType(authSupplier, UnexpectedHttpStatusCodeException::class.java) {
                if (it.statusCode == 401) {
                    withRetry(name, iteration + 1, authSupplier)
                }

                throw it
            }
        } else {
            log.warn("$name: Iteration $iteration needs a retry, but no more attempts left")

            authSupplier
        }
    }
}
