package ru.yandex.travel.hotels.common.partners.bnovo.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.Comparator;

import com.google.common.base.Preconditions;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.partners.bnovo.model.CancellationFineType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Offer;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

public class BNovoRefundRulesBuilder {
    public static RefundRules build(RatePlan plan, Offer offer, Instant checkinDate,
                                    Instant searchDate, String currencyCode) {
        if (checkinDate.isBefore(searchDate)) {
            return RefundRules.builder()
                    .rule(RefundRule.builder()
                            .type(RefundType.NON_REFUNDABLE)
                            .build())
                    .build();
        }
        Preconditions.checkArgument(plan.getCancellationDeadline() != null && plan.getCancellationDeadline() >= 0,
                String.format("Hotel %d, plan %d: cancellation_deadline must be non-null and non-negative",
                        plan.getAccountId(), plan.getId()));
        if (plan.getCancellationFineType() == CancellationFineType.NO_PENALTY) {
            Preconditions.checkArgument(plan.getCancellationDeadline() == 0,
                    String.format("Hotel %d, plan %d: cancellation_deadline should be 0 " +
                            "when cancellation_fine_type is 0", plan.getAccountId(), plan.getId()));
        }
        var builder = RefundRules.builder();
        Instant penaltyStartDate = checkinDate.minus(plan.getCancellationDeadline(), ChronoUnit.DAYS);
        Instant nonRefStartDate = checkinDate;
        if (plan.getCancellationFineType() == CancellationFineType.FULL_AMOUNT ||
                (plan.getCancellationFineType() == CancellationFineType.PERCENTAGE && plan.getCancellationFineAmount().compareTo(BigDecimal.valueOf(100)) >= 0) ||
                (plan.getCancellationFineType() == CancellationFineType.FIRST_NIGHT && offer.getPricesByDates().size() == 1) ||
                (plan.getCancellationFineType() == CancellationFineType.FIXED_AMOUNT && plan.getCancellationFineAmount().compareTo(offer.getPrice()) >= 0)) {
            nonRefStartDate = penaltyStartDate;
        }
        boolean hasInterval = false;

        if (penaltyStartDate.isAfter(searchDate)) {
            builder.rule(RefundRule.builder()
                    .type(RefundType.FULLY_REFUNDABLE)
                    .endsAt(penaltyStartDate)
                    .build());
            hasInterval = true;
        }
        if (nonRefStartDate.isAfter(penaltyStartDate) && nonRefStartDate.isAfter(searchDate)) {
            builder.rule(RefundRule.builder()
                    .type(RefundType.REFUNDABLE_WITH_PENALTY)
                    .startsAt(hasInterval ? penaltyStartDate : null)
                    .endsAt(nonRefStartDate)
                    .penalty(computePenalty(plan, offer, currencyCode))
                    .build());
            hasInterval = true;
        }
        builder.rule(RefundRule.builder()
                .type(RefundType.NON_REFUNDABLE)
                .startsAt(hasInterval ? nonRefStartDate : null)
                .build());
        return builder.build();
    }

    private static Money computePenalty(RatePlan plan, Offer offer, String currencyCode) {
        switch (plan.getCancellationFineType()) {
            case FIXED_AMOUNT:
                return Money.of(plan.getCancellationFineAmount().min(offer.getPrice()),
                        ProtoCurrencyUnit.fromCurrencyCode(currencyCode));
            case FIRST_NIGHT:
                LocalDate firstNight = offer.getPricesByDates().keySet().stream().min(Comparator.naturalOrder()).get();
                BigDecimal firstNightPrice = offer.getPricesByDates().get(firstNight);
                return Money.of(firstNightPrice.min(offer.getPrice()),
                        ProtoCurrencyUnit.fromCurrencyCode(currencyCode));
            case PERCENTAGE:
                Preconditions.checkArgument(plan.getCancellationFineAmount().intValue() <= 100,
                        "Invalid cancellation fine percentage");
                BigDecimal fineAmount = plan.getCancellationFineAmount().divide(BigDecimal.valueOf(100), 4,
                        RoundingMode.HALF_UP)
                        .multiply(offer.getPrice());
                if (fineAmount.compareTo(offer.getPrice()) > 0) {
                    fineAmount = offer.getPrice();
                }
                return Money.of(fineAmount, ProtoCurrencyUnit.fromCurrencyCode(currencyCode));
            case FULL_AMOUNT:
                return Money.of(offer.getPrice(), ProtoCurrencyUnit.fromCurrencyCode(currencyCode));
            default:
                throw new IllegalStateException("Should not be here");
        }
    }
}

