package ru.yandex.travel.hotels.common.partners.booking;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.Getter;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.booking.model.BlockAvailability;
import ru.yandex.travel.hotels.common.partners.booking.model.Hotels;
import ru.yandex.travel.hotels.common.partners.booking.model.Hotel;

public class DefaultBookingClient extends BaseClient<BookingClientProperties> implements BookingClient {

    private static final String BOOKING_DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    private static final String BLOCK_AVAILABILITY = "BlockAvailability";
    private static final String HOTELS = "Hotels";
    private static final String EXTRAS = "cancellation_info,facilities,important_information,internet," +
            "max_children_free,max_children_free_age,mealplans,number_of_rooms_left,photos,policies,room_type_id";
    @Getter
    private static final ClientMethods methods;

    static {
        methods = new ClientMethods()
                .register(BLOCK_AVAILABILITY, "/blockAvailability", BaseClient.HttpMethod.GET, BlockAvailability.class)
                .register(HOTELS, "/hotels", BaseClient.HttpMethod.GET, Hotels.class);
    }

    public DefaultBookingClient(AsyncHttpClientWrapper clientWrapper, BookingClientProperties properties, Retry retryHelper) {
        super(properties, clientWrapper, createObjectMapper(), retryHelper);
    }

    public static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(
                LocalDateTime.class,
                new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern(BOOKING_DATE_TIME_PATTERN)));

        javaTimeModule.addSerializer(
                LocalDateTime.class,
                new LocalDateTimeSerializer(DateTimeFormatter.ofPattern(BOOKING_DATE_TIME_PATTERN)));
        mapper.registerModule(javaTimeModule);
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    @Override
    public CompletableFuture<BlockAvailability> getBlockAvailability(String checkIn, String checkOut,
                                                                     Set<String> hotelIds, String occupancy,
                                                                     ECurrency currency, String requestId,
                                                                     BookingUserPlatform userPlatform) {
        var callArguments = new CallArguments()
                .withQueryParam("checkin", checkIn)
                .withQueryParam("checkout", checkOut)
                .withQueryParam("hotel_ids", String.join(",", hotelIds))
                .withQueryParam("num_rooms", "1")  // TODO: later we'll need a smarted rooming logic
                .withQueryParam("room1", occupancy)
                .withQueryParam("extras", EXTRAS)
                .withQueryParam("language", "ru")
                .withQueryParam("affiliate_id", properties.getAffiliateId())
                .withQueryParam("currency", Utils.CURRENCY_NAMES.get(currency))
                .withRequestId(requestId);
        if (userPlatform != BookingUserPlatform.NOT_SPECIFIED) {
            callArguments.withQueryParam("user_platform", userPlatform.getValue());
        }
        return call(BLOCK_AVAILABILITY, callArguments);
    }

    public CompletableFuture<Hotels> getHotels(Set<String> hotelIds, String requestId) {
        return call(HOTELS,
                new CallArguments()
                    .withQueryParam("extras", "hotel_info")
                    .withQueryParam("hotel_ids", String.join(",", hotelIds))
                    .withQueryParam("language", "ru")
                        .withQueryParam("affiliate_id", properties.getAffiliateId())
                    .withRequestId(requestId));
    }

    public CompletableFuture<Hotel> getHotel(String hotelId, String requestId) {
        Set<String> hotelIds = new HashSet<>();
        hotelIds.add(hotelId);

        return getHotels(hotelIds, requestId)
                .thenApply(result -> result.getResult().get(0));
    }

    @Override
    protected ClientMethods getClientMethods() {
        return methods;
    }

    @Override
    protected @NotNull Map<String, String> getCommonHeaders(String destinationMethod) {
        return Map.of(
                "Authorization", Utils.createBasicAuthHeader(properties.getUsername(), properties.getPassword())
        );
    }
}
