package ru.yandex.travel.hotels.common.partners.booking.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.booking.model.Block;
import ru.yandex.travel.hotels.common.partners.booking.model.CancellationInfo;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

@Slf4j
public class BookingRefundRulesBuilder {
    private static final Duration MAX_GAP = Duration.of(1, ChronoUnit.MINUTES);
    private static final Duration SILENT_MAX_GAP = Duration.of(1, ChronoUnit.SECONDS);


    public static Map<String, String> knownTimezones = Map.ofEntries(
            Map.entry("ACST", "+0930"),
            Map.entry("ADT", "-03"),
            Map.entry("AEDT", "+11"),
            Map.entry("AEST", "+10"),
            Map.entry("AKDT", "-08"),
            Map.entry("AST", "-04"),
            Map.entry("AWST", "+08"),
            Map.entry("BST", "+01"),
            Map.entry("CAT", "+02"),
            Map.entry("CDT", "-05"),
            Map.entry("CEST", "+02"),
            Map.entry("CET", "+01"),
            Map.entry("CST", "-06"),
            Map.entry("ChST", "+10"),
            Map.entry("EAT", "+03"),
            Map.entry("EDT", "-04"),
            Map.entry("EEST", "+03"),
            Map.entry("EET", "+02"),
            Map.entry("EST", "-05"),
            Map.entry("HKT", "+08"),
            Map.entry("HST", "-10"),
            Map.entry("IDT", "+03"),
            Map.entry("IST", "+02"),
            Map.entry("JST", "+09"),
            Map.entry("KST", "+09"),
            Map.entry("MDT", "-06"),
            Map.entry("MSK", "+03"),
            Map.entry("MST", "-07"),
            Map.entry("NDT", "-0230"),
            Map.entry("NZST", "+12"),
            Map.entry("PDT", "-07"),
            Map.entry("PKT", "+05"),
            Map.entry("PST", "-08"),
            Map.entry("SAST", "+02"),
            Map.entry("SST", "+08"),
            Map.entry("WAT", "+01"),
            Map.entry("WEST", "+01"),
            Map.entry("WET", "+00"),
            Map.entry("WIB", "+07"),
            Map.entry("WIT", "+07"),
            Map.entry("WITA", "+08")
    );

    public static RefundRules build(Block block, LocalDateTime startDate, LocalDateTime endDate, double totalPrice) {
        var builder = RefundRules.builder();
        List<RefundRule> rules = new ArrayList<>();

        List<CancellationInfo> bookingCancellationInfo;
        if (block.getCancellationInfo() == null) {
            bookingCancellationInfo = new ArrayList<>();
        } else {
            bookingCancellationInfo = block.getCancellationInfo();
        }
        var maxDate = endDate.plusDays(1);
        if (bookingCancellationInfo.stream().anyMatch(penalty -> penalty.getFrom() != null && penalty.getFrom().compareTo(maxDate) > 0)) {
            throw new PartnerException(String.format("Block %s: Date 'from' of cancellation rule is greater than maxDate",
                    block.getBlockId()));
        }
        bookingCancellationInfo = bookingCancellationInfo.stream()
                .map(ci -> {
                    // Seems like booking uses this value as +inf
                    if (LocalDateTime.of(9999, 12, 31, 23, 59, 59).equals(ci.getUntil())) {
                        var ciBuilder = ci.toBuilder();
                        ciBuilder.until(null);
                        return ciBuilder.build();
                    } else if (ci.getUntil() != null && ci.getUntil().isAfter(maxDate)) {
                        log.warn(String.format("Block %s: Date 'until' of cancellation rule is greater than maxDate, limiting to maxDate",
                                block.getBlockId()));
                        var ciBuilder = ci.toBuilder();
                        ciBuilder.until(maxDate);
                        return ciBuilder.build();
                    } else {
                        return ci;
                    }
                }).filter(ci -> {
                    if (ci.getFrom() != null && ci.getUntil() != null && ci.getFrom().isAfter(ci.getUntil())) {
                        log.warn(String.format("Block %s: Date 'until' of cancellation rule is less than 'from', skipping rule",
                                block.getBlockId()));
                        return false;
                    } else {
                        return true;
                    }
                }).collect(Collectors.toList());
        if (bookingCancellationInfo.stream().filter(penalty -> penalty.getFrom() == null).count() > 1) {
            throw new PartnerException(String.format("Block %s: More than 1 interval with empty start is found",
                    block.getBlockId()));
        }
        if (bookingCancellationInfo.stream().filter(penalty -> penalty.getUntil() == null).count() > 1) {
            throw new PartnerException(String.format("Block %s: More than 1 interval with empty end are found",
                    block.getBlockId()));
        }
        bookingCancellationInfo.sort((penalty1, penalty2) -> {
            if (penalty1.getFrom() == null) {
                return -1;
            }
            if (penalty2.getFrom() == null) {
                return 1;
            }
            int result = penalty1.getFrom().compareTo(penalty2.getFrom());
            if (result != 0) {
                return result;
            }
            if (penalty1.getUntil() == null) {
                return 1;
            }
            if (penalty2.getUntil() == null) {
                return -1;
            }
            return penalty1.getUntil().compareTo(penalty2.getUntil());
        });
        ZoneId zoneId = findZoneId(bookingCancellationInfo);

        Instant firstPenaltyStart;
        RefundRule previousRule = null;
        Instant startInstant = getInstant(startDate, zoneId);
        if (bookingCancellationInfo.isEmpty()) {
            return builder.build();
        } else {
            firstPenaltyStart = getInstant(bookingCancellationInfo.get(0).getFrom(), zoneId);
            if (firstPenaltyStart != null && firstPenaltyStart.isAfter(startInstant)) {
                firstPenaltyStart = startInstant;
            }
        }
        if (firstPenaltyStart != null) {
            RefundRule fullyRefundableRule = RefundRule.builder()
                    .endsAt(firstPenaltyStart)
                    .type(RefundType.FULLY_REFUNDABLE)
                    .build();
            rules.add(fullyRefundableRule);
            previousRule = fullyRefundableRule;
        }
        BigDecimal previousPenaltyAmount = null;
        Instant previousEnd = null;
        for (CancellationInfo cancellationInfo : bookingCancellationInfo) {
            if (cancellationInfo.getFrom() != null && getInstant(cancellationInfo.getFrom(), zoneId).isAfter(startInstant)) {
                continue;
            }
            BigDecimal currentPenaltyAmount = BigDecimal.valueOf(cancellationInfo.getGuestCurrencyFee());

            Instant endsAt = getInstant(cancellationInfo.getUntil(), zoneId);
            Instant startsAt = getInstant(cancellationInfo.getFrom(), zoneId);

            if (previousEnd != null && startsAt != null) {
                if (previousEnd.isBefore(startsAt)) {
                    if (Duration.between(previousEnd, startsAt).compareTo(SILENT_MAX_GAP) <= 0) {
                        startsAt = previousEnd;
                    } else if (Duration.between(previousEnd, startsAt).compareTo(MAX_GAP) <= 0) {
                        log.warn(String.format("Block %s: Gap between intervals found from %s till " +
                                "%s, merging by moving next interval back", block.getBlockId(), previousEnd, startsAt));
                        startsAt = previousEnd;
                    } else {
                        throw new PartnerException(String.format("Block %s: Gap between intervals found from %s till " +
                                "%s", block.getBlockId(), previousEnd, startsAt));
                    }
                } else if (previousEnd.isAfter(startsAt)) {
                    throw new PartnerException(String.format("Block %s: Intervals overlap from %s till %s",
                            block.getBlockId(), startsAt, previousEnd));
                }
            }
            if (startsAt != null && startsAt.equals(endsAt)) {
                //booking may return the last interval, which is not limited, with same start and end
                if (cancellationInfo.equals(bookingCancellationInfo.get(bookingCancellationInfo.size() - 1))) {
                    endsAt = null;
                } else {
                    continue;
                }
            }
            previousEnd = endsAt;

            if (previousPenaltyAmount != null && previousPenaltyAmount.compareTo(currentPenaltyAmount) > 0) {
                throw new PartnerException(String.format("Block %s: previous penalty amount %s is greater then " +
                        "next penalty amount %s", block.getBlockId(), previousPenaltyAmount, currentPenaltyAmount));
            }
            previousPenaltyAmount = currentPenaltyAmount;

            RefundType type;
            Money ruleAmount = null;
            if (currentPenaltyAmount.compareTo(BigDecimal.ZERO) == 0) {
                type = RefundType.FULLY_REFUNDABLE;
            } else if (currentPenaltyAmount.setScale(0, RoundingMode.HALF_UP).compareTo(BigDecimal.valueOf(totalPrice)) >= 0) {
                type = RefundType.NON_REFUNDABLE;
            } else {
                type = RefundType.REFUNDABLE_WITH_PENALTY;
                ruleAmount = Money.of(currentPenaltyAmount, ProtoCurrencyUnit.fromCurrencyCode(cancellationInfo.getGuestCurrency()));
            }
            if (startsAt != null && endsAt != null && !startsAt.isBefore(endsAt)) {
                log.warn(String.format("Block %s: Start date should be before end date", block.getBlockId()));
                continue;
            }
            RefundRule nextRule = RefundRule.builder()
                    .startsAt(startsAt)
                    .endsAt(endsAt)
                    .type(type)
                    .penalty(ruleAmount)
                    .build();
            if (previousRule != null && penaltyAmountsAreEqual(previousRule, nextRule)) {
                rules.remove(rules.size() - 1);
                rules.add(RefundRule.builder()
                        .startsAt(previousRule.getStartsAt())
                        .endsAt(nextRule.getEndsAt())
                        .type(previousRule.getType())
                        .penalty(previousRule.getPenalty())
                        .build());
                previousRule = rules.get(rules.size() - 1);
            } else {
                rules.add(nextRule);
                previousRule = nextRule;
            }
        }
        builder.rules(rules);
        return builder.build();
    }

    private static boolean penaltyAmountsAreEqual(RefundRule rule1, RefundRule rule2) {
        if (!rule1.getType().equals(rule2.getType())) {
            return false;
        }
        if (rule1.getType().equals(RefundType.REFUNDABLE_WITH_PENALTY)) {
            return rule1.getPenalty().compareTo(rule2.getPenalty()) == 0;
        } else {
            return true;
        }
    }

    private static ZoneId findZoneId(List<CancellationInfo> bookingCancellationInfo) {
        for (CancellationInfo cancellationInfo: bookingCancellationInfo) {
            if (cancellationInfo.getTimezone() != null) {
                try {
                    if (knownTimezones.containsKey(cancellationInfo.getTimezone())) {
                        return ZoneId.of(knownTimezones.get(cancellationInfo.getTimezone()));
                    } else {
                        return ZoneId.of(cancellationInfo.getTimezone());
                    }
                } catch (Exception e) {
                    log.info("Unable to parse zoneId: " + cancellationInfo.getTimezone());
                }
            }
        }
        return ZoneOffset.UTC;
    }

    private static Instant getInstant(LocalDateTime dateTime, ZoneId zoneOffset) {
        if (dateTime == null) {
            return null;
        }
        return dateTime.atZone(zoneOffset).toInstant();
    }

}

