package ru.yandex.travel.hotels.common.partners.bronevik;

import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.ws.AsyncHandler;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.handler.Handler;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.apache.xerces.jaxp.datatype.DatatypeFactoryImpl;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.commons.metrics.TravelTag;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.RetryStrategy;
import ru.yandex.travel.hotels.common.partners.base.BaseSOAPClient;
import ru.yandex.travel.hotels.common.partners.base.SOAPOperations;
import ru.yandex.travel.hotels.common.partners.bronevik.handlers.HTTPRequestHeadersHandler;
import ru.yandex.travel.hotels.common.partners.bronevik.model.MealsDevelopment;
import ru.yandex.travel.hotels.common.partners.bronevik.model.MealsProduction;
import ru.yandex.travel.hotels.common.partners.bronevik.model.Operation;
import ru.yandex.travel.hotels.common.partners.bronevik.model.Meal;
import ru.yandex.travel.hotels.common.partners.bronevik.utils.BronevikUtils;

@Slf4j
@Getter
@Setter
public class DefaultBronevikClient extends BaseSOAPClient<BronevikClientProperties> implements BronevikClient {

    private String LANGUAGE = "ru";

    private Service bronevikService;
    private Bronevik bronevik;
    private ObjectFactory bronevikObjectFactory;
    private SOAPOperations operations;
    private Credentials credentials;
    private SOAPType soapType;
    private ConcurrentMap<Integer, Counter> errorCodeCounters;
    private final Retry retryHelper;

    public DefaultBronevikClient(BronevikClientProperties properties, Logger logger, Retry retryHelper) {
        super(properties, logger, "bronevik");
        this.errorCodeCounters = new ConcurrentHashMap<>();
        ClassLoader classLoader = getClass().getClassLoader();
        URL bronevikWSDLUrl = classLoader.getResource(properties.getWsdlLocation());
        this.bronevik = new Bronevik(bronevikWSDLUrl);
        this.bronevikObjectFactory = new ObjectFactory();
        this.credentials = prepareCredentials(properties.getLogin(), properties.getPassword(), properties.getClientKey());
        this.soapType = properties.getSoapType();
        this.retryHelper = retryHelper;

        if (soapType == SOAPType.PRODUCTION) {
            this.bronevikService = this.bronevik.getProductionSOAP();
        } else {
            this.bronevikService = this.bronevik.getDevelopmentSOAP();
        }

        //  this is only into documentation https://javaee.github.io/metro-jax-ws/doc/user-guide/ch03.html

        List<Handler> handlerChain = ((BindingProvider) this.bronevikService).getBinding().getHandlerChain();
        if (handlerChain == null) {
            handlerChain = new ArrayList<>();
        }
        handlerChain.add(new HTTPRequestHeadersHandler());
        ((BindingProvider) this.bronevikService).getBinding().setHandlerChain(handlerChain);

        this.operations = new SOAPOperations();
        operations.register(
            Operation.PING.getValue(),
            (PingRequest request, AsyncHandler<PingResponse> asyncHandler) ->  bronevikService.pingAsync(request, asyncHandler)
        );
        operations.register(
            Operation.GET_MEALS.getValue(),
            (GetMealsRequest request, AsyncHandler<GetMealsResponse> asyncHandler) -> bronevikService.getMealsAsync(request, asyncHandler)
        );
        operations.register(
            Operation.SEARCH_HOTEL_OFFERS.getValue(),
            (SearchHotelOffersRequest request, AsyncHandler<SearchHotelOffersResponse> asyncHandler) -> bronevikService.searchHotelOffersAsync(request, asyncHandler)
        );
        operations.register(
            Operation.GET_HOTEL_INFO.getValue(),
            (GetHotelInfoRequest request, AsyncHandler<GetHotelInfoResponse> asyncHandler) -> bronevikService.getHotelInfoAsync(request, asyncHandler)
        );
        operations.register(
            Operation.GET_HOTEL_OFFER_PRICING.getValue(),
            (GetHotelOfferPricingRequest request, AsyncHandler<GetHotelOfferPricingResponse> asyncHandler) -> bronevikService.getHotelOfferPricingAsync(request, asyncHandler)
        );
        operations.register(
            Operation.CREATE_ORDER.getValue(),
            (CreateOrderRequest request, AsyncHandler<CreateOrderResponse> asyncHandler) -> bronevikService.createOrderAsync(request, asyncHandler)
        );
        operations.register(
            Operation.CANCEL_ORDER.getValue(),
            (CancelOrderRequest request, AsyncHandler<CancelOrderResponse> asyncHandler) -> bronevikService.cancelOrderAsync(request, asyncHandler)
        );
        operations.register(
            Operation.SEARCH_ORDERS.getValue(),
            (SearchOrdersRequest request, AsyncHandler<SearchOrdersResponse> asyncHandler) -> bronevikService.searchOrdersAsync(request, asyncHandler)
        );
        operations.register(
            Operation.GET_AMENITIES.getValue(),
            (GetAmenitiesRequest request, AsyncHandler<GetAmenitiesResponse> asyncHandler) -> bronevikService.getAmenitiesAsync(request, asyncHandler)
        );
    }

    public <R extends BaseRequest, A extends BaseResponse> CompletableFuture<A> call(String operation, R request, String requestId) {
        request.setCredentials(credentials);
        request.setLanguage(LANGUAGE);
        return super.call(operation, request, requestId);
    }

    @SuppressWarnings("unchecked")
    private <R extends BaseRequest, A extends BaseResponse, E extends  Exception> CompletableFuture<A> run(
            String operation,
            R request,
            String requestId,
            Class<E> exceptionClazz
    ) {
        return (CompletableFuture<A>) call(operation, request, requestId).exceptionally(e -> {

            if (e instanceof ExecutionException && exceptionClazz.isInstance(e.getCause())) {

                var fault = e.getCause();

                // TODO: рефлексия, потому что класс ответа один и тот же, просто при генерации из него создаются все возможные классы ошшибок + уже была проверка что это действительно нужный класс
                // Просьба обработку ошибки для каждого метода непосредственно в нем
                try {
                    var method = exceptionClazz.getMethod("getFaultInfo");
                    var faultInfo = (FaultDetail) method.invoke(fault);
                    var counter = errorCodeCounters.computeIfAbsent(faultInfo.getCode(), k -> Metrics.counter(
                            "searcher.partners.bronevik.error_code",
                            List.of(new TravelTag("code", k.toString(), false))
                    ));
                    counter.increment();
                } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException ex) {
                    throw ExceptionUtils.translate(ex);
                }

                throw new BronevikException(fault);

            }
            throw ExceptionUtils.throwException(e);
        });
    }

    private <E extends Exception, A extends BaseResponse, R extends BaseRequest> CompletableFuture<A> runWithRetry(
            String operation,
            R request,
            String requestId,
            Class<E> exceptionClazz) {

        request.setCredentials(credentials);

        RetryStrategy<A> retryStrategy = new BronevikRetryStrategy<>(properties.getNumRetries());
        RetryRateLimiter retryRateLimiter = new RetryRateLimiter(properties.getRetryRateLimiter());
        if (properties.isEnableRetries()){
            return retryHelper.withRetry("BronevikClient", () -> run(operation, request, requestId, exceptionClazz), retryStrategy, retryRateLimiter);
        } else {
            return run(operation, request, requestId, exceptionClazz);
        }
    }


    private ServiceAccommodation prepareServiceAccommodation(String offerCode, List<Guest> guests,
                                                             List<Integer> meals, String referenceId, List<Child> children) {
        ServiceAccommodation serviceAccommodation = new ServiceAccommodation();
        Guests guestsWrapper = new Guests();
        guestsWrapper.setGuest(guests);
        guestsWrapper.setChildren(children);
        serviceAccommodation.setGuests(guestsWrapper);
        serviceAccommodation.setOfferCode(offerCode);
        serviceAccommodation.setMeals(meals);
        serviceAccommodation.setReferenceId(referenceId);

        return serviceAccommodation;
    }

    private SearchHotelOffersRequest prepareSearchHotelOffersRequest(
            int adults,
            List<Integer> children,
            List<Integer> hotelIds,
            String checkIn,
            String checkOut,
            String currency
    ) throws DatatypeConfigurationException {
        var datatypeFactory = DatatypeFactoryImpl.newInstance();

        var arrivalDate = datatypeFactory.newXMLGregorianCalendar(LocalDate.parse(checkIn).toString());
        var departureDate = datatypeFactory.newXMLGregorianCalendar(LocalDate.parse(checkOut).toString());

        var request = bronevikObjectFactory.createSearchHotelOffersRequest();
        // заполняем даты выезда и въезда
        request.setArrivalDate(arrivalDate);
        request.setDepartureDate(departureDate);
        var hotelidsObj = new HotelIds();
        hotelidsObj.setId(hotelIds);
        request.setHotelIds(hotelidsObj);
        request.setCurrency(currency);

        // заполняем условия поиска
        var searchCriterions = new ArrayList<SearchOfferCriterion>();

        var searchOfferCriterionNumberOfGuests = new SearchOfferCriterionNumberOfGuests();
        searchOfferCriterionNumberOfGuests.setAdults(adults);
        searchOfferCriterionNumberOfGuests.setChildren(BronevikUtils.parseChildren(children));
        searchCriterions.add(searchOfferCriterionNumberOfGuests);

        var searchOfferCriterionOnlyOnline = new SearchOfferCriterionOnlyOnline();
        searchCriterions.add(searchOfferCriterionOnlyOnline);

        var searchOfferCriterionPaymentRecipient = new SearchOfferCriterionPaymentRecipient();
        searchOfferCriterionPaymentRecipient.setPaymentRecipient(List.of(PaymentRecipients.AGENCY));
        searchCriterions.add(searchOfferCriterionPaymentRecipient);

        request.setSearchCriteria(searchCriterions);

        return request;
    }

    @NotNull
    private Credentials prepareCredentials(
            String login,
            String password,
            String clientKey
    ) {
        Credentials credentials = bronevikObjectFactory.createCredentials();
        credentials.setClientKey(clientKey);
        credentials.setPassword(password);
        credentials.setLogin(login);
        return credentials;
    }

    private GetHotelInfoRequest prepareHotelInfoRequest(List<Integer> hotelIds) {
        GetHotelInfoRequest getHotelInfoRequest = bronevikObjectFactory.createGetHotelInfoRequest();
        var ids = new HotelIds();
        ids.setId(hotelIds);
        getHotelInfoRequest.setHotelIds(ids);

        return getHotelInfoRequest;
    }

    @Override
    public CompletableFuture<PingResponse> ping() {
        var pingRequest = bronevikObjectFactory.createPingRequest();
        return runWithRetry(Operation.PING.getValue(), pingRequest, UUID.randomUUID().toString(), PingFault.class);
    }

    @Override
    protected SOAPOperations getOperations() {
        return operations;
    }


    @Override
    public CompletableFuture<HotelsWithInfo> getHotelsInfo(List<Integer> hotelIds, String requestId) {
        var request = prepareHotelInfoRequest(hotelIds);
        CompletableFuture<GetHotelInfoResponse> resp
                = runWithRetry(Operation.GET_HOTEL_INFO.getValue(), request, requestId, GetHotelInfoFault.class);
        return resp.thenApply(GetHotelInfoResponse::getHotels);
    }

    @Override
    public CompletableFuture<HotelWithInfo> getHotelInfo(Integer hotelId, String requestId) {
        return getHotelsInfo(List.of(hotelId), requestId).thenApply(
                hotelsWithInfo -> hotelsWithInfo.getHotel().isEmpty() ? null : hotelsWithInfo.getHotel().get(0)
        );
    }

    @Override
    public CompletableFuture<GetMealsResponse> getMeals(String requestId) {
        var request = bronevikObjectFactory.createGetMealsRequest();
        return runWithRetry(Operation.GET_MEALS.getValue(), request, requestId, GetMealsFault.class);
    }

    @Override
    public CompletableFuture<SearchHotelOffersResponse> searchHotelOffers(int adults, List<Integer> children,
                                                                          List<Integer> hotelIds,
                                                                          String checkIn, String checkOut,
                                                                          String currency, String requestId) {
        SearchHotelOffersRequest request;
        try {
            request = prepareSearchHotelOffersRequest(adults, children, hotelIds, checkIn, checkOut, currency);
        } catch (DatatypeConfigurationException e) {
            e.printStackTrace();
            return CompletableFuture.completedFuture(null);
        }
        return runWithRetry(Operation.SEARCH_HOTEL_OFFERS.getValue(), request, requestId, SearchHotelOffersFault.class);

    }

    public Meal getMeal(Integer mealId) {
        if (soapType == SOAPType.PRODUCTION) {
            return MealsProduction.getMeal(mealId);
        } else {
            return MealsDevelopment.getMeal(mealId);
        }
    }

    @Override
    public CompletableFuture<GetHotelOfferPricingResponse> getHotelOfferPricing(String offerCode, List<Guest> guests,
                                                                                List<Integer> meals, String currency,
                                                                                String requestId, List<Child> children) {
        GetHotelOfferPricingRequest request = bronevikObjectFactory.createGetHotelOfferPricingRequest();
        request.setServices(List.of(prepareServiceAccommodation(offerCode, guests, meals, null, children)));
        request.setCurrency(currency);

        return runWithRetry(Operation.GET_HOTEL_OFFER_PRICING.getValue(), request, requestId, GetHotelOfferPricingFault.class);
    }

    @Override
    public CompletableFuture<CreateOrderResponse> createOrder(String offerCode, List<Guest> guests,
                                                              List<Integer> meals, String currency,
                                                              String referenceId, String requestId, List<Child> children) {
        CreateOrderRequest request = bronevikObjectFactory.createCreateOrderRequest();
        request.setServices(List.of(prepareServiceAccommodation(offerCode, guests, meals, referenceId, children)));
        request.setCurrency(currency);

        return runWithRetry(Operation.CREATE_ORDER.getValue(), request, requestId, CreateOrderFault.class);
    }

    @Override
    public CompletableFuture<GetOrderResponse> getOrder(Integer OrderId, String requestId) {
        GetOrderRequest request = bronevikObjectFactory.createGetOrderRequest();
        request.setOrderId(OrderId);

        return runWithRetry(Operation.GET_ORDER.getValue(), request, requestId, GetOrderFault.class);
    }

    @Override
    public CompletableFuture<CancelOrderResponse> cancelOrder(Integer orderId, String requestId) {
        CancelOrderRequest request = bronevikObjectFactory.createCancelOrderRequest();
        request.setOrderId(orderId);

        return runWithRetry(Operation.CANCEL_ORDER.getValue(), request, requestId, CancelOrderFault.class);
    }

    @Override
    public CompletableFuture<SearchOrdersResponse> searchOrdersByReferenceId(String referenceId, String requestId) {
        SearchOrdersRequest request = bronevikObjectFactory.createSearchOrdersRequest();

        List<SearchOrderCriterion> criterions = new ArrayList<>();
        var referenceCriterion = new SearchOrderCriterionServiceReferenceId();
        referenceCriterion.setReferenceId(referenceId);
        criterions.add(referenceCriterion);
        request.setSearchCriteria(criterions);

        return runWithRetry(Operation.SEARCH_ORDERS.getValue(), request, requestId, SearchOrdersFault.class);
    }

    @Override
    public CompletableFuture<GetAmenitiesResponse> getAmenities(String requestId) {
        GetAmenitiesRequest request = bronevikObjectFactory.createGetAmenitiesRequest();

        return runWithRetry(Operation.GET_AMENITIES.getValue(), request, requestId, GetAmenitiesFault.class);
    }

}
