package ru.yandex.travel.hotels.common.partners.bronevik.utils

import org.javamoney.moneta.Money
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit
import ru.yandex.travel.hotels.common.pansions.PansionUnifier
import ru.yandex.travel.hotels.common.partners.bronevik.AvailableMeal
import ru.yandex.travel.hotels.common.partners.bronevik.AvailableMeals
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikRefundRulesException
import ru.yandex.travel.hotels.common.partners.bronevik.Child
import ru.yandex.travel.hotels.common.partners.bronevik.ClientPriceDetails
import ru.yandex.travel.hotels.common.partners.bronevik.HotelOfferCancellationPolicy
import ru.yandex.travel.hotels.common.partners.bronevik.PriceDetails
import ru.yandex.travel.hotels.common.partners.bronevik.SOAPType
import ru.yandex.travel.hotels.common.partners.bronevik.SOAPType.DEVELOPMENT
import ru.yandex.travel.hotels.common.partners.bronevik.SOAPType.PRODUCTION
import ru.yandex.travel.hotels.common.partners.bronevik.model.Meal
import ru.yandex.travel.hotels.common.partners.bronevik.model.MealsDevelopment
import ru.yandex.travel.hotels.common.partners.bronevik.model.MealsProduction
import ru.yandex.travel.hotels.common.refunds.RefundRule
import ru.yandex.travel.hotels.common.refunds.RefundRules
import ru.yandex.travel.hotels.common.refunds.RefundType.FULLY_REFUNDABLE
import ru.yandex.travel.hotels.common.refunds.RefundType.NON_REFUNDABLE
import ru.yandex.travel.hotels.common.refunds.RefundType.REFUNDABLE_WITH_PENALTY
import ru.yandex.travel.hotels.proto.EPansionType
import java.time.Instant
import java.time.LocalDate
import java.util.Stack
import javax.xml.datatype.XMLGregorianCalendar
import kotlin.math.roundToInt
import kotlin.math.roundToLong

class BronevikUtils {

    companion object {

        @JvmStatic
        fun getTotalPrice(offerPriceDetails: PriceDetails, availableMeals: AvailableMeals, countGuests: Int, countNights: Int): Int {
            val mealsPrice: Int = availableMeals
                .meal
                .map { getPriceInt(it.priceDetails) }
                .reduceOrNull { a: Int, b: Int -> a + b }
                ?:0
            val offerPrice = getPriceInt(offerPriceDetails.client)
            return offerPrice + countGuests * mealsPrice * countNights
        }

        @JvmStatic
        fun getPriceInt(offerPriceDetails: ClientPriceDetails): Int {
            return offerPriceDetails.clientCurrency.gross.price.roundToInt()
        }
        @JvmStatic
        fun getPriceLong(offerPriceDetails: ClientPriceDetails): Long {
            return offerPriceDetails.clientCurrency.gross.price.roundToLong()
        }

        @JvmStatic
        fun parseMeals(possibleMeals: AvailableMeals?, soapType: SOAPType): EPansionType {
            var fb = false
            var hb = false
            var breakfast = false
            var lunch = false
            var dinner = false

            possibleMeals?.let {
                for (possibleMeal in it.meal) {
                    val baseMeal = getMeal(possibleMeal.id, soapType)
                    when {
                        baseMeal.isFullBoard -> fb = true
                        baseMeal.isHalfBoard -> hb = true
                        baseMeal.isLunch -> lunch = true
                        baseMeal.isBreakfast -> breakfast = true
                        baseMeal.isDinner -> dinner = true
                    }
                }
            }

            return PansionUnifier.get(false, fb, hb, breakfast, lunch, dinner)
        }

        @JvmStatic
        fun generateMealsForOffer(possibleMeals: AvailableMeals?, soapType: SOAPType): List<AvailableMeals> {
            val includedMeals = possibleMeals?.let { it.meal.filter { meal -> meal.isIncluded } }.orEmpty()
            val notIncludedMeals = possibleMeals?.let { it.meal.filter { meal -> !meal.isIncluded } }.orEmpty()
            val allMeals = generateMeals(notIncludedMeals, 0, Stack())
            allMeals.forEach { availableMeals -> availableMeals.meal.addAll(includedMeals) }
            return allMeals.filter { availableMeals -> checkMeals(availableMeals.meal, soapType) }
        }

        @JvmStatic
        fun getNights(checkin: String, checkout: String): Int {
            return (LocalDate.parse(checkout).toEpochDay() - LocalDate.parse(checkin).toEpochDay()).toInt()
        }

        @JvmStatic
        fun getMeal(id: Int, soapType: SOAPType): Meal {
            return when(soapType) {
                DEVELOPMENT -> MealsDevelopment.getMeal(id)
                PRODUCTION -> MealsProduction.getMeal(id)
            }
        }

        @JvmStatic
        fun getMealName(id: Int, soapType: SOAPType): String {
            val meal = getMeal(id, soapType)
            return when {
                meal.isBreakfast -> "Завтрак"
                meal.isDinner -> "Ужин"
                meal.isLunch -> "Обед"
                meal.isHalfBoard -> "Полупансион"
                meal.isFullBoard -> "Полный пансион"
                else -> ""
            }
        }

        @JvmStatic
        fun checkMeals(meals: List<AvailableMeal>, soapType: SOAPType): Boolean {
            var breakfast = false // завтрак
            var lunch = false // обед
            var dinner = false // ужин
            for (meal in meals) {
                val baseMeal: Meal = getMeal(meal.id, soapType)
                when {
                    baseMeal.isDinner -> {
                        if (dinner) {
                            return false
                        }
                        dinner = true
                    }
                    baseMeal.isBreakfast -> {
                        if (breakfast) {
                            return false
                        }
                        breakfast = true
                    }
                    baseMeal.isLunch -> {
                        if (lunch) {
                            return false
                        }
                        lunch = true
                    }
                    baseMeal.isFullBoard -> {
                        if (breakfast || lunch || dinner) {
                            return false
                        }
                        breakfast = true
                        lunch = true
                        dinner = true
                    }
                    baseMeal.isHalfBoard -> {
                        if (breakfast || dinner) {
                            return false
                        }
                        breakfast = true
                        dinner = true
                    }
                }
            }
            return true
        }

        @JvmStatic
        fun generateMeals(meals: List<AvailableMeal>, index: Int, generatedMeals: Stack<AvailableMeal>): List<AvailableMeals> {
            if (index == meals.size) {
                val ans = AvailableMeals()
                ans.meal = generatedMeals.toMutableList()
                return listOf(ans)
            }

            val notIncluded = generateMeals(meals, index + 1 , generatedMeals)
            generatedMeals.push(meals[index])
            val included = generateMeals(meals, index + 1, generatedMeals)
            generatedMeals.pop()
            val ans = included.toMutableList()
            ans.addAll(notIncluded)
            return ans
        }

        @JvmStatic
        fun parseChildren(children: List<Int>): List<Child> {
            return children.groupingBy { it }.eachCount()
                .map {
                    val child = Child()

                    child.age = it.key
                    child.count = it.value

                    return@map child
                }
        }

        @JvmStatic
        @Throws(BronevikRefundRulesException::class)
        fun parseRefundRules(
            cancellationPolicies: List<HotelOfferCancellationPolicy>,
            total: Int
        ): RefundRules {
            val reversedCancellationPolicies = cancellationPolicies
                .sortedBy { it.penaltyDateTime.toInstant() }
                .reversed()
            val firstPenaltyDateTime = reversedCancellationPolicies.last().penaltyDateTime.toInstant()
            if (firstPenaltyDateTime > Instant.now()) {
                throw BronevikRefundRulesException(
                    "Invalid first startAt $firstPenaltyDateTime :  now ${Instant.now()} lower than first cancelationPolicy")
            }

            val builder = RefundRules.builder()
            val rules = mutableListOf<RefundRule>()

            var endsAt: Instant? = null

            for (cancellationPolicy in reversedCancellationPolicies) {
                val penaltyAmount = Math.round(cancellationPolicy.penaltyPriceDetails.clientCurrency.gross.price)
                val startAt = cancellationPolicy.penaltyDateTime.toInstant()

                if (endsAt != null && startAt.compareTo(endsAt) == 0) {
                    throw BronevikRefundRulesException("startAt $startAt is equaling endsAt $endsAt")
                }
                if (penaltyAmount < 0  || penaltyAmount > total) {
                    throw BronevikRefundRulesException("Invalid penalty for cancellation policy, penalty amount $penaltyAmount, total $total")
                }

                val type = when (penaltyAmount) {
                    0 -> FULLY_REFUNDABLE
                    total -> NON_REFUNDABLE
                    else -> REFUNDABLE_WITH_PENALTY
                }

                val refundRuleBuilder = RefundRule.builder()
                    .startsAt(startAt)
                    .type(type)

                if (type == REFUNDABLE_WITH_PENALTY) {
                    refundRuleBuilder.penalty(
                        Money.of(
                            penaltyAmount,
                            ProtoCurrencyUnit.fromCurrencyCode(
                                cancellationPolicy
                                    .penaltyPriceDetails
                                    .clientCurrency
                                    .gross
                                    .currency
                            )
                        )
                    )
                }

                if (endsAt != null) {
                    refundRuleBuilder.endsAt(endsAt)
                }

                rules.add(refundRuleBuilder.build())
                endsAt = startAt
            }

            rules.sortBy { it.startsAt }

            if (rules.last().type != NON_REFUNDABLE) {
                throw BronevikRefundRulesException("Last cancellation policy must NON_REFUNDABLE ${rules.last()}")
            }

            return builder.rules(rules).build()
        }

        private fun XMLGregorianCalendar.toInstant(): Instant {
            return toGregorianCalendar().toInstant()
        }
    }
}
