package ru.yandex.travel.hotels.common.partners.bronevik.utils.timezone;

import java.util.List;
import java.util.NoSuchElementException;

import com.google.protobuf.InvalidProtocolBufferException;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StringField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.TermQuery;
import org.jetbrains.annotations.NotNull;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import ru.yandex.misc.ExceptionUtils;
import ru.yandex.travel.hotels.common.partners.bronevik.proto.THotelTimezone;
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder;
import ru.yandex.travel.yt_lucene_index.LuceneIndexService;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

@Slf4j
public class YtBronevikHotelTimezone extends LuceneIndexService<THotelTimezone> implements BronevikHotelTimezone, HealthIndicator {

    private final String FIELD_ID_INDEX = "originalId";
    private final String FIELD_HOTEL_TIMEZONE_ID = "timezoneId";
    private final String FIELD_PROTO = "pc";

    @Override
    public Health health() {
        if (getLuceneIndex().isReady()) {
            return Health.up().build();
        } else {
            return Health.down().withDetail("MappingNotLoaded", this.getClass()).build();
        }
    }

    private THotelTimezone toHotelTimezone(Document document) {
        try {
            return THotelTimezone.parser().parseFrom(document.getField(FIELD_PROTO).binaryValue().bytes);
        } catch (InvalidProtocolBufferException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @NotNull
    @Override
    protected Iterable<Document> documentProducer(THotelTimezone row) {
        var document = new Document();
        document.add(new StringField(FIELD_ID_INDEX, row.getOriginalId(), Field.Store.YES));
        document.add(new StringField(FIELD_HOTEL_TIMEZONE_ID, row.getTimezoneId(), Field.Store.YES));
        document.add(new StoredField(FIELD_PROTO, row.toByteArray()));
        return List.of(document);
    }

    @Override
    public String getTimezoneID(String hotelId) {
        return getHotelTimezone(hotelId).getTimezoneId();
    }

    @Override
    public THotelTimezone getHotelTimezone(String hotelId) {
        var query = new TermQuery(new Term(FIELD_ID_INDEX, hotelId));
        var result = searchOne(query);
        log.info("New query for BronevikHotelTimezone {}", hotelId);
        if (result == null) {
            throw new NoSuchElementException(String.format("No such originalId %s", hotelId));
        }
        return toHotelTimezone(result);
    }

    public YtBronevikHotelTimezone(@NotNull YtLuceneIndexParams params, @NotNull LuceneIndexBuilder<THotelTimezone> luceneIndexBuilder) {
        super(params, luceneIndexBuilder, "BronevikHotelTimezone");
    }
}
