package ru.yandex.travel.hotels.common.partners.dolphin;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.base.CallContextBaseClient;
import ru.yandex.travel.hotels.common.partners.dolphin.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateResult;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Area;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AreaType;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CheckinServiceModeEnum;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Cost;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CreateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Guest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.GuestName;
import ru.yandex.travel.hotels.common.partners.dolphin.model.HotelContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.IdNameMap;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Offer;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OfferList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrdersInfoRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Pansion;
import ru.yandex.travel.hotels.common.partners.dolphin.model.PriceKey;
import ru.yandex.travel.hotels.common.partners.dolphin.model.SearchResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourService;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EHotelConfirmationOutcome;
import ru.yandex.travel.hotels.proto.EHotelDataLookupOutcome;
import ru.yandex.travel.hotels.proto.EHotelOfferOutcome;
import ru.yandex.travel.hotels.proto.TDolphinOffer;

@Slf4j
public class CallContextDolphinClient extends CallContextBaseClient implements DolphinClient {
    private final DolphinClient baseClient;

    public CallContextDolphinClient(DolphinClient baseClient, CallContext callContext) {
        super(callContext);
        this.baseClient = baseClient;
    }

    @Override
    public CompletableFuture<HotelContent> getHotelContent(String hotelId) {
        return wrapIf(() -> baseClient.getHotelContent(hotelId),
                () -> CompletableFuture.completedFuture(mockHotelContent(hotelId)),
                tc -> tc.getHotelDataLookupOutcome() == EHotelDataLookupOutcome.HO_MOCKED);
    }

    private HotelContent mockHotelContent(String id) {
        HotelContent result = new HotelContent();
        result.setId(Long.parseLong(id));
        result.setName("Тестовый сгенерированный отель");
        result.setType("Отель");
        result.setStars(5);
        result.setCountryId(1);
        result.setRegionId(2);
        result.setAddress("ул Льва Толстого, 16");
        result.setLatitude(55.734399);
        result.setLongitude(37.587293);
        result.setCheckinTime("14:00");
        result.setCheckoutTime("12:00");
        result.setCheckinServiceFrom("00:00");
        result.setCheckinServiceTill("00:00");
        result.setCheckinServiceMode(CheckinServiceModeEnum.CSM_24_7);
        result.setConstructionInfo("Предвечный и несотворенный");
        result.setInfo("Тестовый отель, сгенерирован тестовым контекстом, для тестового тестирования тестировщиками. " +
                "В реальности не существует");
        result.setTransportAccessibility("Два квартала от черной дыры, потом вниз и налево");
        result.setPhotos1020x700(List.of("https://avatars.mds.yandex.net/get-altay/223006/" +
                        "2a0000015b19222d131711047d866611eaf2/XXXL",
                "https://avatars.mds.yandex.net/get-altay" +
                        "/223006/2a0000015b19222d131711047d866611eaf2/XXXL"));
        return result;

    }

    @Override
    public CompletableFuture<IdNameMap> getRooms() {
        return wrapForce(baseClient::getRooms,
                () -> {
                    IdNameMap mocked = new IdNameMap();
                    mocked.put(0L, "");
                    return CompletableFuture.completedFuture(mocked);
                });
    }

    @Override
    public CompletableFuture<IdNameMap> getRoomCategories() {
        return wrapForce(baseClient::getRoomCategories,
                () -> {
                    IdNameMap mocked = new IdNameMap();
                    mocked.put(0L, getTestContext().getOfferName());
                    return CompletableFuture.completedFuture(mocked);
                });
    }

    @Override
    public CompletableFuture<TourContent> getTourContent(String tourId) {
        return wrapForce(() -> baseClient.getTourContent(tourId),
                () -> {
                    TourContent tc = new TourContent();
                    tc.setId(0L);
                    tc.setName("Сгенерированный тур");
                    tc.setRequiredCharges("Доплата будет вообще за все, что можно себе представить");
                    tc.setDocuments("Справка, что вы не верблюд");
                    return CompletableFuture.completedFuture(tc);
                });
    }

    @Override
    public CompletableFuture<Map<AreaType, Map<Long, Area>>> getAreaMap() {
        return wrapIf(baseClient::getAreaMap,
                () -> {
                    Area russia = new Area();
                    russia.setId(1);
                    russia.setTitle("Россия");
                    russia.setType(AreaType.Country);
                    Area moscow = new Area();
                    moscow.setId(2);
                    moscow.setTitle("Москва");
                    moscow.setCityId(1);
                    moscow.setType(AreaType.Region);
                    return CompletableFuture.completedFuture(Map.of(
                            AreaType.Country, Map.of(1L, russia),
                            AreaType.Region, Map.of(2L, moscow))
                    );
                }, tc -> tc.getHotelDataLookupOutcome() == EHotelDataLookupOutcome.HO_MOCKED);
    }

    @Override
    public CompletableFuture<Order> createOrder(CreateOrderRequest request) {
        return wrap(() -> baseClient.createOrder(request),
                () -> CompletableFuture.completedFuture(mockOrder(UUID.randomUUID().toString(),
                        getTestContext().getConfirmationOutcome(), OrderState.IN_WORK)));
    }

    @Override
    public CompletableFuture<Map<Long, Pansion>> getPansionMap() {
        return wrapForce(baseClient::getPansionMap,
                () -> {
                    Pansion p = new Pansion();
                    p.setId(0L);
                    p.setName("Тестовый пансион");
                    switch (getTestContext().getPansionType()) {
                        case PT_BB:
                            p.setType(1);
                            break;
                        case PT_HB:
                            p.setType(2);
                            break;
                        case PT_FB:
                            p.setType(3);
                            break;
                        case PT_AI:
                            p.setType(8);
                            break;
                        case PT_BD:
                            p.setType(10);
                            break;
                    }
                    return CompletableFuture.completedFuture(Map.of(0L, p));
                }
        );
    }

    @Override
    public CompletableFuture<SearchResponse> searchCheckins(LocalDate checkin, LocalDate checkout,
                                                            Occupancy occupancy, List<String> hotelIds,
                                                            String httpRequestId) {
        return wrapForce(() -> baseClient.searchCheckins(checkin, checkout, occupancy, hotelIds, httpRequestId),
                () -> {
                    log.info("Will return a mocked price availability for hotels {} on {} - {} for {}",
                            hotelIds, checkin, checkout, occupancy);
                    return CompletableFuture.completedFuture(mockSearchResponse(checkin, checkout, occupancy,
                            hotelIds));
                });
    }

    private SearchResponse mockSearchResponse(LocalDate checkin, LocalDate checkout, Occupancy occupancy,
                                              List<String> hotelIds) {
        SearchResponse response = new SearchResponse();
        response.setOfferLists(new ArrayList<>());
        List<Integer> beds = new ArrayList<>(occupancy.getAdults() + occupancy.getChildren().size());
        for (int i = 0; i < occupancy.getAdults(); i++) {
            beds.add(1);
        }
        for (var ignored : occupancy.getChildren()) {
            beds.add(3);
        }

        for (var hotelId : hotelIds) {
            OfferList offerList = new OfferList();
            offerList.setPansionId(0L);
            offerList.setTourId(0L);
            offerList.setDate(checkin);
            offerList.setNights((int) ChronoUnit.DAYS.between(checkin, checkout));
            offerList.setHotelId(Long.parseLong(hotelId));
            Offer offer = new Offer();
            offer.setRoomId(0L);
            offer.setCategoryId(0L);
            offer.setQuoted(true);
            offer.setPrice(getTestContext().getPriceAmount());
            offer.setBeds(beds);
            offer.setPlaces(10);
            offer.setRate("рб");
            offerList.setOffers(List.of(offer));
            response.getOfferLists().add(offerList);
        }
        return response;
    }

    @Override
    public CompletableFuture<CalculateOrderResponse> calculateOrder(CalculateOrderRequest request) {
        return wrap(() -> baseClient.calculateOrder(request),
                () -> {
                    switch (callContext.getPhase()) {
                        case OFFER_VALIDATION:
                            return CompletableFuture.completedFuture(mockCalculateResponse(getTestContext().getGetOfferOutcome()));
                        case ORDER_CREATION:
                            return CompletableFuture.completedFuture(mockCalculateResponse(getTestContext().getCreateOrderOutcome()));
                        default:
                            throw new IllegalStateException();
                    }
                }
        );
    }

    protected CalculateOrderResponse mockCalculateResponse(EHotelOfferOutcome outcome) {
        double price;
        Offer offer = getOffer();
        switch (outcome) {
            case OO_SUCCESS:
                price = offer.getPrice();
                break;
            case OO_SOLD_OUT:
                throw new SoldOutException();
            case OO_PRICE_MISMATCH:
                price = offer.getPrice() * getTestContext().getPriceMismatchRate();
                break;
            default:
                throw new IllegalStateException();
        }
        final double feeRate = 0.13;
        final double brutto = price * (1 - feeRate);
        final double fee = price * feeRate;
        CalculateOrderResponse response = new CalculateOrderResponse();
        Cost cost = new Cost();
        cost.setBrutto(brutto);
        cost.setNetto(0.0);
        cost.setFee(fee);
        response.setCost(cost);
        response.setAvailableRooms(10);
        TourService service = new TourService();
        service.setType("Hotel");
        service.setName("Виртуальное проживание в выдуманном отеле");
        service.setBrutto(brutto);
        service.setFee(fee);
        response.setServiceList(List.of(service));
        return response;
    }

    protected Order mockOrder(String code, EHotelConfirmationOutcome outcome, OrderState state) {
        DolphinHotelItinerary itinerary = getItinerary();
        Order response = new Order();
        double price;
        switch (outcome) {
            case CO_SUCCESS:
            case CO_WAITLIST_THEN_ANNULATE:
            case CO_WAITLIST_THEN_OK:
            case CO_WAITLIST_ALWAYS:
                price = itinerary.getFiscalPrice().getNumber().doubleValue();
                break;
            case CO_NOT_FOUND:
                throw new SoldOutException();
            case CO_PRICE_MISMATCH:
                price = itinerary.getFiscalPrice().getNumber().doubleValue() * getTestContext().getPriceMismatchRate();
                break;
            default:
                throw new IllegalStateException();
        }
        final double feeRate = 0.13;
        final double brutto = price * (1 - feeRate);
        final double fee = price * feeRate;
        response.setState(state);
        response.setCode(code);
        Cost cost = new Cost();
        cost.setBrutto(brutto);
        cost.setNetto(0.0);
        cost.setFee(fee);
        response.setCost(cost);
        PriceKey key = new PriceKey(itinerary.getHotelId(), itinerary.getTourId(), itinerary.getPansionId(),
                itinerary.getRoomId(), itinerary.getRoomCatId(), itinerary.getCheckinDate(), itinerary.getNights(),
                itinerary.getBeds());
        List<Guest> guests = itinerary.getGuests().stream().map(g -> {
            var guest = new Guest();
            guest.setIsMale(true);
            var name = new GuestName(g.getFirstName(), g.getLastName(), null);
            guest.setCyrillic(name);
            guest.setLatin(name);
            return guest;
        }).collect(Collectors.toList());
        CreateOrderRequest request = new CreateOrderRequest(itinerary.getOccupancy().getAdults(), key,
                itinerary.getOccupancy().getChildren(), guests, itinerary.getCustomerEmail(),
                itinerary.getCustomerPhone());
        response.setRequest(request);
        TourService service = new TourService();
        service.setType("Hotel");
        service.setName("Виртуальное проживание в выдуманном отеле");
        service.setBrutto(brutto);
        service.setFee(fee);
        response.setServiceList(List.of(service));
        return response;
    }


    @Override
    public CompletableFuture<AnnulateResult> annulateOrder(String id) {
        return wrap(() -> baseClient.annulateOrder(id),
                () -> CompletableFuture.completedFuture(new AnnulateResult(true))
        );
    }

    @Override
    public CompletableFuture<OrderList> getOrders(OrdersInfoRequest request) {
        return wrap(() -> baseClient.getOrders(request),
                () -> {
                    var res = new OrderList();
                    if (StringUtils.isNotBlank(getTestContext().getDolphinExistingOrderId())) {
                        res.add(mockOrder(getTestContext().getDolphinExistingOrderId(),
                                EHotelConfirmationOutcome.CO_SUCCESS, OrderState.OK));
                    }
                    return CompletableFuture.completedFuture(res);
                });
    }

    @Override
    public CompletableFuture<Order> getOrder(String id) {
        return wrap(() -> baseClient.getOrder(id),
                () -> {
                    switch (callContext.getPhase()) {
                        case ORDER_CANCELLATION:
                            return CompletableFuture.completedFuture(mockOrder(id,
                                    getTestContext().getConfirmationOutcome(), OrderState.ANNULATED));
                        case ORDER_CONFIRMATION:
                            OrderState state;
                            switch (getTestContext().getConfirmationOutcome()) {
                                case CO_SUCCESS:
                                case CO_PRICE_MISMATCH:
                                case CO_NOT_FOUND:
                                    state = OrderState.OK;
                                    break;
                                case CO_WAITLIST_ALWAYS:
                                    state = OrderState.WAIT_LIST;
                                    break;
                                case CO_WAITLIST_THEN_OK:
                                    if (getItinerary().getManualTicketId() == null) {
                                        state = OrderState.WAIT_LIST;
                                    } else {
                                        state = OrderState.OK;
                                    }
                                    break;
                                case CO_WAITLIST_THEN_ANNULATE:
                                    if (getItinerary().getManualTicketId() == null) {
                                        state = OrderState.WAIT_LIST;
                                    } else {
                                        state = OrderState.ANNULATED;
                                    }
                                    break;
                                default:
                                    throw new IllegalStateException();
                            }
                            return CompletableFuture.completedFuture(mockOrder(id,
                                    getTestContext().getConfirmationOutcome(), state));
                        case ORDER_REFUND:
                            return CompletableFuture.completedFuture(mockOrder(id,
                                    EHotelConfirmationOutcome.CO_SUCCESS, OrderState.OK));
                        default:
                            throw new IllegalStateException();
                    }
                });
    }

    private Offer getOffer() {
        Preconditions.checkState(callContext.getOfferData() != null,
                "Offer data is not passed to call context");
        Preconditions.checkState(callContext.getOfferData() instanceof TDolphinOffer, "Invalid offer data");
        return ProtoUtils.fromTJson(((TDolphinOffer) callContext.getOfferData()).getShoppingOffer(), Offer.class);
    }

    private DolphinHotelItinerary getItinerary() {
        Preconditions.checkState(callContext.getItinerary() != null,
                "Itinerary is not passes to call context");
        Preconditions.checkState(callContext.getItinerary() instanceof DolphinHotelItinerary, "Invalid itinerary");
        return (DolphinHotelItinerary) callContext.getItinerary();
    }


}
