package ru.yandex.travel.hotels.common.partners.dolphin;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Getter;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.base.IgnoredBodyResponse;
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnexpectedHttpStatusCodeException;
import ru.yandex.travel.hotels.common.partners.dolphin.exceptions.SoldOutException;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateResult;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Area;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AreaList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AreaType;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CreateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.HotelContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.IdNameMap;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrdersInfoRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Pansion;
import ru.yandex.travel.hotels.common.partners.dolphin.model.PansionList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.SearchRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.SearchResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourContent;
import ru.yandex.travel.hotels.common.token.Occupancy;

public class DefaultDolphinClient extends BaseClient<DolphinClientProperties> implements DolphinClient {
    private static final String SEARCH_CHECKINS = "SearchCheckins";
    private static final String CALCULATE_ORDER = "CalculateOrder";
    private static final String HOTEL_CONTENT = "HotelContent";
    private static final String TOUR_CONTENT = "TourContent";
    private static final String ROOMS = "Rooms";
    private static final String ROOM_CATEGORIES = "RoomCategories";
    private static final String AREAS = "Areas";
    private static final String PANSIONS = "Pansions";
    private static final String CREATE_ORDER = "CreateOrder";
    private static final String ORDERS_INFO = "OrdersInfo";
    private static final String ANNULATE_ORDER = "AnnulateOrder";
    @Getter
    private static final ClientMethods methods;

    static {
        methods = new ClientMethods()
            .register(SEARCH_CHECKINS, "/SearchPartnerCheckins", HttpMethod.POST, SearchResponse.class)
            .register(CALCULATE_ORDER, "/CalculateOrder", HttpMethod.POST, CalculateOrderResponse.class)
            .register(HOTEL_CONTENT, "/HotelContent", HttpMethod.GET, HotelContent.class)
            .register(TOUR_CONTENT, "/TourContent", HttpMethod.GET, TourContent.class)
            .register(ROOMS, "/Rooms", HttpMethod.GET, IdNameMap.class)
            .register(ROOM_CATEGORIES, "/RoomCategories", HttpMethod.GET, IdNameMap.class)
            .register(AREAS, "/Areas", HttpMethod.GET, AreaList.class)
            .register(PANSIONS, "/Pansions", HttpMethod.GET, PansionList.class)
            .register(CREATE_ORDER, "/CreateOrder", HttpMethod.POST, Order.class)
            .register(ORDERS_INFO, "/OrdersInfo", HttpMethod.POST, OrderList.class)
            .register(ANNULATE_ORDER, "/AnnulateOrder", HttpMethod.POST, IgnoredBodyResponse.class, Set.of(200, 403));
    }

    public DefaultDolphinClient(AsyncHttpClientWrapper clientWrapper, DolphinClientProperties properties, Retry retryHelper) {
        super(properties, clientWrapper, createObjectMapper(), retryHelper);
    }

    public static ObjectMapper createObjectMapper() {
        var mapper = new ObjectMapper();
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    @Override
    public CompletableFuture<HotelContent> getHotelContent(String hotelId) {
        return call(HOTEL_CONTENT, new CallArguments().withQueryParam("id", hotelId));
    }

    @Override
    public CompletableFuture<IdNameMap> getRooms() {
        return call(ROOMS);
    }

    @Override
    public CompletableFuture<IdNameMap> getRoomCategories() {
        return call(ROOM_CATEGORIES);
    }

    @Override
    public CompletableFuture<TourContent> getTourContent(String tourId) {
        return call(TOUR_CONTENT, new CallArguments().withQueryParam("id", tourId));
    }

    @Override
    public CompletableFuture<Map<AreaType, Map<Long, Area>>> getAreaMap() {
        CompletableFuture<AreaList> res = call(AREAS);
        return res.thenApply(AreaList::toMap);
    }

    @Override
    public CompletableFuture<Order> createOrder(CreateOrderRequest request) {
        request.authorize(properties.getLogin(), properties.getPassword());
        CompletableFuture<Order> res =  call(CREATE_ORDER, new CallArguments().withBody(request));
        return FutureUtils.handleExceptionOfType(res, UnexpectedHttpStatusCodeException.class, r -> {
            if (r.getStatusCode() == 403 && r.getResponseBody().startsWith("Цена для расселения устарела")) {
                throw new SoldOutException();
            } else {
                throw r;
            }
        });
    }

    @Override
    public CompletableFuture<Map<Long, Pansion>> getPansionMap() {
        CompletableFuture<PansionList> res = call(PANSIONS);
        return res.thenApply(PansionList::toMap);
    }

    @Override
    public CompletableFuture<SearchResponse> searchCheckins(LocalDate checkin, LocalDate checkout, Occupancy occupancy,
                                                            List<String> hotelIds, String httpRequestId) {
        var request = SearchRequest.build(checkin, checkout, occupancy.getAdults(), occupancy.getChildren(), hotelIds,
            properties.getLogin(), properties.getPassword());
        return call(SEARCH_CHECKINS, new CallArguments().withBody(request).withRequestId(httpRequestId));
    }

    @Override
    public CompletableFuture<CalculateOrderResponse> calculateOrder(CalculateOrderRequest request) {
        request.authorize(properties.getLogin(), properties.getPassword());
        CompletableFuture<CalculateOrderResponse> res = call(CALCULATE_ORDER, new CallArguments().withBody(request));
        return FutureUtils.handleExceptionOfType(res, UnexpectedHttpStatusCodeException.class, r -> {
            if (r.getStatusCode() == 403 && r.getResponseBody().startsWith("Цена для расселения устарела")) {
                throw new SoldOutException();
            } else {
                throw r;
            }
        });
    }

    @Override
    public CompletableFuture<AnnulateResult> annulateOrder(String id) {
        var request = new AnnulateRequest(id);
        request.authorize(properties.getLogin(), properties.getPassword());
        CompletableFuture<IgnoredBodyResponse> res = call(ANNULATE_ORDER, new CallArguments().withBody(request));
        return res.thenApply(rsp -> new AnnulateResult(rsp.getStatusCode() == 200));
    }

    @Override
    public CompletableFuture<OrderList> getOrders(OrdersInfoRequest request) {
        request.authorize(properties.getLogin(), properties.getPassword());
        return call(ORDERS_INFO, new CallArguments().withBody(request));
    }

    @Override
    public CompletableFuture<Order> getOrder(String id) {
        return getOrders(OrdersInfoRequest.builder().code(id).build()).thenApply(r -> r.stream().findFirst().orElse(null));
    }

    @Override
    protected ClientMethods getClientMethods() {
        return methods;
    }
}
