package ru.yandex.travel.hotels.common.partners.dolphin;

import java.time.LocalDate;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.base.ForcedTestScenario;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AnnulateResult;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Area;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AreaType;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CalculateOrderResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.CreateOrderRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.HotelContent;
import ru.yandex.travel.hotels.common.partners.dolphin.model.IdNameMap;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Order;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrderState;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OrdersInfoRequest;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Pansion;
import ru.yandex.travel.hotels.common.partners.dolphin.model.SearchResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.model.TourContent;
import ru.yandex.travel.hotels.common.token.Occupancy;

@Slf4j
public class WrappedDolphinClient implements DolphinClient {
    private final int MAX_MOCKED_REQUESTS = 100;
    private final DolphinClient wrappedClient;
    private final LRUMap mockedOrders;

    public WrappedDolphinClient(DolphinClient wrappedClient) {
        this.wrappedClient = wrappedClient;
        this.mockedOrders = new LRUMap();
    }

    @Override
    public CompletableFuture<HotelContent> getHotelContent(String hotelId) {
        return wrappedClient.getHotelContent(hotelId);
    }

    @Override
    public CompletableFuture<IdNameMap> getRooms() {
        return wrappedClient.getRooms();
    }

    @Override
    public CompletableFuture<IdNameMap> getRoomCategories() {
        return wrappedClient.getRoomCategories();
    }

    @Override
    public CompletableFuture<TourContent> getTourContent(String tourId) {
        return wrappedClient.getTourContent(tourId);
    }

    @Override
    public CompletableFuture<Map<AreaType, Map<Long, Area>>> getAreaMap() {
        return wrappedClient.getAreaMap();
    }

    @Override
    public CompletableFuture<Order> createOrder(CreateOrderRequest request) {
        ForcedTestScenario test = ForcedTestScenario.get(
                request.getGuests().get(0).getCyrillic().getFirstName(),
                request.getGuests().get(0).getCyrillic().getLastName());
        switch (test) {
            case NO_BOOKING_ON_CONFIRM:
            case SOLD_OUT_ON_HOLD:
                logMock(test, "create");
                String orderId = ProtoUtils.randomId();
                Order mockedWaitListOrder = new Order();
                mockedWaitListOrder.setState(OrderState.WAIT_LIST);
                mockedWaitListOrder.setCode(orderId);
                mockedOrders.put(orderId, Tuple2.tuple(test, mockedWaitListOrder));
                return CompletableFuture.completedFuture(mockedWaitListOrder);
            case PRICE_MISMATCH_ON_CONFIRM:
            case PRICE_MISMATCH_ON_HOLD:
            case MINOR_PRICE_MISMATCH_ON_HOLD:
            case MINOR_PRICE_MISMATCH_ON_CONFIRM:
                double multiplier;
                if (test == ForcedTestScenario.MINOR_PRICE_MISMATCH_ON_HOLD || test == ForcedTestScenario.MINOR_PRICE_MISMATCH_ON_CONFIRM) {
                    multiplier = 1.04;
                } else {
                    multiplier = 1.1;
                }
                logMock(test, "create");
                return wrappedClient.calculateOrder(CalculateOrderRequest.builder()
                        .adults(request.getAdults())
                        .children(request.getChildren())
                        .priceKey(request.getPriceKey())
                        .build())
                        .thenApply(calculateOrderResponse -> {
                                    String mockedOrderId = ProtoUtils.randomId();
                                    Order mockedMismatchedOrder = new Order();
                                    mockedMismatchedOrder.setState(OrderState.OK);
                                    mockedMismatchedOrder.setCode(mockedOrderId);
                                    mockedMismatchedOrder.setCost(calculateOrderResponse.getCost());
                                    mockedMismatchedOrder.getCost().setBrutto(mockedMismatchedOrder.getCost().getBrutto() * multiplier);
                                    mockedOrders.put(mockedOrderId, Tuple2.tuple(test, mockedMismatchedOrder));
                                    return mockedMismatchedOrder;
                                });
        }
        return wrappedClient.createOrder(request);
    }

    @Override
    public CompletableFuture<Map<Long, Pansion>> getPansionMap() {
        return wrappedClient.getPansionMap();
    }

    @Override
    public CompletableFuture<SearchResponse> searchCheckins(LocalDate checkin, LocalDate checkout,
                                                            Occupancy occupancy, List<String> hotelIds,
                                                            String httpRequestId) {
        return wrappedClient.searchCheckins(checkin, checkout, occupancy, hotelIds, httpRequestId);
    }

    @Override
    public CompletableFuture<CalculateOrderResponse> calculateOrder(CalculateOrderRequest request) {
        return wrappedClient.calculateOrder(request);
    }

    @Override
    public CompletableFuture<AnnulateResult> annulateOrder(String id) {
        if (mockedOrders.containsKey(id)) {
            var tuple = mockedOrders.get(id);
            switch (tuple.get1()) {
                case NO_BOOKING_ON_CONFIRM:
                case SOLD_OUT_ON_HOLD:
                case PRICE_MISMATCH_ON_CONFIRM:
                case PRICE_MISMATCH_ON_HOLD:
                case MINOR_PRICE_MISMATCH_ON_HOLD:
                case MINOR_PRICE_MISMATCH_ON_CONFIRM:
                    logMock(tuple.get1(), "annulate");
                    tuple.get2().setState(OrderState.ANNULATED);
                    return CompletableFuture.completedFuture(new AnnulateResult(true));
                default:
                    return wrappedClient.annulateOrder(id);
            }
        } else {
            return wrappedClient.annulateOrder(id);
        }
    }

    @Override
    public CompletableFuture<OrderList> getOrders(OrdersInfoRequest request) {
        return wrappedClient.getOrders(request);
    }

    @Override
    public CompletableFuture<Order> getOrder(String id) {
        if (mockedOrders.containsKey(id)) {
            var tuple = mockedOrders.get(id);
            switch (tuple.get1()) {
                case NO_BOOKING_ON_CONFIRM:
                case SOLD_OUT_ON_HOLD:
                case PRICE_MISMATCH_ON_HOLD:
                case PRICE_MISMATCH_ON_CONFIRM:
                case MINOR_PRICE_MISMATCH_ON_HOLD:
                case MINOR_PRICE_MISMATCH_ON_CONFIRM:
                    logMock(tuple.get1(), "get");
                    return CompletableFuture.completedFuture(tuple.get2());
            }
        }
        return wrappedClient.getOrder(id);
    }

    private void logMock(ForcedTestScenario scenario, String call) {
        log.warn(String.format("A '%s' scenario forced during '%s' call , will do no real API call, " +
                "will return a mocked object instead", scenario, call));
    }

    class LRUMap extends LinkedHashMap<String, Tuple2<ForcedTestScenario, Order>> {
        @Override
        protected boolean removeEldestEntry(Map.Entry<String, Tuple2<ForcedTestScenario, Order>> eldest) {
            return this.size() >= MAX_MOCKED_REQUESTS;
        }
    }
}
