package ru.yandex.travel.hotels.common.partners.dolphin.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.partners.dolphin.model.AverageNightStayFeeRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.model.DolphinRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.model.FixedPercentFeeRefundParams;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

public class DolphinRefundRulesBuilder {
    private static RefundRules build(Instant searchMoment, Instant checkinMoment,
                                     BigDecimal fullAmount, String currency,
                                     Map<Integer, BigDecimal> penalties) {
        AtomicReference<Instant> lastBorder = new AtomicReference<>(null);
        AtomicReference<RefundRule> lastRule = new AtomicReference<>(null);
        var builder = RefundRules.builder();

        penalties.entrySet().stream().sorted((e1, e2) -> Integer.compare(e2.getKey(), e1.getKey()))
                .filter(e -> !checkinMoment.minus(e.getKey(), ChronoUnit.DAYS).isBefore(searchMoment))
                .forEach(e -> {
                    Instant leftBorder = lastBorder.get();
                    Instant rightBorder =
                            checkinMoment.minus(e.getKey(), ChronoUnit.DAYS);
                    RefundType refundType;
                    if (e.getValue().compareTo(BigDecimal.ZERO) == 0) {
                        refundType = RefundType.FULLY_REFUNDABLE;
                    } else if (e.getValue().compareTo(BigDecimal.ONE) == 0) {
                        refundType = RefundType.NON_REFUNDABLE;
                    } else {
                        refundType = RefundType.REFUNDABLE_WITH_PENALTY;
                    }
                    BigDecimal amount = null;
                    if (refundType == RefundType.REFUNDABLE_WITH_PENALTY) {
                        amount = e.getValue().multiply(fullAmount);
                    }
                    if (refundType == RefundType.NON_REFUNDABLE) {
                        rightBorder = null;
                    }
                    Money refundAmount = null;
                    if (amount != null) {
                        refundAmount = Money.of(amount, ProtoCurrencyUnit.fromCurrencyCode(currency));
                    }
                    RefundRule rule = RefundRule.builder()
                            .startsAt(leftBorder)
                            .endsAt(rightBorder)
                            .type(refundType)
                            .penalty(refundAmount)
                            .build();
                    lastBorder.set(rightBorder);
                    lastRule.set(rule);
                    builder.rule(rule);
                });
        if (lastRule.get() != null && lastRule.get().getType() != RefundType.NON_REFUNDABLE) {
            builder.rule(RefundRule.builder()
                    .startsAt(lastBorder.get())
                    .type(RefundType.NON_REFUNDABLE)
                    .build());
        }
        return builder.build();
    }

    private static RefundRules build(Instant searchMoment, Instant checkinMoment,
                                     BigDecimal fullAmount, int nights, String currency,
                                     int cancellationPenaltyDaysPeriod) {
        var builder = RefundRules.builder();
        boolean isRefundable = false;
        boolean addLast = true;
        Instant penaltyStart = checkinMoment.minus(cancellationPenaltyDaysPeriod, ChronoUnit.DAYS);
        if (!penaltyStart.isBefore(searchMoment)) {
            RefundRule rule = RefundRule.builder()
                    .startsAt(null)
                    .endsAt(penaltyStart)
                    .type(RefundType.FULLY_REFUNDABLE)
                    .penalty(null)
                    .build();
            builder.rule(rule);
            isRefundable = true;
        }
        if (!checkinMoment.isBefore(searchMoment)) {
            if (nights > 1) {
                BigDecimal nightsNum = BigDecimal.valueOf(nights);
                BigDecimal refundAmount = fullAmount.divide(nightsNum, 0, RoundingMode.HALF_UP);
                Money refundMoney = Money.of(refundAmount, ProtoCurrencyUnit.fromCurrencyCode(currency));
                RefundRule rule = RefundRule.builder()
                        .startsAt(penaltyStart)
                        .endsAt(checkinMoment)
                        .type(RefundType.REFUNDABLE_WITH_PENALTY)
                        .penalty(refundMoney)
                        .build();
                builder.rule(rule);
                isRefundable = true;
            } else {
                builder.rule(RefundRule.builder()
                        .startsAt(penaltyStart)
                        .type(RefundType.NON_REFUNDABLE)
                        .build());
                addLast = false;
            }
        }
        if (isRefundable && addLast) {
            builder.rule(RefundRule.builder()
                    .startsAt(checkinMoment)
                    .type(RefundType.NON_REFUNDABLE)
                    .build());
        }
        return builder.build();
    }

    public static RefundRules build(Instant searchMoment, Instant checkinMoment,
                                    BigDecimal fullAmount, String currency,
                                    DolphinRefundParams refundStrategy) {
        if (refundStrategy instanceof AverageNightStayFeeRefundParams) {
            var averageNightsRefund = (AverageNightStayFeeRefundParams) refundStrategy;
            return build(searchMoment, checkinMoment, fullAmount, averageNightsRefund.getNights(), currency,
                    averageNightsRefund.getPenaltyDaysPeriod());
        } else if (refundStrategy instanceof FixedPercentFeeRefundParams) {
            var fixedPercentRefund = (FixedPercentFeeRefundParams) refundStrategy;
            return build(searchMoment, checkinMoment, fullAmount, currency, fixedPercentRefund.getPenalties());
        } else {
            throw new IllegalArgumentException("Unknown DolphinRefundStrategy");
        }
    }
}
