package ru.yandex.travel.hotels.common.partners.expedia;

import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.google.common.base.Preconditions;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.logging.IAsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.base.EmptyResponse;
import ru.yandex.travel.hotels.common.partners.base.exceptions.UnexpectedHttpStatusCodeException;
import ru.yandex.travel.hotels.common.partners.expedia.exceptions.ErrorException;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.CancellationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryList;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryReservationRequest;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ReservationResult;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ResumeReservationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.Error;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.PropertyContent;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.PropertyContentMap;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailabilityList;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomPriceCheck;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.SalesChannel;

@Slf4j
public class DefaultExpediaClient extends BaseClient<ExpediaClientProperties> implements ExpediaClient {
    private static final String PROPERTY_AVAILABILITY = "PROPERTY_AVAILABILITY";
    private static final String PROPERTY_CONTENT = "PROPERTY_CONTENT";
    private static final String PRICE_CHECK = "PRICE_CHECK";
    private static final String CREATE_RESERVATION = "CREATE_RESERVATION";
    private static final String GET_RESERVATION = "GET_RESERVATION";
    private static final String GET_RESERVATION_BY_AFFILIATE_ID = "GET_RESERVATION_BY_AFFILIATE_ID";
    private static final String RESUME_HELD_RESERVATION = "RESUME_HELD_RESERVATION";
    private static final String CANCEL_HELD_RESERVATION = "CANCEL_HELD_RESERVATION";
    private static final String CANCEL_CONFIRMED_RESERVATION = "CANCEL_CONFIRMED_RESERVATION";

    private static final Set<Integer> ERRORS_4XX = Set.of(400, 409, 410);
    private final static Map<String, CancellationDetails.ValidatedFieldType> VALIDATED_FIELD_MAP = Map.of(
            "family_name", CancellationDetails.ValidatedFieldType.LAST_NAME,
            "given_name", CancellationDetails.ValidatedFieldType.FIRST_NAME,
            "phone", CancellationDetails.ValidatedFieldType.PHONE,
            "email", CancellationDetails.ValidatedFieldType.EMAIL
    );
    @Getter
    private static final ClientMethods methods;

    static {
        methods = new ClientMethods()
                .register(PROPERTY_AVAILABILITY, "/properties/availability", HttpMethod.GET,
                        PropertyAvailabilityList.class)
                .register(PROPERTY_CONTENT, "/properties/content", HttpMethod.GET, PropertyContentMap.class)
                .register(PRICE_CHECK, "/properties/{0}/rooms/{1}/rates/{2}",
                        HttpMethod.GET, RoomPriceCheck.class, Set.of(200, 409, 410))
                .register(CREATE_RESERVATION, "/itineraries", HttpMethod.POST, ReservationResult.class, Set.of(201))
                .register(RESUME_HELD_RESERVATION, "/itineraries/{0}", HttpMethod.PUT, EmptyResponse.class,
                        Set.of(202, 204))
                .register(GET_RESERVATION, "/itineraries/{0}", HttpMethod.GET, Itinerary.class)
                .register(GET_RESERVATION_BY_AFFILIATE_ID, "/itineraries", HttpMethod.GET, ItineraryList.class)
                .register(CANCEL_HELD_RESERVATION, "/itineraries/{0}", HttpMethod.DELETE, EmptyResponse.class,
                        Set.of(202, 204))
                .register(CANCEL_CONFIRMED_RESERVATION, "/itineraries/{0}/rooms/{1}", HttpMethod.DELETE,
                        EmptyResponse.class, Set.of(202, 204));
    }

    @Getter
    private final ApiVersion apiVersion;

    public DefaultExpediaClient(ExpediaClientProperties properties, IAsyncHttpClientWrapper clientWrapper,
                                Retry retryHelper) {
        super(properties, clientWrapper, createObjectMapper(), retryHelper);
        this.apiVersion = properties.getDefaultApiVersion();
    }

    private DefaultExpediaClient(ExpediaClientProperties properties, IAsyncHttpClientWrapper clientWrapper,
                                 ObjectMapper objectMapper, Retry retryHelper, ApiVersion apiVersion) {
        super(properties, clientWrapper, objectMapper, retryHelper);
        this.apiVersion = apiVersion;
    }

    public static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(new Jdk8Module());
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(
                LocalDate.class,
                new LocalDateDeserializer(DateTimeFormatter.ISO_DATE));
        javaTimeModule.addSerializer(
                LocalDate.class,
                new LocalDateSerializer(DateTimeFormatter.ISO_DATE));
        mapper.registerModule(javaTimeModule);
        mapper.configure(MapperFeature.DEFAULT_VIEW_INCLUSION, false);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, true);
        mapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
        return mapper;
    }

    @Override
    public ExpediaClient usingApi(ApiVersion version) {
        return new DefaultExpediaClient(properties, clientWrapper, objectMapper, retryHelper, version);
    }

    @Override
    public CompletableFuture<PropertyContent> getPropertyContent(String propertyId, String clientIp, String userAgent,
                                                                 String sessionId) {
        CallArguments arguments = new CallArguments()
                .withQueryParam("language", "ru-RU")
                .withQueryParam("property_id", propertyId)
                .withHeader("Customer-Ip", clientIp)
                .withHeader("User-Agent", userAgent)
                .withHeader("Customer-Session-Id", sessionId);

        if (apiVersion == ApiVersion.V3) {
            arguments = arguments.withQueryParam("supply_source", "expedia");
        }

        try {
            arguments = addProfileToArgument(arguments);
        } catch (IllegalArgumentException ex) {
            return CompletableFuture.failedFuture(ex);
        }
        CompletableFuture<PropertyContentMap> res = call(PROPERTY_CONTENT, arguments);
        return FutureUtils.handleExceptionOfType(res, UnexpectedHttpStatusCodeException.class,
                e -> {
                    if (e.getStatusCode() == 404) {
                        Error error = bodyAs(e.getResponseBody(), Error.class);
                        if ("property_content.not_found".equals(error.getType())) {
                            return new PropertyContentMap();
                        }
                    }
                    throw e;
                }
        ).thenApply(pc -> pc.get(propertyId));
    }

    @Override
    public CompletableFuture<RoomPriceCheck> checkPrice(String propertyId, String roomId, String rateId, String token,
                                                        String clientIp, String userAgent, String sessionId) {
        return call(PRICE_CHECK,
                new CallArguments()
                        .withUrlParams(propertyId, roomId, rateId)
                        .withQueryParam("token", token)
                        .withHeader("Customer-Ip", clientIp)
                        .withHeader("User-Agent", userAgent)
                        .withHeader("Customer-Session-Id", sessionId));
    }

    @Override
    public CompletableFuture<ReservationResult> reserveItinerary(ItineraryReservationRequest reservationRequest,
                                                                 String token, String clientIp, String userAgent,
                                                                 String sessionId) {
        CompletableFuture<ReservationResult> res = call(CREATE_RESERVATION,
                new CallArguments()
                        .withBody(reservationRequest)
                        .withQueryParam("token", token)
                        .withHeader("Customer-Ip", clientIp)
                        .withHeader("User-Agent", userAgent)
                        .withHeader("Customer-Session-Id", sessionId));
        return FutureUtils.handleExceptionOfType(res, UnexpectedHttpStatusCodeException.class,
                t -> {
                    UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                            UnexpectedHttpStatusCodeException.class);
                    if (ERRORS_4XX.contains(ex.getStatusCode())) {
                        Error error = bodyAs(ex.getResponseBody(), Error.class);
                        throw new ErrorException(error);
                    }
                    throw ex;
                });
    }

    @Override
    public CompletableFuture<ResumeReservationStatus> resumeItinerary(String itineraryId, String token, String clientIp,
                                                                      String userAgent, String sessionId) {
        CompletableFuture<EmptyResponse> res = call(RESUME_HELD_RESERVATION,
                new CallArguments()
                        .withUrlParams(itineraryId)
                        .withQueryParam("token", token)
                        .withHeader("Customer-Ip", clientIp)
                        .withHeader("User-Agent", userAgent)
                        .withHeader("Customer-Session-Id", sessionId));
        return res.thenApply(emptyResponse -> {
            switch (emptyResponse.getStatusCode()) {
                case 202:
                    return ResumeReservationStatus.UNKNOWN;
                case 204:
                    return ResumeReservationStatus.SUCCESS;
                default:
                    throw new UnexpectedHttpStatusCodeException(emptyResponse.getStatusCode(), null);
            }
        }).exceptionally(t -> {
            UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                    UnexpectedHttpStatusCodeException.class);
            if (ex.getStatusCode() == 400 || ex.getStatusCode() == 404) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("resume.already_resumed".equals(error.getType())) {
                    return ResumeReservationStatus.DUPLICATE;
                }
                if ("resume.rolled_back".equals(error.getType())) {
                    return ResumeReservationStatus.ROLLED_BACK;
                }
                if ("resource_not_found".equals(error.getType())) {
                    return ResumeReservationStatus.NOT_FOUND;
                }
            }
            throw ex;
        });
    }

    @Override
    public CompletableFuture<Itinerary> getItinerary(String itineraryId, String token, String clientIp,
                                                     String userAgent, String sessionId) {
        CompletableFuture<Itinerary> res = call(GET_RESERVATION, new CallArguments()
                .withUrlParams(itineraryId)
                .withQueryParam("token", token)
                .withHeader("Customer-Ip", clientIp)
                .withHeader("User-Agent", userAgent)
                .withHeader("Customer-Session-Id", sessionId));
        return res.exceptionally(t -> {
            UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                    UnexpectedHttpStatusCodeException.class);
            if (ex.getStatusCode() == 404) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("resource_not_found".equals(error.getType())) {
                    return null;
                }
            }
            throw ex;
        });
    }

    @Override
    public CompletableFuture<Itinerary> getItineraryByAffiliateId(String affiliateId, String email,
                                                                  String clientIp, String userAgent,
                                                                  String sessionId) {
        CallArguments arguments = new CallArguments()
                .withQueryParam("affiliate_reference_id", affiliateId)
                .withQueryParam("email", email)
                .withHeader("Customer-Ip", clientIp)
                .withHeader("User-Agent", userAgent)
                .withHeader("Customer-Session-Id", sessionId);
        try {
            arguments = addProfileToArgument(arguments);
        } catch (IllegalArgumentException ex) {
            return CompletableFuture.failedFuture(ex);
        }

        CompletableFuture<ItineraryList> res = call(GET_RESERVATION_BY_AFFILIATE_ID, arguments);
        CompletableFuture<Itinerary> mapped = res.thenApply(r -> {
            if (r.size() == 0) {
                return null;
            } else {
                Preconditions.checkState(r.size() == 1,
                        String.format("Unexpected response size: expected 0 or 1 objects, got %d", r.size()));
                return r.get(0);
            }
        });
        return FutureUtils.handleExceptionOfType(mapped, UnexpectedHttpStatusCodeException.class, ex -> {
            if (ex.getStatusCode() == 404 || ex.getStatusCode() == 400) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("resource_not_found".equals(error.getType())) {
                    return null;
                }
                if ("invalid_input".equals(error.getType())) {
                    log.warn("Invalid input for {}: affiliateId={}, email={}, will return no itinerary",
                            GET_RESERVATION_BY_AFFILIATE_ID, affiliateId, email);
                    return null;
                }
            }
            throw ex;
        });
    }

    @Override
    public CompletableFuture<CancellationStatus> cancelItinerary(String itineraryId, String token, String clientIp,
                                                                 String userAgent, String sessionId) {
        CompletableFuture<EmptyResponse> res = call(CANCEL_HELD_RESERVATION,
                new CallArguments()
                        .withUrlParams(itineraryId)
                        .withQueryParam("token", token)
                        .withHeader("Customer-Ip", clientIp)
                        .withHeader("User-Agent", userAgent)
                        .withHeader("Customer-Session-Id", sessionId));
        return res.thenApply(emptyResponse -> {
            switch (emptyResponse.getStatusCode()) {
                case 202:
                    return CancellationStatus.UNKNOWN;
                case 204:
                    return CancellationStatus.SUCCESS;
                default:
                    throw new UnexpectedHttpStatusCodeException(emptyResponse.getStatusCode(), null);
            }
        }).exceptionally(t -> {
            UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                    UnexpectedHttpStatusCodeException.class);
            if (ex.getStatusCode() == 404) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("resource_not_found".equals(error.getType())) {
                    return CancellationStatus.NOT_FOUND;
                }
            }
            throw ex;
        });
    }

    @Override
    public CompletableFuture<Map<String, PropertyAvailability>> findAvailabilities(List<String> hotelIds,
                                                                                   LocalDate checkin,
                                                                                   LocalDate checkout, String occupancy,
                                                                                   String currency,
                                                                                   boolean includeFencedRates,
                                                                                   String clientIp,
                                                                                   String sessionId,
                                                                                   String requestId,
                                                                                   SalesChannel salesChannel) {
        CallArguments callArguments = new CallArguments()
                .withQueryParam("checkin", checkin.toString())
                .withQueryParam("checkout", checkout.toString())
                .withQueryParam("currency", currency)
                .withQueryParam("occupancy", occupancy)
                .withQueryParam("country_code", "RU")
                .withQueryParam("language", "ru-RU")
                .withQueryParam("sales_channel", salesChannel.getValue())
                .withQueryParam("sales_environment", "hotel_only")
                .withQueryParam("sort_type", "preferred")
                .withQueryParam("rate_plan_count", "250")
                .withHeader("Customer-Ip", clientIp)
                .withHeader("Customer-Session-Id", sessionId)
                .withRequestId(requestId);
        try {
            callArguments = addProfileToArgument(callArguments);
        } catch (IllegalArgumentException ex) {
            return CompletableFuture.failedFuture(ex);
        }

        if (includeFencedRates) {
            callArguments.withQueryParam("rate_option", "member");
        }


        CallArguments finalCallArguments = callArguments;
        hotelIds.forEach(id -> finalCallArguments.withQueryParam("property_id", id));
        CompletableFuture<PropertyAvailabilityList> res = call(PROPERTY_AVAILABILITY, callArguments);
        return res.thenApply(paList -> paList.stream()
                .collect(Collectors.toMap(PropertyAvailability::getPropertyId, i -> i)))
                .exceptionally(t -> {
                    UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                            UnexpectedHttpStatusCodeException.class);
                    if (ex.getStatusCode() == 404) {
                        Error error = bodyAs(ex.getResponseBody(), Error.class);
                        if ("no_availability".equals(error.getType()) ||
                                "availability.not_found".equals(error.getType())) {
                            return Collections.emptyMap();
                        }
                    }
                    throw ex;
                });
    }

    @Override
    public CompletableFuture<CancellationStatus> cancelConfirmedItinerary(String itineraryId, String token,
                                                                          String refundId, String clientIp,
                                                                          String userAgent, String sessionId) {
        CompletableFuture<EmptyResponse> res = call(CANCEL_CONFIRMED_RESERVATION, new CallArguments()
                .withUrlParams(itineraryId, refundId)
                .withQueryParam("token", token)
                .withHeader("Customer-Ip", clientIp)
                .withHeader("User-Agent", userAgent)
                .withHeader("Customer-Session-Id", sessionId));
        return res.thenApply(emptyResponse -> {
            switch (emptyResponse.getStatusCode()) {
                case 202:
                    return CancellationStatus.UNKNOWN;
                case 204:
                    return CancellationStatus.SUCCESS;
                default:
                    throw new UnexpectedHttpStatusCodeException(emptyResponse.getStatusCode(), null);
            }
        }).exceptionally(t -> {
            UnexpectedHttpStatusCodeException ex = unwrapExceptionAs(t,
                    UnexpectedHttpStatusCodeException.class);
            if (ex.getStatusCode() == 400) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("room_already_cancelled".equals(error.getType())) {
                    return CancellationStatus.ALREADY_CANCELLED;
                }
                if ("cancel.post_checkin".equals(error.getType())) {
                    return CancellationStatus.POST_CHECKIN_CANCEL;
                }
                if ("cancel.post_checkout".equals(error.getType())) {
                    return CancellationStatus.POST_CHECKOUT_CANCEL;
                }
            }
            if (ex.getStatusCode() == 404) {
                Error error = bodyAs(ex.getResponseBody(), Error.class);
                if ("resource_not_found".equals(error.getType())) {
                    return CancellationStatus.ALREADY_CANCELLED;
                }
            }
            throw ex;
        });
    }

    @Override
    protected ClientMethods getClientMethods() {
        return methods;
    }

    @Override
    protected @NotNull Map<String, String> getCommonHeaders(String destinationMethod) {
        return Map.of(
                "Accept", "application/json",
                "Accept-Encoding", "gzip",
                "Authorization", getAuthenticationString());
    }

    @Override
    protected String getEndpoint(String method, List<String> urlParams) {
        String endpoint = super.getEndpoint(method, urlParams);

        return String.format("/%s%s", apiVersion.getValue(), endpoint);
    }

    private String getAuthenticationString() {
        Long timestamp = System.currentTimeMillis() / 1000;
        String signature;
        try {
            String toBeHashed = properties.getApiKey() + properties.getApiSecret() + timestamp;
            MessageDigest md = MessageDigest.getInstance("SHA-512");
            byte[] bytes = md.digest(toBeHashed.getBytes("UTF-8"));
            StringBuilder sb = new StringBuilder();
            for (byte aByte : bytes) {
                sb.append(Integer.toString((aByte & 0xff) + 0x100, 16).substring(1));
            }
            signature = sb.toString();
        } catch (NoSuchAlgorithmException | UnsupportedEncodingException e) {
            throw new IllegalStateException("Failed to generate Expedia Authorization header", e);
        }
        return "EAN APIKey=" + properties.getApiKey() + ",Signature=" + signature + ",timestamp=" + timestamp;
    }

    private BaseClient.CallArguments addProfileToArgument(BaseClient.CallArguments arguments) {
        switch (properties.getProfileType()) {
            case EAC_SA:
                return arguments
                        .withQueryParam("payment_terms", "EAC_SA")
                        .withQueryParam("billing_terms", "EAC_SA")
                        .withQueryParam("partner_point_of_sale", "EAC_SA");

            case STANDALONE:
                return arguments
                        .withQueryParam("payment_terms", "EAC_SA")
                        .withQueryParam("partner_point_of_sale", "Standalone");
            default:
                throw new IllegalArgumentException("Unsupported profile");
        }
    }
}
