package ru.yandex.travel.hotels.common.partners.expedia;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import ru.yandex.travel.hotels.common.partners.expedia.model.booking.CancellationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ResumeReservationStatus;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryReservationRequest;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ReservationResult;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.PropertyContent;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomPriceCheck;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.SalesChannel;

import static ru.yandex.travel.hotels.common.partners.base.Helpers.sync;

public interface ExpediaClient {
    CompletableFuture<PropertyContent> getPropertyContent(String propertyId, String clientIp, String userAgent,
                                                          String sessionId);

    CompletableFuture<RoomPriceCheck> checkPrice(String propertyId, String roomId, String rateId, String token,
                                                 String clientIp, String userAgent, String sessionId);

    CompletableFuture<ReservationResult> reserveItinerary(ItineraryReservationRequest reservationRequest,
                                                          String token,
                                                          String clientIp, String userAgent, String sessionId);

    CompletableFuture<ResumeReservationStatus> resumeItinerary(String itineraryId, String token, String clientIp,
                                                               String userAgent, String sessionId);

    CompletableFuture<Itinerary> getItinerary(String itineraryId, String token,
                                              String clientIp, String userAgent, String sessionId);

    CompletableFuture<Itinerary> getItineraryByAffiliateId(String affiliateId, String email,
                                                           String clientIp, String userAgent, String sessionId);

    CompletableFuture<CancellationStatus> cancelItinerary(String itineraryId, String token, String clientIp,
                                                          String userAgent, String sessionId);

    CompletableFuture<Map<String, PropertyAvailability>> findAvailabilities(List<String> hotelIds, LocalDate checkin,
                                                                            LocalDate checkout,
                                                                            String occupancy, String currency,
                                                                            boolean includeFencedRates, String clientIp,
                                                                            String sessionId,
                                                                            String requestId,
                                                                            SalesChannel salesChannel);

    CompletableFuture<CancellationStatus> cancelConfirmedItinerary(String itineraryId, String token, String refundId,
                                                                   String clientIp, String userAgent,
                                                                   String sessionId);

    ExpediaClient usingApi(ApiVersion version);

    ApiVersion getApiVersion();

    default PropertyContent getPropertyContentSync(String propertyId, String clientIp, String userAgent,
                                                   String sessionId) {
        return sync(getPropertyContent(propertyId, clientIp, userAgent, sessionId));
    }

    default RoomPriceCheck checkPriceSync(String propertyId, String roomId, String rateId, String token,
                                          String clientIp, String userAgent, String sessionId) {
        return sync(checkPrice(propertyId, roomId, rateId, token, clientIp, userAgent, sessionId));
    }

    default ReservationResult reserveItinerarySync(ItineraryReservationRequest reservationRequest, String token,
                                                   String clientIp, String userAgent, String sessionId) {
        return sync(reserveItinerary(reservationRequest, token, clientIp, userAgent, sessionId));
    }

    default ResumeReservationStatus resumeItinerarySync(String itineraryId, String token, String clientIp,
                                                        String userAgent,
                                                        String sessionId) {
        return sync(resumeItinerary(itineraryId, token, clientIp, userAgent, sessionId));
    }

    default Itinerary getItinerarySync(String itineraryId, String token, String clientIp, String userAgent,
                                       String sessionId) {
        return sync(getItinerary(itineraryId, token, clientIp, userAgent, sessionId));
    }

    default Itinerary getItineraryByAffiliateIdSync(String affiliateId, String email,
                                                    String clientIp, String userAgent, String sessionId) {
        return sync(getItineraryByAffiliateId(affiliateId, email, clientIp, userAgent, sessionId));
    }

    default CancellationStatus cancelItinerarySync(String itineraryId, String token, String clientIp, String userAgent,
                                                   String sessionId) {
        return sync(cancelItinerary(itineraryId, token, clientIp, userAgent, sessionId));
    }

    default Map<String, PropertyAvailability> findAvailabilitiesSync(List<String> hotelIds, LocalDate checkin,
                                                                     LocalDate checkout,
                                                                     String occupancy, String currency, String clientIp,
                                                                     String sessionId,
                                                                     String requestId,
                                                                     SalesChannel salesChannel,
                                                                     boolean includeFencedRates) {
        return sync(findAvailabilities(hotelIds, checkin, checkout, occupancy, currency, includeFencedRates, clientIp,
                sessionId, requestId, salesChannel));
    }

    default CancellationStatus cancelConfirmedItinerarySync(String itineraryId, String token, String refundId,
                                                            String clientIp, String userAgent,
                                                            String sessionId) {
        return sync(cancelConfirmedItinerary(itineraryId, token, refundId, clientIp, userAgent, sessionId));
    }
}
