package ru.yandex.travel.hotels.common.partners.expedia;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PriceItem;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PricingInformation;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingRate;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

public class ExpediaRefundRulesBuilder {

    public static RefundRules build(ShoppingRate rate, String occupancyString, Duration extraInterval) {
        var builder = RefundRules.builder();
        List<RefundRule> rules = new ArrayList<>();
        if (rate.isRefundable()) {
            PricingInformation pricing = rate.getOccupancyPricing().get(occupancyString);
            BigDecimal total = pricing.getTotals().getInclusive().getBillableCurrency().getValue();

            List<BigDecimal> nightPrices = new ArrayList<>();
            for (List<PriceItem> night : pricing.getNightly()) {
                BigDecimal nightSum = BigDecimal.ZERO;
                for (PriceItem item : night) {
                    nightSum = nightSum.add(item.getValue());
                }
                nightPrices.add(nightSum);
            }
            for (var penalty : rate.getCancelPenalties()) {
                BigDecimal penaltyAmount = BigDecimal.ZERO;
                Instant startInstant = penalty.getStart().minus(extraInterval);
                Instant endInstant = penalty.getEnd().minus(extraInterval);

                if (penalty.getPercent() != null) {
                    penaltyAmount = total.multiply(penalty.getPercent());
                } else if (penalty.getNights() != null) {
                    int nights = penalty.getNights();
                    if (nights > nightPrices.size()) {
                        nights = nightPrices.size();
                    }
                    for (int i = 0; i < nights; i++) {
                        penaltyAmount = penaltyAmount.add(nightPrices.get(i));
                    }
                } else {
                    penaltyAmount = penalty.getAmount();
                }
                RefundType type;
                if (penaltyAmount.compareTo(BigDecimal.ZERO) == 0) {
                    type = RefundType.FULLY_REFUNDABLE;
                    penaltyAmount = null;
                } else {
                    if (penaltyAmount.compareTo(total) >= 0) {
                        type = RefundType.NON_REFUNDABLE;
                        penaltyAmount = null;
                    } else {
                        type = RefundType.REFUNDABLE_WITH_PENALTY;
                        penaltyAmount = penaltyAmount.setScale(0, RoundingMode.HALF_UP).setScale(2,
                                RoundingMode.HALF_UP);
                    }
                }
                Money ruleAmount = null;
                if (penaltyAmount != null) {
                    ruleAmount = Money.of(penaltyAmount,
                            ProtoCurrencyUnit.fromCurrencyCode(pricing.getTotals().getInclusive().getBillableCurrency().getCurrency()));
                }
                rules.add(RefundRule.builder()
                        .startsAt(startInstant)
                        .type(type)
                        .endsAt(endInstant)
                        .penalty(ruleAmount)
                        .build());
            }
            rules.sort(Comparator.comparing(RefundRule::getStartsAt));
            if (rules.size() == 0) {
                return builder.build();
            }
            if (rules.get(0).getStartsAt().isAfter(Instant.now())) {
                // first  penalty occurrs in future
                if (rules.get(0).getType() != RefundType.FULLY_REFUNDABLE) {
                    // we have a no-penalty grace period, let's add it
                    builder.rule(RefundRule.builder()
                            .type(RefundType.FULLY_REFUNDABLE)
                            .endsAt(rules.get(0).getStartsAt())
                            .build());
                } else {
                    // need to change first period boundary to null
                    builder.rule(rules.get(0).toBuilder().startsAt(null).build());
                    rules.remove(0);
                }
            }
            RefundRule lastRule = null;
            for (RefundRule rule : rules) {
                if (rule.getType() == RefundType.NON_REFUNDABLE && rule.getEndsAt() != null) {
                    rule = rule.toBuilder().endsAt(null).build();
                }
                builder.rule(rule);
                lastRule = rule;
                if (rule.getType() == RefundType.NON_REFUNDABLE) {
                    return builder.build();
                }
            }
            if (lastRule != null) {
                builder.rule(RefundRule.builder()
                        .type(RefundType.NON_REFUNDABLE)
                        .startsAt(lastRule.getEndsAt())
                        .build());
            }
        }
        return builder.build();
    }
}
