package ru.yandex.travel.hotels.common.partners.expedia;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.IDN;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.google.i18n.phonenumbers.NumberParseException;
import com.google.i18n.phonenumbers.PhoneNumberUtil;
import com.google.i18n.phonenumbers.Phonenumber;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.partners.expedia.exceptions.UnexpectedResponseException;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Phone;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PriceItem;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PricingInformation;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.ValueWithCurrency;
import ru.yandex.travel.hotels.common.partners.expedia.model.content.BedGroup;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PriceItemType;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingBedGroup;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.TotalsItem;

@Slf4j
public class Helpers {
    private static final Pattern priceCheckLinkPattern = Pattern.compile("/(2\\.4|v3)/properties/(\\d+)" +
            "/rooms/(\\d+)/rates/(\\d+)\\?token=(.*)");
    private static final Pattern reservationLinkPattern = Pattern.compile("/(2\\.4|v3)/itineraries\\?token=(.*)");
    private static final Pattern confirmationLinkPattern = Pattern.compile("/(2\\.4|v3)/itineraries/(\\w+)\\?token=(" +
            ".*)");
    private static final Pattern refundLinkPattern = Pattern.compile("/(2\\.4|v3)/itineraries/(\\w+)/rooms/([\\w-]+)" +
            "\\?token=(.*)");
    private static final int MAX_NAME_LENGTH = 60;

    public static String convertEmailDomainNameToAscii(String email) {
        var parts = email.split("@");
        if (parts.length != 2) {
            return email;
        }
        return parts[0] + "@" + IDN.toASCII(parts[1]);
    }

    public static String limitName(String name) {
        if (name.length() > MAX_NAME_LENGTH) {
            return name.substring(0, MAX_NAME_LENGTH);
        }
        return name;
    }

    public static String retrievePriceCheckToken(String priceCheckLink) {
        return retrievePriceCheckToken(priceCheckLink, null, null, null);
    }

    public static String retrievePriceCheckToken(String priceCheckLink, String propertyId, String roomId,
                                                 String rateId) {
        Matcher matcher = priceCheckLinkPattern.matcher(priceCheckLink);
        if (matcher.find()) {
            String matchedPropertyId = matcher.group(2);
            String matchedRoomId = matcher.group(3);
            String matchedRateId = matcher.group(4);
            String token = matcher.group(5);

            if (propertyId != null && !propertyId.equals(matchedPropertyId)) {
                throw new UnexpectedResponseException("Unexpected property id in url");
            }
            if (roomId != null && !roomId.equals(matchedRoomId)) {
                throw new UnexpectedResponseException("Unexpected room id in url");
            }
            if (rateId != null && !rateId.equals(matchedRateId)) {
                throw new UnexpectedResponseException("Unexpected rate id in url");
            }
            return token;
        }
        return null;
    }

    public static String retrieveReservationToken(String bookLink) {
        Matcher matcher = reservationLinkPattern.matcher(bookLink);
        if (matcher.find()) {
            return matcher.group(2);
        }
        return null;
    }

    public static String retrieveConfirmationToken(String resumeLink) {
        return retrieveConfirmationToken(resumeLink, null);
    }

    public static String retrieveConfirmationToken(String resumeLink, String itineraryId) {
        Matcher matcher = confirmationLinkPattern.matcher(resumeLink);
        if (matcher.find()) {
            String matchedItineraryId = matcher.group(2);
            String token = matcher.group(3);
            if (itineraryId != null && !itineraryId.equals(matchedItineraryId)) {
                throw new UnexpectedResponseException("Unexpected itinerary id in url");
            }
            return token;
        }
        return null;
    }

    public static String[] retriveRoomAndRefundToken(String refundLink, String itineraryId) {
        Matcher matcher = refundLinkPattern.matcher(refundLink);
        if (matcher.find()) {
            String matchedItineraryId = matcher.group(2);
            String matchedRoomId = matcher.group(3);
            String token = matcher.group(4);
            if (itineraryId != null && !itineraryId.equals(matchedItineraryId)) {
                throw new UnexpectedResponseException("Unexpected itinerary id in url");
            }
            return new String[]{matchedRoomId, token};
        }
        return null;
    }


    public static List<ShoppingBedGroup> mapBedGroupDetails(List<ShoppingBedGroup> offerBedGroups,
                                                            Collection<BedGroup> contentBedGroups) {
        var bedGroupsByConfiguration =
                contentBedGroups.stream().collect(Collectors.toMap(bg -> new HashSet<>(bg.getConfiguration()),
                        Function.identity()));
        return offerBedGroups.stream().map(bg -> {
            var config = new HashSet<>(bg.getConfiguration());
            var found = bedGroupsByConfiguration.get(config);
            if (found != null) {
                return bg.toBuilder()
                        .id(found.getId())
                        .description(found.getDescription())
                        .build();
            } else {
                return bg.toBuilder().build();
            }
        }).collect(Collectors.toList());
    }

    public static PricingInformation round(PricingInformation pricingInformation) {
        var builder = PricingInformation.builder();
        Map<PriceItemType, BigDecimal> roundingErrors = new HashMap<>();
        List<PriceItem.PriceItemBuilder> baseRateBuilders = new ArrayList<>();

        List<List<PriceItem.PriceItemBuilder>> roundedNightlyBuilders =
                new ArrayList<>(pricingInformation.getNightly().size());
        for (List<PriceItem> night : pricingInformation.getNightly()) {
            List<PriceItem.PriceItemBuilder> roundedNight = new ArrayList<>(night.size());
            roundedNightlyBuilders.add(roundedNight);
            for (PriceItem item : night) {
                PriceItem.PriceItemBuilder roundedItemBuilder = roundItem(item, roundingErrors).toBuilder();
                roundedNight.add(roundedItemBuilder);
                if (item.getType() == PriceItemType.BASE_RATE) {
                    baseRateBuilders.add(roundedItemBuilder);
                }
            }
        }

        builder.stay(pricingInformation.getStay().stream()
                .map(i -> roundItem(i, roundingErrors))
                .collect(Collectors.toList()));

        BigDecimal exclusiveTotalError = roundingErrors.getOrDefault(PriceItemType.BASE_RATE, BigDecimal.ZERO);
        BigDecimal inclusiveTotalError = BigDecimal.ZERO;
        for (BigDecimal error : roundingErrors.values()) {
            inclusiveTotalError = inclusiveTotalError.add(error);
        }
        while (inclusiveTotalError.compareTo(BigDecimal.ONE) >= 0) {
            if (baseRateBuilders.size() == 0) {
                break;
            }
            PriceItem.PriceItemBuilder lastBuilder = baseRateBuilders.get(baseRateBuilders.size() - 1);
            BigDecimal lastNightPrice = lastBuilder.build().getValue();
            if (lastNightPrice.compareTo(BigDecimal.ONE) > 0) {
                lastNightPrice = lastNightPrice.subtract(BigDecimal.ONE);
                lastBuilder.value(lastNightPrice.setScale(2, RoundingMode.HALF_UP));
                inclusiveTotalError = inclusiveTotalError.subtract(BigDecimal.ONE);
                exclusiveTotalError = exclusiveTotalError.subtract(BigDecimal.ONE);
            } else {
                baseRateBuilders.remove(baseRateBuilders.size() - 1);
            }
        }
        BigDecimal exclusiveBillableValue =
                pricingInformation.getTotals().getExclusive().getBillableCurrency().getValue();
        BigDecimal inclusiveBillableValue =
                pricingInformation.getTotals().getInclusive().getBillableCurrency().getValue();

        exclusiveBillableValue = exclusiveBillableValue.add(exclusiveTotalError);
        inclusiveBillableValue = inclusiveBillableValue.add(inclusiveTotalError);
        boolean requestMatchesBillable = pricingInformation.getTotals().getExclusive().getBillableCurrency()
                .equals(pricingInformation.getTotals().getExclusive().getRequestCurrency()) &&
                pricingInformation.getTotals().getInclusive().getBillableCurrency()
                        .equals(pricingInformation.getTotals().getInclusive().getRequestCurrency());


        builder.totals(pricingInformation.getTotals().toBuilder()
                .exclusive(TotalsItem.builder()
                        .billableCurrency(
                                ValueWithCurrency.builder()
                                        .value(exclusiveBillableValue)
                                        .currency(pricingInformation.getTotals().getExclusive().getBillableCurrency().getCurrency())
                                        .build())
                        .requestCurrency(
                                pricingInformation.getTotals().getExclusive().getRequestCurrency() == null ? null :
                                        ValueWithCurrency.builder()
                                                .value(requestMatchesBillable ? exclusiveBillableValue :
                                                        pricingInformation.getTotals().getExclusive().getRequestCurrency().getValue())
                                                .currency(pricingInformation.getTotals().getExclusive().getRequestCurrency().getCurrency())
                                                .build())
                        .build())
                .inclusive(TotalsItem.builder()
                        .billableCurrency(
                                ValueWithCurrency.builder()
                                        .value(inclusiveBillableValue)
                                        .currency(pricingInformation.getTotals().getInclusive().getBillableCurrency().getCurrency())
                                        .build())
                        .requestCurrency(
                                pricingInformation.getTotals().getExclusive().getRequestCurrency() == null ? null :
                                        ValueWithCurrency.builder()
                                                .value(requestMatchesBillable ? inclusiveBillableValue :
                                                        pricingInformation.getTotals().getInclusive().getRequestCurrency().getValue())
                                                .currency(pricingInformation.getTotals().getInclusive().getRequestCurrency().getCurrency())
                                                .build())
                        .build())
                .build());
        builder.nightly(roundedNightlyBuilders.stream().map(
                n -> n.stream().map(PriceItem.PriceItemBuilder::build).collect(Collectors.toList())
        ).collect(Collectors.toList()));
        return builder.build();
    }

    private static PriceItem roundItem(PriceItem item, Map<PriceItemType, BigDecimal> roundingErrors) {
        BigDecimal roundedValue;
        BigDecimal roundedDefault = item.getValue().setScale(0, RoundingMode.HALF_UP);
        BigDecimal roundedUp = item.getValue().setScale(0, RoundingMode.UP);
        BigDecimal roundedDown = item.getValue().setScale(0, RoundingMode.DOWN);
        BigDecimal defaultError = roundedDefault.subtract(item.getValue()); // may be both positive and negative
        BigDecimal upError = roundedUp.subtract(item.getValue());           // always >= 0
        BigDecimal downError = roundedDown.subtract(item.getValue());       // always <= 0
        BigDecimal accumulatedError = roundingErrors.getOrDefault(item.getType(), BigDecimal.ZERO);
        if (accumulatedError.compareTo(BigDecimal.ZERO) == 0) {
            accumulatedError = defaultError;
            roundedValue = roundedDefault;
        } else if (accumulatedError.add(upError).abs().compareTo(BigDecimal.ONE) < 1) { // abs(accomulated_error) <= 1
            accumulatedError = accumulatedError.add(upError);
            roundedValue = roundedUp;
        } else {
            accumulatedError = accumulatedError.add(downError);
            roundedValue = roundedDown;
        }
        roundedValue = roundedValue.setScale(2, RoundingMode.HALF_UP);
        roundingErrors.put(item.getType(), accumulatedError);
        return item.toBuilder().value(roundedValue).build();
    }

    public static Phone parsePhone(String phone) {
        try {
            String phoneWithPlus;
            if (phone.charAt(0) == '8' && phone.length() == 11) {
                phoneWithPlus = String.format("+7%s", phone.substring(1));
            } else {
                phoneWithPlus = String.format("+%s", phone);
            }
            Phonenumber.PhoneNumber number = PhoneNumberUtil.getInstance().parse(phoneWithPlus, "RU");
            if (!number.hasCountryCode() || !number.hasNationalNumber()) {
                log.warn("Unable to detect country code in {}, will fallback to naive country code detection", number);
                return Phone.builder()
                        .countryCode(phone.substring(0, 1))
                        .number(phone.substring(1))
                        .build();
            } else {
                return Phone.builder()
                        .countryCode(String.valueOf(number.getCountryCode()))
                        .number(String.valueOf(number.getNationalNumber()))
                        .build();
            }
        } catch (NumberParseException e) {
            log.warn("Unable to parse phone, will fallback to naive country code detection");
            return Phone.builder()
                    .countryCode(phone.substring(0, 1))
                    .number(phone.substring(1))
                    .build();
        }

    }
}
