package ru.yandex.travel.hotels.common.partners.expedia.model.common;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.BeanProperty;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.deser.ContextualDeserializer;

import ru.yandex.travel.hotels.common.partners.expedia.model.booking.Itinerary;
import ru.yandex.travel.hotels.common.partners.expedia.model.booking.ItineraryList;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.Fees;

public class Deserializers {

    public static class PercentStringToDecimalDeserializer extends JsonDeserializer<BigDecimal> {

        @Override
        public BigDecimal deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                JsonProcessingException {
            if (parser.getValueAsString().endsWith("%")) {
                var valWithoutPercent = parser.getValueAsString().substring(0, parser.getValueAsString().length() - 1);
                valWithoutPercent = valWithoutPercent.replaceAll(",", ".");
                BigDecimal parsed = BigDecimal.valueOf(Double.parseDouble(valWithoutPercent));
                return parsed.divide(BigDecimal.valueOf(100), 2, RoundingMode.HALF_UP);
            } else {
                return BigDecimal.valueOf(parser.getValueAsDouble());
            }
        }
    }

    public static class ListToMapDeserializer extends JsonDeserializer<Map<String, StringIdEntity>> implements ContextualDeserializer {

        @Override
        public JsonDeserializer<Map<String, StringIdEntity>> createContextual(DeserializationContext outerContext,
                                                                              BeanProperty property) throws JsonMappingException {
            JavaType contextualType = outerContext.getContextualType();
            return new JsonDeserializer<Map<String, StringIdEntity>>() {
                @Override
                public Map<String, StringIdEntity> deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                        JsonProcessingException {
                    if (JsonToken.START_ARRAY.equals(parser.getCurrentToken())) {
                        var listType = context.getTypeFactory().constructCollectionLikeType(ArrayList.class,
                                contextualType.getContentType());
                        var list = (List<StringIdEntity>) context.findRootValueDeserializer(listType)
                                .deserialize(parser, context);
                        AtomicInteger unknownKeys = new AtomicInteger(0);
                        return list.stream().collect(Collectors.toMap(i -> {
                            if (i.getId() != null) {
                                return i.getId();
                            } else {
                                return "unknown-" + unknownKeys.getAndIncrement();
                            }
                        }, Function.identity()));
                    } else {
                        var result = context.findRootValueDeserializer(contextualType).deserialize(parser, context);
                        return (Map<String, StringIdEntity>) result;
                    }
                }
            };
        }

        @Override
        public Map<String, StringIdEntity> deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            return null;
        }
    }

    public static class ItineraryListDeserializer extends JsonDeserializer<ItineraryList> {

        @Override
        public ItineraryList deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                JsonProcessingException {
            if (JsonToken.START_OBJECT == parser.getCurrentToken()) {
                JavaType itemType = context.getTypeFactory().constructType(Itinerary.class);
                var item = (Itinerary) context.findRootValueDeserializer(itemType).deserialize(parser, context);
                var res = new ItineraryList();
                res.add(item);
                return res;
            } else {
                JavaType itemType = context.getTypeFactory().constructCollectionLikeType(ArrayList.class,
                        Itinerary.class);
                var list = (List<Itinerary>) context.findRootValueDeserializer(itemType).deserialize(parser, context);
                var res = new ItineraryList();
                res.addAll(list);
                return res;
            }
        }
    }


    public static class ListFromListOrMapDeserializer extends JsonDeserializer<List<?>> implements ContextualDeserializer {

        @Override
        public JsonDeserializer<List<?>> createContextual(DeserializationContext outerContext,
                                                          BeanProperty property) throws JsonMappingException {
            JavaType contextualType = outerContext.getContextualType();
            return new JsonDeserializer<List<?>>() {
                @Override
                public List<?> deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                        JsonProcessingException {
                    if (JsonToken.START_ARRAY.equals(parser.getCurrentToken())) {
                        var result = context.findRootValueDeserializer(contextualType).deserialize(parser, context);
                        return (List<?>) result;
                    } else {

                        var mapType = context.getTypeFactory().constructMapType(HashMap.class,
                                context.getTypeFactory().constructType(Object.class), contextualType.getContentType());
                        var map = (Map<?, ?>) context.findRootValueDeserializer(mapType).deserialize(parser, context);
                        return new ArrayList<Object>(map.values());
                    }
                }
            };
        }

        @Override
        public List<?> deserialize(JsonParser p, DeserializationContext ctxt) throws IOException,
                JsonProcessingException {
            return null;
        }
    }

    public static class FeesDeserializer extends JsonDeserializer<Fees> {

        @Override
        public Fees deserialize(JsonParser parser, DeserializationContext context) throws IOException,
                JsonProcessingException {
            if (JsonToken.START_ARRAY.equals(parser.getCurrentToken())) {
                var listType = context.getTypeFactory().constructCollectionLikeType(ArrayList.class, LegacyFee.class);
                List<LegacyFee> legacyFees = (List<LegacyFee>) context.findRootValueDeserializer(listType)
                        .deserialize(parser, context);
                return Fees.fromLegacyList(legacyFees);
            } else {
                return (Fees) context.findRootValueDeserializer(context.constructType(Fees.class)).deserialize(parser,
                        context);
            }
        }
    }
}
