package ru.yandex.travel.hotels.common.partners.expedia.utils.coordinates

import org.apache.lucene.document.Document
import org.apache.lucene.document.Field
import org.apache.lucene.document.StoredField
import org.apache.lucene.document.StringField
import org.apache.lucene.index.Term
import org.apache.lucene.search.Query
import org.apache.lucene.search.TermQuery
import org.springframework.boot.actuate.health.Health
import org.springframework.boot.actuate.health.HealthIndicator
import ru.yandex.travel.hotels.common.partners.expedia.proto.THotelCoordinates
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder
import ru.yandex.travel.yt_lucene_index.LuceneIndexService
import ru.yandex.travel.yt_lucene_index.SwitchableYtLuceneIndexParams
import ru.yandex.travel.yt_lucene_index.utils.DocumentUtils.getProtoField

class YtExpediaHotelCoordinatesService(
    params: SwitchableYtLuceneIndexParams,
    luceneIndexBuilder: LuceneIndexBuilder<THotelCoordinates>
) : LuceneIndexService<THotelCoordinates>(params, luceneIndexBuilder, "ExpediaHotelCoordinatesIndex"),
    ExpediaHotelCoordinatesProvider, HealthIndicator {
    companion object {
        private val FIELD_ID_INDEX = "ii"
        private val FIELD_PROTO_STORE = "ps"
    }

    override fun getCoordinates(originalId: String): THotelCoordinates? {
        val query: Query = TermQuery(Term(FIELD_ID_INDEX, originalId))

        return searchOne(query)?.toHotelCoordinates()
    }

    override fun documentProducer(row: THotelCoordinates): Iterable<Document> {
        val document = Document()

        document.add(StringField(FIELD_ID_INDEX, row.originalId, Field.Store.NO))
        document.add(StoredField(FIELD_PROTO_STORE, row.toByteArray()))

        return listOf(document)
    }

    override fun health(): Health {
        return if (isReady()) {
            Health.up().build()
        } else {
            Health.down().build()
        }
    }

    private fun Document.toHotelCoordinates(): THotelCoordinates {
        return getProtoField(THotelCoordinates.parser(), FIELD_PROTO_STORE)
    }
}
