package ru.yandex.travel.hotels.common.partners.travelline;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ThreadLocalRandom;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.Address;
import ru.yandex.travel.hotels.common.partners.travelline.model.AgeGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelPenaltyBasis;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelPenaltyGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancellationInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.ConfirmReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ContactInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.Currency;
import ru.yandex.travel.hotels.common.partners.travelline.model.Email;
import ru.yandex.travel.hotels.common.partners.travelline.model.Error;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.GuestPlacementKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.Hotel;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelChainDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventoryResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelRef;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelStatusChangedResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.Image;
import ru.yandex.travel.hotels.common.partners.travelline.model.ListHotelsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.MakeExtraPaymentResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.MealPlanType;
import ru.yandex.travel.hotels.common.partners.travelline.model.Phone;
import ru.yandex.travel.hotels.common.partners.travelline.model.Placement;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRate;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRef;
import ru.yandex.travel.hotels.common.partners.travelline.model.Policy;
import ru.yandex.travel.hotels.common.partners.travelline.model.Price;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlanKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReadReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.RespHotelReservation;
import ru.yandex.travel.hotels.common.partners.travelline.model.ResponseRoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomTypeQuota;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomTypeUnitKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.Service;
import ru.yandex.travel.hotels.common.partners.travelline.model.ServiceKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayDates;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayUnitKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.Tax;
import ru.yandex.travel.hotels.common.partners.travelline.model.Timezone;
import ru.yandex.travel.hotels.common.partners.travelline.model.VatTaxType;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.OfferDto;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundType;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EHotelConfirmationOutcome;
import ru.yandex.travel.hotels.proto.EHotelOfferOutcome;
import ru.yandex.travel.hotels.proto.EHotelRefundOutcome;
import ru.yandex.travel.hotels.proto.EHotelReservationOutcome;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.THotelTestContext;
import ru.yandex.travel.hotels.proto.TTravellineOffer;


@Slf4j
public class CallContextTravellineClient implements TravellineClient {
    private final TravellineClient baseClient;
    private final CallContext callContext;

    CallContextTravellineClient(TravellineClient baseClient, CallContext callContext) {
        this.baseClient = baseClient;
        this.callContext = callContext;
    }

    @Override
    public CompletableFuture<VerifyReservationResponse> verifyReservation(VerifyReservationRequest request) {
        if (getTestContext() == null) {
            return baseClient.verifyReservation(request);
        } else {
            log.info("Will return a mocked verifyReservation response");
            try {
                return CompletableFuture.completedFuture(mockVerifyReservationResponse());
            } catch (Exception ex) {
                return CompletableFuture.failedFuture(ex);
            }
        }
    }

    @Override
    public CompletableFuture<HotelInfo> getHotelInfo(String hotelCode, String requestId) {
        if (getTestContext() == null) {
            return baseClient.getHotelInfo(hotelCode, requestId);
        }
        try {
            switch (getTestContext().getHotelDataLookupOutcome()) {
                case HO_MOCKED:
                    log.info("Will return a completely mocked Hotel Info");
                    return CompletableFuture.completedFuture(mockHotelInfo());
                case HO_REAL:
                    if (getTestContext().getForceAvailability()) {
                        log.info("Will use a real hotel info with some mocks added");
                        return baseClient.getHotelInfo(hotelCode, requestId).thenApply(this::addMockedDataToRealHotelInfo);
                    } else {
                        log.info("Will use a real hotel info as is");
                        return baseClient.getHotelInfo(hotelCode, requestId);
                    }
                case HO_DISCONNECTED:
                    log.info("Will return a disconnected hotel info");
                    return CompletableFuture.failedFuture(new ReturnedErrorException(Collections.singletonList(
                            new Error(ErrorType.HOTEL_NOT_FOUND, "Hotel is not found or is not available for Yandex " +
                                    "(forced by test context)", null, null))));
                default:
                    return CompletableFuture.failedFuture(new IllegalArgumentException("Invalid " +
                            "HotelDataLookupOutcome"));
            }
        } catch (Exception ex) {
            return CompletableFuture.failedFuture(ex);
        }
    }

    @Override
    public CompletableFuture<HotelOfferAvailability> findOfferAvailability(String hotelCode, LocalDate checkinDate,
                                                                           LocalDate checkoutDate, String requestId) {
        if (getTestContext() != null) {
            if (getTestContext().getForceAvailability()) {
                log.info("Will return a mocked offer availability");
                try {
                    return CompletableFuture.completedFuture(mockOffer());
                } catch (Exception ex) {
                    return CompletableFuture.failedFuture(ex);
                }
            } else {
                return baseClient.findOfferAvailability(hotelCode, checkinDate, checkoutDate, requestId);
            }
        } else {
            return baseClient.findOfferAvailability(hotelCode, checkinDate, checkoutDate, requestId);
        }
    }


    @Override
    public CompletableFuture<HotelReservationResponse> createReservation(HotelReservationRequest request) {
        if (getTestContext() != null) {
            HotelReservationResponse response = new HotelReservationResponse();
            // random sleep, used primarily for stress testing
            if (getTestContext().getReservationCallDelayMax() > 0) {
                sleepRandomly(
                        getTestContext().getReservationCallDelayMin(), getTestContext().getReservationCallDelayMax()
                );
            }

            if (getTestContext().getReservationOutcome() == EHotelReservationOutcome.RO_SOLD_OUT) {
                return CompletableFuture.failedFuture(new ReturnedErrorException(Collections.singletonList(
                        new Error(ErrorType.SOLD_OUT, "SoldOut forced by TestContext", null, null))));
            }
            if (getTestContext().getReservationOutcome() == EHotelReservationOutcome.RO_DISCONNECTED) {
                return CompletableFuture.failedFuture(new ReturnedErrorException(Collections.singletonList(
                        new Error(ErrorType.HOTEL_NOT_FOUND,
                                "Hotel is not found or is not available for Yandex (forced by test context)", null,
                                null))));
            }
            log.info("Will return a mocked createReservation response");
            List<RespHotelReservation> reservation = mockReservedItinerary(BookingStatus.PENDING);
            response.setHotelReservations(reservation);
            return CompletableFuture.completedFuture(response);
        } else {
            return baseClient.createReservation(request);
        }
    }

    private List<RespHotelReservation> mockReservedItinerary(BookingStatus status) {
        Preconditions.checkNotNull(getTestContext());
        BigDecimal expectedPrice = callContext.getItinerary().getFiscalPrice().getNumberStripped();
        switch (callContext.getPhase()) {
            case ORDER_RESERVATION:
                if (getTestContext().getReservationOutcome() == EHotelReservationOutcome.RO_PRICE_MISMATCH) {
                    Preconditions.checkArgument(Math.abs(getTestContext().getPriceMismatchRate()) > 1e-6,
                            "When forcing price mismatch scenarios non-zero priceMismatchRate has to be specified");
                    expectedPrice = expectedPrice.multiply(BigDecimal.valueOf(getTestContext().getPriceMismatchRate()));
                }
                break;
            case ORDER_CONFIRMATION:
                if (getTestContext().getConfirmationOutcome() == EHotelConfirmationOutcome.CO_PRICE_MISMATCH) {
                    Preconditions.checkArgument(Math.abs(getTestContext().getPriceMismatchRate()) > 1e-6,
                            "When forcing price mismatch scenarios non-zero priceMismatchRate has to be specified");
                    expectedPrice = expectedPrice.multiply(BigDecimal.valueOf(getTestContext().getPriceMismatchRate()));
                }
                break;
            case ORDER_REFUND:
                if (getTestContext().getRefundOutcome() == EHotelRefundOutcome.RF_UNEXPECTED_PENALTY) {
                    Preconditions.checkArgument(Math.abs(getTestContext().getPriceMismatchRate()) > 1e-6,
                            "When forcing price mismatch scenarios non-zero priceMismatchRate has to be specified");
                    expectedPrice = expectedPrice.multiply(BigDecimal.valueOf(getTestContext().getPriceMismatchRate()));
                }
                break;
        }

        var rsBuilder = ResponseRoomStay.builder();
        if (getTestContext().getPansionType() != EPansionType.PT_RO && getTestContext().getMealPrice().isInitialized()) {
            var rp = mockRatePlan();

            rsBuilder
                    .ratePlan(rp.toBuilder().vatTaxType(VatTaxType.VAT0).build())
                    .services(List.of(ResponseRoomStay.Service.builder()
                    .kind(ServiceKind.MEAL)
                    .inclusive(true)
                    .name("Сгенерированное питание")
                    .vatTaxType(VatTaxType.VAT120)
                    .price(Price.builder()
                            .priceBeforeTax(getTestContext().getMealPrice().getValue())
                            .priceAfterTax(getTestContext().getMealPrice().getValue())
                            .build())

                    .build()));
        } else {
            rsBuilder.services(Collections.emptyList());
        }

        return List.of(
                RespHotelReservation.builder()
                        .status(status)
                        .reservationNumber("GENERATED-" + ProtoUtils.randomId())
                        .roomStay(rsBuilder.build())
                        .total(RespHotelReservation.Total.builder()
                                .priceBeforeTax(expectedPrice.doubleValue())
                                .currency(Currency.RUB)
                                .build())
                        .build());
    }

    @Override
    public CompletableFuture<ConfirmReservationResponse> confirmReservation(String yandexNumber,
                                                                            String transactionNumber, Money amount) {
        if (getTestContext() != null) {
            if (getTestContext().getConfirmationOutcome() == EHotelConfirmationOutcome.CO_DISCONNECTED) {
                return CompletableFuture.failedFuture(new ReturnedErrorException(Collections.singletonList(
                        new Error(ErrorType.HOTEL_NOT_FOUND,
                                "Hotel is not found or is not available for Yandex (forced by test context)", null,
                                null))));
            } else {
                log.info("Will return a mocked ConfirmReservation response");
                if (getTestContext().getConfirmationCallDelayMin() > 0) {
                    sleepRandomly(
                            getTestContext().getConfirmationCallDelayMin(),
                            getTestContext().getConfirmationCallDelayMax()
                    );
                }
                ConfirmReservationResponse resp = new ConfirmReservationResponse();
                resp.setHotelReservations(mockReservedItinerary(BookingStatus.CONFIRMED));
                return CompletableFuture.completedFuture(resp);
            }
        } else {
            return baseClient.confirmReservation(yandexNumber, transactionNumber, amount);
        }
    }

    @Override
    public CompletableFuture<MakeExtraPaymentResponse> makeExtraPayment(String yandexNumber, String transactionNumber,
                                                                        Money amount) {
        if (getTestContext() != null) {
            log.info("Will return a mocked MakeExtraPayment response");
            MakeExtraPaymentResponse resp = new MakeExtraPaymentResponse();
            resp.setHotelReservations(mockReservedItinerary(BookingStatus.CONFIRMED));
            return CompletableFuture.completedFuture(resp);
        } else {
            return baseClient.makeExtraPayment(yandexNumber, transactionNumber, amount);
        }
    }

    @Override
    public CompletableFuture<CancelReservationResponse> cancelReservation(String yandexNumber) {
        if (getTestContext() != null) {
            log.info("Will return a mocked CancelReservation response");
            CancelReservationResponse resp = new CancelReservationResponse();
            if (callContext.getPhase() == CallContext.CallPhase.ORDER_CANCELLATION) {
                resp.setHotelReservations(mockReservedItinerary(BookingStatus.CANCELLED));
                return CompletableFuture.completedFuture(resp);
            } else if (callContext.getPhase() == CallContext.CallPhase.ORDER_REFUND) {
                if (!isRefundableNow() || getTestContext().getRefundOutcome() == EHotelRefundOutcome.RF_UNABLE_TO_REFUND) {
                    return CompletableFuture.failedFuture(new ReturnedErrorException(List.of(
                            new Error(ErrorType.BOOKING_CANNOT_BE_CANCELLED,
                                    "Reservation cannot be cancelled", null, null))));
                }
                BigDecimal penalty = getExpectedPenaltyAmount();
                if (getTestContext().getRefundOutcome() == EHotelRefundOutcome.RF_UNEXPECTED_PENALTY) {
                    Preconditions.checkArgument(Math.abs(getTestContext().getPriceMismatchRate()) > 1e-6,
                            "When forcing price mismatch scenarios non-zero priceMismatchRate has to be specified");
                    penalty = penalty.multiply(BigDecimal.valueOf(getTestContext().getPriceMismatchRate()));
                }
                var mocked = mockReservedItinerary(BookingStatus.CANCELLED);
                var patched = mocked.get(0).toBuilder().cancellationInfo(
                                CancellationInfo.builder()
                                        .penalty(CancellationInfo.Penalty.builder()
                                                .priceBeforeTax(penalty.doubleValue())
                                                .build())
                                        .build())
                        .build();
                resp.setHotelReservations(Collections.singletonList(patched));
                return CompletableFuture.completedFuture(resp);
            } else {
                return CompletableFuture.failedFuture(new UnsupportedOperationException());
            }
        } else {
            return baseClient.cancelReservation(yandexNumber);
        }
    }

    BigDecimal getExpectedPenaltyAmount() {
        var refundRules = callContext.getItinerary().getRefundRules();
        Instant now = Instant.now();
        RefundRule rule = refundRules.getRuleAtInstant(now);
        if (rule != null) {
            switch (rule.getType()) {
                case FULLY_REFUNDABLE:
                    return BigDecimal.ZERO;
                case REFUNDABLE_WITH_PENALTY:
                    return rule.getPenalty().getNumberStripped();
                case NON_REFUNDABLE:
                    return callContext.getItinerary().getFiscalPrice().getNumberStripped();
            }
        }
        return null;
    }

    boolean isRefundableNow() {
        var refundRules = callContext.getItinerary().getRefundRules();
        Instant now = Instant.now();
        RefundRule rule = refundRules.getRuleAtInstant(now);
        if (rule != null) {
            return rule.getType() != RefundType.NON_REFUNDABLE;
        } else {
            return false;
        }
    }

    @Override
    public CompletableFuture<ReadReservationResponse> readReservation(String yandexNumber) {
        if (getTestContext() != null) {
            ReadReservationResponse resp;
            switch (callContext.getPhase()) {
                case ORDER_RESERVATION:
                    log.info("Will return a mocked ReadReservation response with no itinerary");
                    return CompletableFuture.completedFuture(null);
                case ORDER_CONFIRMATION:
                    if (getTestContext().getConfirmationOutcome() == EHotelConfirmationOutcome.CO_NOT_FOUND) {
                        resp = new ReadReservationResponse();
                        log.info("Will return a mocked ReadReservation response with cancelled itinerary");
                        resp.setHotelReservations(mockReservedItinerary(BookingStatus.CANCELLED));
                        return CompletableFuture.completedFuture(resp);
                    } else {
                        resp = new ReadReservationResponse();
                        log.info("Will return a mocked ReadReservation response with pending itinerary");
                        resp.setHotelReservations(mockReservedItinerary(BookingStatus.PENDING));
                        return CompletableFuture.completedFuture(resp);
                    }
                case ORDER_REFUND:
                    resp = new ReadReservationResponse();
                    log.info("Will return a mocked ReadReservation response with confirmed itinerary");
                    resp.setHotelReservations(mockReservedItinerary(BookingStatus.CONFIRMED));
                    return CompletableFuture.completedFuture(resp);
                case ORDER_CANCELLATION:
                    resp = new ReadReservationResponse();
                    log.info("Will return a mocked ReadReservation response with pending itinerary");
                    resp.setHotelReservations(mockReservedItinerary(BookingStatus.PENDING));
                    return CompletableFuture.completedFuture(resp);
                default:
                    log.warn("A test-context mock will return a failed future as this call is unexpected");
                    return CompletableFuture.failedFuture(new UnsupportedOperationException());
            }
        } else {
            return baseClient.readReservation(yandexNumber);
        }
    }

    @Override
    public CompletableFuture<HotelInventoryResponse> getHotelInventory(String hotelCode) {
        return baseClient.getHotelInventory(hotelCode);
    }

    @Override
    public CompletableFuture<ListHotelsResponse> listHotels() {
        return baseClient.listHotels();
    }

    @Override
    public TravellineClient withCallContext(CallContext testContext) {
        return new CallContextTravellineClient(this.baseClient, callContext);
    }

    @Override
    public CompletableFuture<HotelStatusChangedResponse> notifyHotelStatusChanged(String hotelCode) {
        return baseClient.notifyHotelStatusChanged(hotelCode);
    }

    @Override
    public CompletableFuture<HotelDetailsResponse> getHotelDetails(String hotelCode) {
        return baseClient.getHotelDetails(hotelCode);
    }

    @Override
    public CompletableFuture<HotelChainDetailsResponse> getHotelChainDetails(String inn) {
        return baseClient.getHotelChainDetails(inn);
    }

    private VerifyReservationResponse mockVerifyReservationResponse() {
        Preconditions.checkNotNull(getTestContext());
        VerifyReservationResponse response = new VerifyReservationResponse();
        if ((callContext.getPhase() == CallContext.CallPhase.OFFER_VALIDATION &&
                getTestContext().getGetOfferOutcome() == EHotelOfferOutcome.OO_SOLD_OUT) ||
                (callContext.getPhase() == CallContext.CallPhase.ORDER_CREATION &&
                        getTestContext().getCreateOrderOutcome() == EHotelOfferOutcome.OO_SOLD_OUT)) {
            throw new ReturnedErrorException(Collections.singletonList(new Error(ErrorType.SOLD_OUT,
                    "SoldOut forced by TestContext", null, null)));
        }

        if ((callContext.getPhase() == CallContext.CallPhase.OFFER_VALIDATION &&
                getTestContext().getGetOfferOutcome() == EHotelOfferOutcome.OO_DISCONNECTED) ||
                (callContext.getPhase() == CallContext.CallPhase.ORDER_CREATION &&
                        getTestContext().getCreateOrderOutcome() == EHotelOfferOutcome.OO_DISCONNECTED)) {
            throw new ReturnedErrorException(Collections.singletonList(new Error(ErrorType.HOTEL_NOT_FOUND,
                    "Hotel is not found or is not available for Yandex (by TestContext)", null, null)));
        }

        double priceMultiplier = 1.0;
        if ((callContext.getPhase() == CallContext.CallPhase.OFFER_VALIDATION &&
                getTestContext().getGetOfferOutcome() == EHotelOfferOutcome.OO_PRICE_MISMATCH) ||
                (callContext.getPhase() == CallContext.CallPhase.ORDER_CREATION &&
                        getTestContext().getCreateOrderOutcome() == EHotelOfferOutcome.OO_PRICE_MISMATCH)) {
            priceMultiplier = 2.0;
        }
        OfferDto offerDto = getOffer();

        var hotelReservationBuilder = RespHotelReservation.builder()
                .hotelRef(RespHotelReservation.HotelRef.builder()
                        .build())
                .total(RespHotelReservation.Total.builder()
                        .priceBeforeTax(BigDecimal.valueOf(offerDto.getBestPrice()).multiply(BigDecimal.valueOf(priceMultiplier)).setScale(0, RoundingMode.HALF_UP).intValue())
                        .taxes(getTaxesOutOfOffer(offerDto))
                        .currency(Currency.RUB)
                        .build())
                .roomStay(ResponseRoomStay.builder()
                        .placementRates(getPlacementRatesOutOufOffer(offerDto))
                        .cancelPenaltyGroup(getCancelPenaltiesGroupOutOfOffer(offerDto))
                        .ratePlan(RatePlan.builder()
                                .code(offerDto.getRatePlan().getCode())
                                .build())
                        .stayDates(offerDto.getStayDates())
                        .build());
        response.setHotelReservations(Collections.singletonList(hotelReservationBuilder.build()));
        return response;
    }

    private CancelPenaltyGroup getCancelPenaltiesGroupOutOfOffer(OfferDto offerDto) {
        var builder = CancelPenaltyGroup.builder();
        for (var rule : offerDto.getRefundRules().getRules()) {
            if (rule.getType() == RefundType.FULLY_REFUNDABLE) {
                continue;
            }
            if (rule.getType() == RefundType.NON_REFUNDABLE) {
                if (rule.getStartsAt() != null && rule.getStartsAt().atOffset(ZoneOffset.ofHours(3)).toLocalDateTime().equals(offerDto.getStayDates().getStartDate())) {
                    continue;
                }
                builder.cancelPenalty(CancelPenaltyGroup.CancelPenalty.builder()
                        .percent(100.0)
                        .basis(CancelPenaltyBasis.FULL_STAY)
                        .currency(Currency.RUB)
                        .start(rule.getStartsAt() == null ? null :
                                rule.getStartsAt().atOffset(ZoneOffset.ofHours(3)).toLocalDateTime())
                        .end(rule.getEndsAt() == null ? null :
                                rule.getEndsAt().atOffset(ZoneOffset.ofHours(3)).toLocalDateTime())
                        .build()
                );
            }
            if (rule.getType() == RefundType.REFUNDABLE_WITH_PENALTY) {
                var percent = rule.getPenalty().getNumberStripped().divide(BigDecimal.valueOf(offerDto.getBestPrice()),
                                4, RoundingMode.HALF_UP).multiply(BigDecimal.valueOf(100))
                        .setScale(0, RoundingMode.HALF_UP)
                        .doubleValue();

                builder.cancelPenalty(CancelPenaltyGroup.CancelPenalty.builder()
                        .percent(percent)
                        .basis(CancelPenaltyBasis.FULL_STAY)
                        .currency(Currency.RUB)
                        .start(rule.getStartsAt() == null ? null :
                                rule.getStartsAt().atOffset(ZoneOffset.ofHours(3)).toLocalDateTime())
                        .end(rule.getEndsAt() == null ? null :
                                rule.getEndsAt().atOffset(ZoneOffset.ofHours(3)).toLocalDateTime())
                        .build()
                );
            }
        }
        return builder.build();
    }

    private List<PlacementRate> getPlacementRatesOutOufOffer(OfferDto offer) {
        var placementSet = offer.getPossiblePlacements().get(offer.getBestPlacementIndex());
        List<PlacementRate> result = new ArrayList<>();
        for (var placement : placementSet.getPlacements()) {
            result.add(PlacementRate.builder()
                    .roomTypeCode(offer.getRoomType().getCode())
                    .ratePlanCode(offer.getRatePlan().getCode())
                    .placement(PlacementRef.builder()
                            .code(placement.getCode())
                            .index(placement.getIndex())
                            .kind(placement.getKind())
                            .build())
                    .rates(placement.getRates())
                    .build());
        }
        return result;
    }

    private List<Tax> getTaxesOutOfOffer(OfferDto offer) {
        var placementSet = offer.getPossiblePlacements().get(offer.getBestPlacementIndex());
        Map<Tuple4<String, String, String, String>, Tax> taxMap = new HashMap<>();
        for (Placement p : placementSet.getPlacements()) {
            for (Tax tax : p.getTaxes()) {
                var key = Tuple4.tuple(tax.getKind(), tax.getCode(), tax.getName(), tax.getDescription());
                if (taxMap.containsKey(key)) {
                    taxMap.put(key,
                            taxMap.get(key).toBuilder().amount(taxMap.get(key).getAmount() + tax.getAmount()).build());
                } else {
                    taxMap.put(key, tax);
                }
            }
        }
        return new ArrayList<>(taxMap.values());
    }

    private HotelInfo addMockedDataToRealHotelInfo(HotelInfo hotelInfo) {
        Preconditions.checkNotNull(callContext);
        Hotel.HotelBuilder hotelBuilder = hotelInfo.getHotel().toBuilder();
        hotelBuilder.roomType(mockRoomType(Occupancy.fromString(callContext.getSearchOffersReq().getOccupancy())));
        hotelBuilder.ratePlan(mockRatePlan());
        hotelBuilder.clearAgeGroups();
        hotelBuilder.ageGroup(mockAgeGroup());
        for (Service srv : mockServices()) {
            hotelBuilder.service(srv);
        }
        HotelInfo result = new HotelInfo();
        result.setHotel(hotelBuilder.build());
        return result;
    }

    private HotelInfo mockHotelInfo() {
        Preconditions.checkNotNull(callContext);
        HotelInfo hi = new HotelInfo();
        Occupancy occupancy = Occupancy.fromString(callContext.getSearchOffersReq().getOccupancy());
        hi.setHotel(Hotel.builder()
                .roomType(mockRoomType(occupancy))
                .ratePlan(mockRatePlan())
                .timezone(Timezone.builder()
                        .name("(UTC+03:00) Moscow, St. Petersburg")
                        .offset("+03:00")
                        .build())
                .policy(Policy.builder()
                        .checkInTime(LocalTime.of(14, 0))
                        .checkOutTime(LocalTime.of(12, 0))
                        .build())
                .contactInfo(ContactInfo.builder()
                        .address(Address.builder()
                                .postalCode("119021")
                                .countryCode("RUS")
                                .region("г. Москва")
                                .county("Хамовники")
                                .cityName("г. Москва")
                                .addressLine(Collections.singletonList("ул. Льва Толстого, 16"))
                                .remark("http://www.yandex.ru")
                                .latitude(55.734399)
                                .longitude(37.587293)
                                .build())
                        .phone(Phone.builder()
                                .phoneNumber("8 800 511-71-04")
                                .remark("Поддержка Яндекс Путешествий")
                                .build())
                        .email(Email.builder()
                                .emailAddress("orders@travel.yandex.ru")
                                .build())
                        .build())
                .currency(Currency.RUB)
                .ageGroup(mockAgeGroup())
                .services(mockServices())
                .stayUnitKind(StayUnitKind.NIGHT)
                .stars(5)
                .image(Image.builder()
                        .url("https://avatars.mds.yandex.net/get-altay" +
                                "/223006/2a0000015b19222d131711047d866611eaf2/XXXL")
                        .build())
                .image(Image.builder() // twice, since the first one is skipped
                        .url("https://avatars.mds.yandex.net/get-altay" +
                                "/223006/2a0000015b19222d131711047d866611eaf2/XXXL")
                        .build())
                .code("0")
                .name("Тестовый сгенерированный отель")
                .build()
        );
        return hi;
    }

    private AgeGroup mockAgeGroup() {
        return AgeGroup.builder()
                .code("0")
                .minAge(0)
                .maxAge(17)
                .build();
    }

    private RatePlan mockRatePlan() {
        return RatePlan.builder()
                .code("0")
                .name("Сгенерированный тарифный план")
                .description("Тарифный план, доступный только для тестировщиков Яндекса")
                .kind(RatePlanKind.COMMON)
                .sources(Set.of("yandex"))
                .currency(Currency.RUB)
                .build();
    }

    private RoomType mockRoomType(Occupancy occupancy) {
        Preconditions.checkNotNull(getTestContext());
        return RoomType.builder()
                .code("0")
                .name(getTestContext().getOfferName())
                .description("Ненастоящий номер в ненастоящем отеле, цена на размещение в котором " +
                        "взимается в ненастоящих деньгах. Ну и вообще вам всё это кажется.")
                .kind(RoomTypeUnitKind.ROOM)
                .amenities(Collections.emptyList())
                .maxAdultOccupancy(occupancy.getAdults() + occupancy.getChildren().size())
                .image(Image.builder()
                        .url("https://avatars.mds.yandex.net/get-altay/200322/2a0000015b1717fa178796a0b1d5ec6b0689" +
                                "/XXXL")
                        .build())
                .build();
    }

    private List<Service> mockServices() {
        Preconditions.checkNotNull(getTestContext());
        switch (getTestContext().getPansionType()) {
            case PT_AI:
            case PT_UAI:
            case PT_LAI:
                return Collections.singletonList(Service.builder()
                        .code("0")
                        .name("Все включено")
                        .description("Все питание включено, хотя бы с целью тестирования")
                        .kind(ServiceKind.MEAL)
                        .mealPlanType(MealPlanType.ALL_INCLUSIVE)
                        .build());
            case PT_BB:
                return Collections.singletonList(Service.builder()
                        .code("0")
                        .name("Завтрак")
                        .description("Вкусный, но — увы! — полностью выдуманный завтрак")
                        .kind(ServiceKind.MEAL)
                        .mealPlanType(MealPlanType.BREAKFAST)
                        .build());
            case PT_FB:
                return Collections.singletonList(Service.builder()
                        .code("0")
                        .name("Завтрак, обед и ужин")
                        .description("Полный сгенерированный пансион")
                        .kind(ServiceKind.MEAL)
                        .mealPlanType(MealPlanType.FULL_BOARD)
                        .build());
            case PT_HB:
                return Collections.singletonList(Service.builder()
                        .code("0")
                        .name("Завтрак и ужин")
                        .description("Пансион половинчатый, зато сгенерирован полностью")
                        .kind(ServiceKind.MEAL)
                        .mealPlanType(MealPlanType.HALF_BOARD)
                        .build());
            case PT_RO:
                return Collections.emptyList();
            case PT_BD:
                return Collections.singletonList(Service.builder()
                        .code("0")
                        .name("Ужин")
                        .description("Вкусный, но — увы! — полностью выдуманный ужин")
                        .kind(ServiceKind.MEAL)
                        .mealPlanType(MealPlanType.DINNER)
                        .build());
        }
        return Collections.emptyList();
    }

    private HotelOfferAvailability mockOffer() {
        HotelOfferAvailability offer = new HotelOfferAvailability();
        offer.setRoomStays(Collections.singletonList(mockRoomStay()));
        offer.setRoomTypeQuotas(Collections.singletonList(RoomTypeQuota.builder()
                .rph("0")
                .quantity(100)
                .build()));
        var services = mockServices();
        if (services.size() > 0) {
            offer.setServices(Collections.singletonList(
                    HotelOfferAvailability.ServiceConditions.builder()
                            .code(services.get(0).getCode())
                            .inclusive(true)
                            .rph("0")
                            .build()
            ));
        } else {
            offer.setServices(Collections.emptyList());
        }
        return offer;
    }

    private THotelTestContext getTestContext() {
        if (callContext == null) {
            return null;
        } else {
            return callContext.getTestContext();
        }
    }

    private List<Placement> mockPlacementsForOccupancy() {
        Preconditions.checkNotNull(getTestContext());
        List<Placement> res = new ArrayList<>(2);
        Occupancy occupancy = Occupancy.fromString(callContext.getSearchOffersReq().getOccupancy());
        int numAdults = occupancy.getAdults();
        int numChildren = occupancy.getChildren().size();
        Placement adultPlacement = Placement.builder()
                .kind(GuestPlacementKind.ADULT)
                .index(0)
                .code("0")
                .capacity(numAdults)
                .priceBeforeTax(getTestContext().getPriceAmount())
                .currency(Currency.RUB)
                .build();
        res.add(adultPlacement);
        if (numChildren > 0) {
            Placement childPlacement = Placement.builder()
                    .kind(GuestPlacementKind.CHILD)
                    .index(1)
                    .code("1")
                    .capacity(numChildren)
                    .ageGroup(0)
                    .priceBeforeTax(0.0)
                    .currency(Currency.RUB)
                    .build();
            res.add(childPlacement);
        }
        return res;
    }

    private CancelPenaltyGroup mockCancelPenaltyGroup() {
        Preconditions.checkNotNull(getTestContext());
        var builder = CancelPenaltyGroup.builder();
        switch (getTestContext().getCancellation()) {
            case CR_FULLY_REFUNDABLE:
                return builder.build();
            case CR_NON_REFUNDABLE:
                return builder.cancelPenalties(Collections.singletonList(
                        CancelPenaltyGroup.CancelPenalty.builder()
                                .percent(100.0)
                                .basis(CancelPenaltyBasis.FULL_STAY)
                                .build()
                )).build();
            case CR_PARTIALLY_REFUNDABLE:
                return builder.cancelPenalties(Collections.singletonList(
                        CancelPenaltyGroup.CancelPenalty.builder()
                                .percent(getTestContext().getPartiallyRefundableRate())
                                .basis(CancelPenaltyBasis.FULL_STAY)
                                .build()
                )).build();
            case CR_CUSTOM:
                LocalDateTime border1 =
                        LocalDateTime.now().plusMinutes(getTestContext().getPartiallyRefundableInMinutes());
                LocalDateTime border2 =
                        LocalDateTime.now().plusMinutes(getTestContext().getNonRefundableInMinutes());
                return builder
                        .cancelPenalties(new ArrayList<>(List.of(
                                CancelPenaltyGroup.CancelPenalty.builder()
                                        .start(border1)
                                        .end(border2)
                                        .percent(getTestContext().getPartiallyRefundableRate())
                                        .basis(CancelPenaltyBasis.FULL_STAY)
                                        .build(),
                                CancelPenaltyGroup.CancelPenalty.builder()
                                        .start(border2)
                                        .percent(100.0)
                                        .basis(CancelPenaltyBasis.FULL_STAY)
                                        .build())))
                        .build();
        }
        return builder.build();
    }

    private RoomStay mockRoomStay() {
        Preconditions.checkNotNull(getTestContext());
        var roomStayBuilder = RoomStay.builder()
                .hotelRef(HotelRef.builder()
                        .code(callContext.getSearchOffersReq().getHotelId().getOriginalId())
                        .build())
                .roomType(RoomStay.RoomTypeRef.builder()
                        .code("0")
                        .placements(mockPlacementsForOccupancy())
                        .roomTypeQuotaRph("0")
                        .build()
                )
                .ratePlan(RoomStay.RatePlanRef.builder()
                        .code("0")
                        .cancelPenaltyGroup(mockCancelPenaltyGroup())
                        .build())
                .placementRates(mockPlacementRates())
                .stayDates(mockStayDates());
        if (mockServices().size() > 0) {
            roomStayBuilder.service(RoomStay.ServiceRef.builder()
                    .rph("0")
                    .build());
        } else {
            roomStayBuilder.services(Collections.emptyList());
        }
        return roomStayBuilder.build();
    }

    private StayDates mockStayDates() {
        Preconditions.checkNotNull(callContext);
        LocalDate checkIn = LocalDate.parse(callContext.getSearchOffersReq().getCheckInDate());
        LocalDate checkOut = LocalDate.parse(callContext.getSearchOffersReq().getCheckOutDate());
        return StayDates.builder()
                .startDate(checkIn.atTime(14, 0))
                .endDate(checkOut.atTime(12, 0))
                .build();
    }

    private List<PlacementRate> mockPlacementRates() {
        Preconditions.checkNotNull(getTestContext());
        Occupancy occupancy = Occupancy.fromString(callContext.getSearchOffersReq().getOccupancy());
        LocalDate checkIn = LocalDate.parse(callContext.getSearchOffersReq().getCheckInDate());
        LocalDate checkOut = LocalDate.parse(callContext.getSearchOffersReq().getCheckOutDate());
        var adultRateBuilder = PlacementRate.builder()
                .roomTypeCode("0")
                .ratePlanCode("0")
                .placement(PlacementRef.builder()
                        .kind(GuestPlacementKind.ADULT)
                        .index(0)
                        .code("0")
                        .build());
        var childRateBuilder = (occupancy.getChildren().size() > 0) ? PlacementRate.builder() : null;
        if (childRateBuilder != null) {
            childRateBuilder = childRateBuilder
                    .roomTypeCode("0")
                    .ratePlanCode("0")
                    .placement(PlacementRef.builder()
                            .kind(GuestPlacementKind.CHILD)
                            .index(1)
                            .code("1")
                            .build());
        }
        long numNights = ChronoUnit.DAYS.between(checkIn, checkOut);
        int pricePerNight = getTestContext().getPriceAmount() / (int) numNights;
        int remainingPrice = getTestContext().getPriceAmount();
        LocalDate current = checkIn;
        while (current.plusDays(1).isBefore(checkOut)) {
            adultRateBuilder.rate(PlacementRate.Rate.builder()
                    .date(current)
                    .priceBeforeTax((double) pricePerNight)
                    .currency(Currency.RUB)
                    .build());
            remainingPrice -= pricePerNight;
            current = current.plusDays(1);
            if (childRateBuilder != null) {
                childRateBuilder.rate(PlacementRate.Rate.builder()
                        .date(current)
                        .priceBeforeTax(0.0)
                        .currency(Currency.RUB)
                        .build());
            }
        }
        adultRateBuilder.rate(PlacementRate.Rate.builder()
                .date(checkOut.minusDays(1))
                .priceBeforeTax((double) remainingPrice)
                .currency(Currency.RUB)
                .build());
        if (childRateBuilder != null) {
            childRateBuilder.rate(PlacementRate.Rate.builder()
                    .date(checkOut.minusDays(1))
                    .priceBeforeTax(0.0)
                    .currency(Currency.RUB)
                    .build());
        }
        if (childRateBuilder != null) {
            return List.of(adultRateBuilder.build(), childRateBuilder.build());
        } else {
            return Collections.singletonList(adultRateBuilder.build());
        }
    }

    private OfferDto getOffer() {
        Preconditions.checkNotNull(callContext);
        Preconditions.checkNotNull(callContext.getOfferData());
        return ProtoUtils.fromTJson(((TTravellineOffer) callContext.getOfferData()).getTravellineOfferDTO(),
                OfferDto.class);
    }

    private final void sleepRandomly(int minDelay, int maxDelay) {
        long sleepTime = ThreadLocalRandom.current().nextLong(minDelay, maxDelay + 1);
        if (sleepTime == 0) {
            return;
        }
        try {
            Thread.sleep(sleepTime);
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
