package ru.yandex.travel.hotels.common.partners.travelline;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import com.google.common.base.Preconditions;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.BaseClient;
import ru.yandex.travel.hotels.common.partners.base.ClientMethods;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.CacheNotFoundException;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BaseTravellineResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancellationReason;
import ru.yandex.travel.hotels.common.partners.travelline.model.ConfirmReservationOrExtraPaymentRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.ConfirmReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelChainDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventoryResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelStatusChangedResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ListHotelsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.MakeExtraPaymentResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReadReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationResponse;

@Slf4j
public class DefaultTravellineClient extends BaseClient<TravellineClientProperties> implements TravellineClient {

    private static final String TRAVELLINE_DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm";
    private static final String DEFAULT_LANGUAGE = "ru-ru";
    private static final String API_KEY_HEADER = "X-ApiKey";
    private static final String HOTEL_CODE_PARAM = "hotel.code";

    private static final String VERIFY_RESERVATION = "VerifyReservation";
    private static final String OFFER_AVAILABILITY = "OfferAvailability";
    private static final String HOTEL_INFO = "HotelInfo";
    private static final String HOTEL_RESERVATION = "HotelReservation";
    private static final String CONFIRM_RESERVATION = "ConfirmReservation";
    private static final String CANCEL_RESERVATION = "CancelReservation";
    private static final String READ_RESERVATION = "ReadReservation";
    private static final String INVENTORY = "HotelInventory";
    private static final String HOTELS = "Hotels";
    private static final String HOTEL_STATUS_CHANGED = "HotelStatusChanged";
    private static final String HOTEL_DETAILS = "HotelDetails";
    private static final String HOTEL_CHAIN_DETAILS = "HotelChainDetails";
    private static final String EXTRA_PAYMENT = "ExtraPayment";
    @Getter
    private static final ClientMethods methods;
    private static final Set<String> OFFER_METHODS;

    static {
        methods = new ClientMethods()
            .register(VERIFY_RESERVATION, "/booking/verify_reservation", HttpMethod.POST, VerifyReservationResponse.class)
            .register(OFFER_AVAILABILITY, "/booking/hotel_offer_availability", HttpMethod.GET, HotelOfferAvailability.class)
            .register(HOTEL_INFO, "/booking/hotel_info", HttpMethod.GET, HotelInfo.class)
            .register(HOTEL_RESERVATION, "/booking/hotel_reservation", HttpMethod.POST, HotelReservationResponse.class)
            .register(CONFIRM_RESERVATION, "/booking/confirm_reservation", HttpMethod.POST, ConfirmReservationResponse.class)
            .register(CANCEL_RESERVATION, "/booking/cancel_reservation", HttpMethod.POST, CancelReservationResponse.class)
            .register(READ_RESERVATION, "/booking/read_reservation", HttpMethod.GET, ReadReservationResponse.class)
            .register(INVENTORY, "/booking/hotel_inventory", HttpMethod.GET, HotelInventoryResponse.class)
            .register(HOTELS, "/booking/hotels", HttpMethod.GET, ListHotelsResponse.class)
            .register(HOTEL_STATUS_CHANGED, "/offer/hotel_status_changed", HttpMethod.GET, HotelStatusChangedResponse.class)

            .register(HOTEL_DETAILS, "/offer/hotel_details", HttpMethod.GET, HotelDetailsResponse.class)
            .register(HOTEL_CHAIN_DETAILS, "/offer/hotel_chain_details", HttpMethod.GET, HotelChainDetailsResponse.class)
            .register(EXTRA_PAYMENT, "/booking/make_extra_payment", HttpMethod.POST, MakeExtraPaymentResponse.class);

        OFFER_METHODS = Set.of(HOTEL_DETAILS, HOTEL_CHAIN_DETAILS, HOTEL_STATUS_CHANGED);

    }

    public DefaultTravellineClient(AsyncHttpClientWrapper clientWrapper, TravellineClientProperties properties, Retry retryHelper) {
        super(properties, clientWrapper, createObjectMapper(), retryHelper);
    }

    public static ObjectMapper createObjectMapper() {
        ObjectMapper mapper = new ObjectMapper();
        JavaTimeModule javaTimeModule = new JavaTimeModule();
        javaTimeModule.addDeserializer(
            LocalDateTime.class,
            new LocalDateTimeDeserializer(DateTimeFormatter.ofPattern(TRAVELLINE_DATE_TIME_PATTERN)));

        javaTimeModule.addSerializer(
            LocalDateTime.class,
            new LocalDateTimeSerializer(DateTimeFormatter.ofPattern(TRAVELLINE_DATE_TIME_PATTERN)));
        mapper.registerModule(javaTimeModule);
        mapper.disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS);
        mapper.setPropertyNamingStrategy(PropertyNamingStrategy.SNAKE_CASE);
        mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        return mapper;
    }

    @Override
    public CompletableFuture<HotelInfo> getHotelInfo(String hotelCode, String requestId) {
        return call(HOTEL_INFO,
            new CallArguments()
                .withQueryParam(HOTEL_CODE_PARAM, hotelCode)
                .withQueryParam("language", DEFAULT_LANGUAGE)
                .withRequestId(requestId));
    }

    @Override
    public CompletableFuture<HotelOfferAvailability> findOfferAvailability(String hotelCode,
                                                                           LocalDate checkinDate,
                                                                           LocalDate checkoutDate,
                                                                           String requestId) {
        CompletableFuture<HotelOfferAvailability> future = call(OFFER_AVAILABILITY,
            new CallArguments()
                .withQueryParam(HOTEL_CODE_PARAM, hotelCode)
                .withQueryParam("start_date", checkinDate.toString())
                .withQueryParam("end_date", checkoutDate.toString())
                .withRequestId(requestId));
        return future.thenApply(r -> {
            if (r.getRoomStays().isEmpty() && r.getWarnings() !=null && r.getWarnings().stream().anyMatch(w -> w.getErrorCode() == ErrorType.CACHE_NOT_FOUND)) {
                throw new CacheNotFoundException();
            }
            return r;
        });
    }

    @Override
    public CompletableFuture<VerifyReservationResponse> verifyReservation(VerifyReservationRequest request) {
        return call(VERIFY_RESERVATION, new CallArguments().withBody(request));
    }

    @Override
    public CompletableFuture<HotelReservationResponse> createReservation(HotelReservationRequest request) {
        return call(HOTEL_RESERVATION, new CallArguments().withBody(request));
    }

    @Override
    public CompletableFuture<ConfirmReservationResponse> confirmReservation(String yandexNumber,
                                                                            String transactionNumber, Money amount) {
        final ConfirmReservationOrExtraPaymentRequest.Guarantee.GuaranteeBuilder guaranteeBuilder =
                ConfirmReservationOrExtraPaymentRequest.Guarantee.builder().transactionId(transactionNumber);
        if (amount != null) {
            guaranteeBuilder.paymentAmount(amount.getNumber().doubleValue());
            guaranteeBuilder.paymentCurrency(amount.getCurrency().getCurrencyCode());
        }
        return call(CONFIRM_RESERVATION,
            new CallArguments()
                .withBody(ConfirmReservationOrExtraPaymentRequest.builder()
                    .yandexNumber(yandexNumber)
                    .guarantee(guaranteeBuilder.build())
                    .build()));
    }

    @Override
    public CompletableFuture<MakeExtraPaymentResponse> makeExtraPayment(String yandexNumber, String transactionNumber,
                                                                        Money amount) {
        Preconditions.checkNotNull(amount, "extra amount is null");

        return call(EXTRA_PAYMENT,
                new CallArguments()
                        .withBody(ConfirmReservationOrExtraPaymentRequest.builder()
                                .yandexNumber(yandexNumber)
                                .guarantee(ConfirmReservationOrExtraPaymentRequest.Guarantee.builder()
                                        .transactionId(transactionNumber)
                                        .paymentAmount(amount.getNumber().doubleValue())
                                        .paymentCurrency(amount.getCurrency().getCurrencyCode())
                                        .build())
                                .build()));
    }

    @Override
    public CompletableFuture<CancelReservationResponse> cancelReservation(String yandexNumber) {
        //TODO(tivelkov): think about passing CancellationReason here
        return call(CANCEL_RESERVATION,
            new CallArguments().withBody(CancelReservationRequest.builder()
                .yandexNumber(yandexNumber)
                .reason(CancellationReason.CANCEL_BOOKING)
                .build()));
    }

    @Override
    public CompletableFuture<ReadReservationResponse> readReservation(String yandexNumber) {
        CompletableFuture<ReadReservationResponse> res = call(READ_RESERVATION,
            new CallArguments().withQueryParam("yandex_number", yandexNumber));
        return res.exceptionally(t -> {
            Throwable cause = t.getCause();
            if (cause instanceof ReturnedErrorException) {
                ReturnedErrorException err = (ReturnedErrorException) cause;
                if (err.getErrors().size() == 1 && err.getErrors().get(0).getErrorCode() == ErrorType.BOOKING_NOT_FOUND) {
                    return null;
                }
            }
            if (cause instanceof PartnerException) {
                throw (PartnerException) cause;
            } else {
                throw new PartnerException(cause);
            }
        });
    }

    @Override
    public CompletableFuture<HotelInventoryResponse> getHotelInventory(String hotelCode) {
        return call(INVENTORY, new CallArguments().withQueryParam(HOTEL_CODE_PARAM, hotelCode));
    }

    @Override
    public CompletableFuture<ListHotelsResponse> listHotels() {
        CompletableFuture<ListHotelsResponse> future = call(HOTELS);
        return future.thenApply(lhr -> new ListHotelsResponse(lhr.getHotels().stream().distinct().collect(Collectors.toList())));
    }

    @Override
    public CompletableFuture<HotelStatusChangedResponse> notifyHotelStatusChanged(String hotelCode) {
        return call(HOTEL_STATUS_CHANGED, new CallArguments().withQueryParam(HOTEL_CODE_PARAM, hotelCode));
    }

    @Override
    public CompletableFuture<HotelDetailsResponse> getHotelDetails(String hotelCode) {
        return call(HOTEL_DETAILS, new CallArguments().withQueryParam(HOTEL_CODE_PARAM, hotelCode));
    }

    @Override
    public CompletableFuture<HotelChainDetailsResponse> getHotelChainDetails(String inn) {
        return call(HOTEL_CHAIN_DETAILS, new CallArguments().withQueryParam("inn", inn));
    }

    @Override
    protected ClientMethods getClientMethods() {
        return methods;
    }

    @Override
    protected <R> R handleResult(R response, Throwable throwable) {
        if (throwable == null && response instanceof BaseTravellineResponse) {
            BaseTravellineResponse travellineResponse = (BaseTravellineResponse) response;
            if (travellineResponse.getErrors() != null && travellineResponse.getErrors().size() > 0) {
                throw new ReturnedErrorException(travellineResponse.getErrors());
            }
            if (travellineResponse.getWarnings() != null) {
                travellineResponse.getWarnings().forEach(w -> log.warn(w.toString()));
            }
        }
        return super.handleResult(response, throwable);
    }

    @Override
    protected @NotNull Map<String, String> getCommonHeaders(String destinationMethod) {
        String key = OFFER_METHODS.contains(destinationMethod) && properties.getOfferApiKey() != null ? properties.getOfferApiKey() : properties.getApiKey();
        return Map.of(API_KEY_HEADER, key);
    }
}
