package ru.yandex.travel.hotels.common.partners.travelline;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelPenaltyBasis;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelPenaltyGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.Currency;
import ru.yandex.travel.hotels.common.partners.travelline.model.Placement;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRate;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayDates;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;

@Slf4j
public class TravellineRefundRulesBuilder {

    //TODO parameterize this method with checkin time and set this time to startDate
    public static RefundRules build(CancelPenaltyGroup penaltyGroup, List<Placement> placements,
                                    String ratePlanCode,
                                    StayDates stayDates, ZoneOffset zoneOffset, int totalPrice) {
        var builder = RefundRules.builder();
        List<RefundRule> rules = new ArrayList<>();
        BigDecimal total = BigDecimal.valueOf(totalPrice);
        Instant startInstant = getInstant(stayDates.getStartDate(), zoneOffset);
        Instant endInstant = getInstant(stayDates.getEndDate(), zoneOffset);
        int days = (int) ChronoUnit.DAYS.between(startInstant.atOffset(ZoneOffset.UTC).toLocalDate(),
                endInstant.atOffset(ZoneOffset.UTC).toLocalDate());

        List<CancelPenaltyGroup.CancelPenalty> travellinePenalties;
        if (penaltyGroup != null && penaltyGroup.getCancelPenalties() != null) {
            travellinePenalties = new ArrayList<>(penaltyGroup.getCancelPenalties());
        } else {
            travellinePenalties = new ArrayList<>();
        }
        if (travellinePenalties.stream().filter(penalty -> penalty.getStart() == null).count() > 1) {
            throw new PartnerException(String.format("Rate plan %s: More than 1 interval with empty start are found",
                    ratePlanCode));
        }
        if (travellinePenalties.stream().filter(penalty -> penalty.getEnd() == null).count() > 1) {
            throw new PartnerException(String.format("Rate plan %s: More than 1 interval with empty end are found",
                    ratePlanCode));
        }
        travellinePenalties.sort((penalty1, penalty2) -> {
            if (penalty1.getStart() == null) {
                return -1;
            }
            if (penalty2.getStart() == null) {
                return 1;
            }
            return penalty1.getStart().compareTo(penalty2.getStart());
        });

        Instant firstPenaltyStart;
        RefundRule previousRule = null;
        if (travellinePenalties.isEmpty()) {
            firstPenaltyStart = startInstant;
        } else {
            firstPenaltyStart = getInstant(travellinePenalties.get(0).getStart(), zoneOffset);
            if (firstPenaltyStart != null && firstPenaltyStart.isAfter(startInstant)) {
                firstPenaltyStart = startInstant;
            }
        }
        if (firstPenaltyStart != null) {
            RefundRule fullyRefundableRule = RefundRule.builder()
                    .endsAt(firstPenaltyStart)
                    .type(RefundType.FULLY_REFUNDABLE)
                    .build();
            rules.add(fullyRefundableRule);
            previousRule = fullyRefundableRule;
        }
        BigDecimal previousPenaltyAmount = null;
        Instant previousEnd = null;
        for (CancelPenaltyGroup.CancelPenalty travellinePenalty : travellinePenalties) {
            if (travellinePenalty.getStart() != null && getInstant(travellinePenalty.getStart(), zoneOffset).isAfter(startInstant)) {
                continue;
            }
            BigDecimal penaltyAmount;
            switch (travellinePenalty.getBasis()) {
                case NIGHTS:
                    if (placements != null) {
                        penaltyAmount = BigDecimal.valueOf(
                                placements
                                        .stream()
                                        .filter(placementRate -> placementRate.getRates() != null)
                                        .mapToDouble(placementRate -> placementRate.getRates().subList(0,
                                                travellinePenalty.getNights())
                                                .stream()
                                                .filter(rate -> rate.getPriceBeforeTax() != null)
                                                .mapToDouble(PlacementRate.Rate::getPriceBeforeTax)
                                                .sum())
                                        .sum());
                    } else {
                        penaltyAmount = total
                                .multiply(BigDecimal.valueOf(travellinePenalty.getNights()))
                                .divide(BigDecimal.valueOf(days), RoundingMode.HALF_UP);
                    }
                    break;
                case FIRST_NIGHT:
                    BigDecimal nightPrice;
                    if (placements != null) {
                        nightPrice = BigDecimal.valueOf(
                                placements
                                        .stream()
                                        .filter(placementRate -> placementRate.getRates() != null
                                                && !placementRate.getRates().isEmpty()
                                                && placementRate.getRates().get(0).getPriceBeforeTax() != null)
                                        .mapToDouble(placementRate -> placementRate.getRates().get(0).getPriceBeforeTax())
                                        .sum());
                    } else {
                        nightPrice = total
                                .divide(BigDecimal.valueOf(days), RoundingMode.HALF_UP);
                    }
                    penaltyAmount = nightPrice
                            .multiply(BigDecimal.valueOf(travellinePenalty.getPercent()))
                            .divide(BigDecimal.valueOf(100), RoundingMode.HALF_UP);
                    break;
                case PREPAYMENT:
                case FULL_STAY:
                    penaltyAmount = total
                            .multiply(BigDecimal.valueOf(travellinePenalty.getPercent()))
                            .divide(BigDecimal.valueOf(100), RoundingMode.HALF_UP);
                    break;
                default:
                    throw new RuntimeException("Unknown CancellationPenaltyBasis");
            }

            if (placements != null && travellinePenalty.getAmount() != null && penaltyAmount.setScale(0,
                    RoundingMode.HALF_UP).compareTo(BigDecimal.valueOf(travellinePenalty.getAmount()).setScale(0,
                    RoundingMode.HALF_UP)) != 0) {
                log.warn("Rate plan {}: calculated penalty amount {} did not match provided amount {}",
                        ratePlanCode,
                        penaltyAmount.setScale(0, RoundingMode.HALF_UP).toString(),
                        BigDecimal.valueOf(travellinePenalty.getAmount()).setScale(0, RoundingMode.HALF_UP));
            }

            Instant endsAt = getInstant(travellinePenalty.getEnd(), zoneOffset);
            if (endsAt == null || endsAt.isAfter(startInstant)) {
                endsAt = startInstant;
            }
            Instant startsAt = getInstant(travellinePenalty.getStart(), zoneOffset);

            if (previousEnd != null && startsAt != null) {
                if (previousEnd.isBefore(startsAt)) {
                    throw new PartnerException(String.format("Rate plan %s: Gap between intervals found from %s till " +
                            "%s", ratePlanCode, previousEnd, startsAt));
                } else if (previousEnd.isAfter(startsAt)) {
                    throw new PartnerException(String.format("Rate plan %s: Intervals overlap from %s till %s",
                            ratePlanCode, startsAt, previousEnd));
                }
            }
            if (startsAt != null && startsAt.equals(endsAt)) {
                continue;
            }

            previousEnd = endsAt;

            BigDecimal currentPenaltyAmount;
            // workaround for CancelPenaltyBasis.PREPAYMENT, see TRAVELBACK-2749
            if (travellinePenalty.getAmount() == null || travellinePenalty.getBasis() == CancelPenaltyBasis.PREPAYMENT) {
                currentPenaltyAmount = penaltyAmount;
            } else {
                currentPenaltyAmount = BigDecimal.valueOf(travellinePenalty.getAmount());
            }
            if (previousPenaltyAmount != null && previousPenaltyAmount.compareTo(currentPenaltyAmount) > 0) {
                throw new PartnerException(String.format("Rate plan %s: previous penalty amount %s is greater then " +
                        "next penalty amount %s", ratePlanCode, previousPenaltyAmount, currentPenaltyAmount));
            }
            previousPenaltyAmount = currentPenaltyAmount;

            RefundType type;
            String penaltyCurrency = Currency.RUB.getValue();
            if (currentPenaltyAmount.compareTo(BigDecimal.ZERO) == 0) {
                type = RefundType.FULLY_REFUNDABLE;
                currentPenaltyAmount = null;
                penaltyCurrency = null;
            } else if (currentPenaltyAmount.setScale(0, RoundingMode.HALF_UP).compareTo(total) >= 0) { //TODO there
                // can be problems here with added services
                type = RefundType.NON_REFUNDABLE;
                currentPenaltyAmount = null;
                penaltyCurrency = null;
            } else {
                type = RefundType.REFUNDABLE_WITH_PENALTY;
            }
            if (startsAt != null) {
                Preconditions.checkState(startsAt.isBefore(endsAt), "Start date should be before end date");
            }
            Money ruleAmount = null;
            if (currentPenaltyAmount != null) {
                ruleAmount = Money.of(currentPenaltyAmount, ProtoCurrencyUnit.fromCurrencyCode(penaltyCurrency)); //TODO support different currencies
            }
            RefundRule nextRule = RefundRule.builder()
                    .startsAt(startsAt)
                    .endsAt(endsAt)
                    .type(type)
                    .penalty(ruleAmount)
                    .build();
            if (previousRule != null && penaltyAmountsAreEqual(previousRule, nextRule)) {
                rules.remove(rules.size() - 1);
                rules.add(RefundRule.builder()
                        .startsAt(previousRule.getStartsAt())
                        .endsAt(nextRule.getEndsAt())
                        .type(previousRule.getType())
                        .penalty(previousRule.getPenalty())
                        .build());
                previousRule = rules.get(rules.size() - 1);
            } else {
                rules.add(nextRule);
                previousRule = nextRule;
            }
        }
        RefundRule lastRule = rules.get(rules.size() - 1);
        if (lastRule.getEndsAt() != null) {
            if (lastRule.getEndsAt().compareTo(startInstant) < 0) {
                throw new PartnerException(String.format("Rate plan %s: Gap between intervals found from %s till %s",
                        ratePlanCode, previousEnd, startInstant));
            }
            if (lastRule.getType().equals(RefundType.NON_REFUNDABLE)) {
                rules.remove(rules.size() - 1);
                rules.add(RefundRule.builder().type(RefundType.NON_REFUNDABLE).startsAt(lastRule.getStartsAt()).build());
            } else {
                rules.add(RefundRule.builder().type(RefundType.NON_REFUNDABLE).startsAt(lastRule.getEndsAt()).build());
            }
        }
        builder.rules(rules);
        return builder.build();
    }

    private static boolean penaltyAmountsAreEqual(RefundRule rule1, RefundRule rule2) {
        if (!rule1.getType().equals(rule2.getType())) {
            return false;
        }
        if (rule1.getType().equals(RefundType.REFUNDABLE_WITH_PENALTY)) {
            return rule1.getPenalty().compareTo(rule2.getPenalty()) == 0;
        } else {
            return true;
        }
    }

    private static Instant getInstant(LocalDateTime dateTime, ZoneOffset zoneOffset) {
        if (dateTime == null) {
            return null;
        }
        return dateTime.atZone(zoneOffset).toInstant();
    }

}
