package ru.yandex.travel.hotels.common.partners.travelline;

import java.time.LocalDate;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.base.ForcedTestScenario;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.ReturnedErrorException;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingStatus;
import ru.yandex.travel.hotels.common.partners.travelline.model.CancelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ConfirmReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.Error;
import ru.yandex.travel.hotels.common.partners.travelline.model.ErrorType;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelChainDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelDetailsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventoryResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelStatusChangedResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ListHotelsResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.MakeExtraPaymentResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.ReadReservationResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.RespHotelReservation;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationRequest;
import ru.yandex.travel.hotels.common.partners.travelline.model.VerifyReservationResponse;

@Slf4j
public class WrappedTravellineClient implements TravellineClient {
    private final int MAX_MOCKED_REQUESTS = 100;
    private final TravellineClient wrappedClient;
    private final LRUMap forcedScenarios;

    public WrappedTravellineClient(TravellineClient wrappedClient) {
        this.wrappedClient = wrappedClient;
        forcedScenarios = new LRUMap();
    }


    @Override
    public CompletableFuture<HotelInfo> getHotelInfo(String hotelCode, String requestId) {
        return wrappedClient.getHotelInfo(hotelCode, requestId);
    }

    @Override
    public CompletableFuture<HotelOfferAvailability> findOfferAvailability(String hotelCode, LocalDate checkinDate,
                                                                           LocalDate checkoutDate, String requestId) {
        return wrappedClient.findOfferAvailability(hotelCode, checkinDate, checkoutDate, requestId);
    }

    @Override
    public CompletableFuture<VerifyReservationResponse> verifyReservation(VerifyReservationRequest request) {
        return wrappedClient.verifyReservation(request);
    }

    @Override
    public CompletableFuture<HotelReservationResponse> createReservation(HotelReservationRequest request) {
        var test = ForcedTestScenario.get(
                request.getHotelReservations().get(0).getRoomStays().get(0).getGuests().get(0).getFirstName(),
                request.getHotelReservations().get(0).getRoomStays().get(0).getGuests().get(0).getLastName());
        switch (test) {
            case PRICE_MISMATCH_ON_HOLD:
                logMock(test, "reservation");
                return mockReservation(request, true).thenApply(r ->
                {
                    forcedScenarios.put(r.getHotelReservations().get(0).getYandexNumber(),
                            Tuple2.tuple(test, r.getHotelReservations().get(0)));
                    return r;
                });
            case PRICE_MISMATCH_ON_CONFIRM:
            case NO_BOOKING_ON_CONFIRM:
                logMock(test, "reservation");
                return mockReservation(request, false).thenApply(r ->
                        {
                            forcedScenarios.put(r.getHotelReservations().get(0).getYandexNumber(),
                                    Tuple2.tuple(test, r.getHotelReservations().get(0)));
                            return r;
                        }
                );
            case SOLD_OUT_ON_HOLD:
                logMock(test, "reservation");
                return CompletableFuture.failedFuture(
                        new ReturnedErrorException(Collections.singletonList(
                                new Error(ErrorType.UNABLE_TO_PROCESS, "FORCED BY TEST NAME", null, null)))
                );
            default:
                return wrappedClient.createReservation(request);
        }
    }

    @Override
    public CompletableFuture<ConfirmReservationResponse> confirmReservation(String yandexNumber,
                                                                            String transactionNumber, Money amount) {
        var tuple = forcedScenarios.get(yandexNumber);
        if (tuple == null) {
            return wrappedClient.confirmReservation(yandexNumber, transactionNumber, amount);
        } else {
            var reservation = tuple.get2();
            switch (tuple.get1()) {
                case PRICE_MISMATCH_ON_CONFIRM:
                    logMock(tuple.get1(), "confirmation");
                    var patchedTotal = reservation.getTotal().toBuilder()
                            .priceAfterTax(reservation.getTotal().getPriceAfterTax() * 1.1)
                            .priceBeforeTax(reservation.getTotal().getPriceBeforeTax() * 1.1)
                            .build();
                    var patchedReservation = reservation.toBuilder()
                            .total(patchedTotal)
                            .status(BookingStatus.CONFIRMED)
                            .build();
                    ConfirmReservationResponse patchedResponse = new ConfirmReservationResponse();
                    patchedResponse.setHotelReservations(List.of(patchedReservation));
                    return CompletableFuture.completedFuture(patchedResponse);
                case NO_BOOKING_ON_CONFIRM:
                    logMock(tuple.get1(), "confirmation");
                    var cancelledReservation = reservation.toBuilder()
                            .status(BookingStatus.CANCELLED)
                            .build();
                    ConfirmReservationResponse cancelledResponse = new ConfirmReservationResponse();
                    cancelledResponse.setHotelReservations(List.of(cancelledReservation));
                    return CompletableFuture.completedFuture(cancelledResponse);
                default:
                    throw new RuntimeException("We should not be here");
            }
        }
    }

    @Override
    public CompletableFuture<MakeExtraPaymentResponse> makeExtraPayment(String yandexNumber, String transactionNumber,
                                                                        Money amount) {
        return wrappedClient.makeExtraPayment(yandexNumber, transactionNumber, amount);
    }

    @Override
    public CompletableFuture<CancelReservationResponse> cancelReservation(String yandexNumber) {
        var tuple = forcedScenarios.get(yandexNumber);
        if (tuple == null) {
            return wrappedClient.cancelReservation(yandexNumber);
        } else {
            var reservation = tuple.get2();
            if (tuple.get1() == ForcedTestScenario.PRICE_MISMATCH_ON_CONFIRM || tuple.get1() == ForcedTestScenario.PRICE_MISMATCH_ON_HOLD) {
                logMock(tuple.get1(), "cancellation");
                var patchedReservation = reservation.toBuilder()
                        .status(BookingStatus.CANCELLED)
                        .build();
                CancelReservationResponse response = new CancelReservationResponse();
                response.setHotelReservations(List.of(patchedReservation));
                return CompletableFuture.completedFuture(response);
            } else {
                throw new RuntimeException("We should not be here");
            }
        }
    }

    @Override
    public CompletableFuture<ReadReservationResponse> readReservation(String yandexNumber) {
        var tuple = forcedScenarios.get(yandexNumber);
        if (tuple == null) {
            return wrappedClient.readReservation(yandexNumber);
        } else {
            var reservation = tuple.get2();
            logMock(tuple.get1(), "read");
            ReadReservationResponse response = new ReadReservationResponse();
            response.setHotelReservations(List.of(reservation));
            return CompletableFuture.completedFuture(response);
        }
    }

    @Override
    public CompletableFuture<HotelInventoryResponse> getHotelInventory(String hotelCode) {
        return wrappedClient.getHotelInventory(hotelCode);
    }

    @Override
    public CompletableFuture<ListHotelsResponse> listHotels() {
        return wrappedClient.listHotels();
    }

    @Override
    public CompletableFuture<HotelStatusChangedResponse> notifyHotelStatusChanged(String hotelCode) {
        return wrappedClient.notifyHotelStatusChanged(hotelCode);
    }

    @Override
    public CompletableFuture<HotelDetailsResponse> getHotelDetails(String hotelCode) {
        return wrappedClient.getHotelDetails(hotelCode);
    }

    @Override
    public CompletableFuture<HotelChainDetailsResponse> getHotelChainDetails(String inn) {
        return wrappedClient.getHotelChainDetails(inn);
    }

    private void logMock(ForcedTestScenario scenario, String call) {
        log.warn(String.format("A '%s' scenario forced during '%s' call , will do no real API call, " +
                "will return a mocked object instead", scenario, call));
    }

    private CompletableFuture<HotelReservationResponse> mockReservation(HotelReservationRequest request,
                                                                        boolean increasePrice) {
        String mockedNumber = ProtoUtils.randomId();
        return wrappedClient.verifyReservation(
                VerifyReservationRequest.builder()
                        .hotelReservations(request.getHotelReservations())
                        .build())
                .thenApply(verifyResponse -> {
                    Preconditions.checkArgument(verifyResponse.getHotelReservations().size() == 1,
                            "Unexpected number of hotel reservations");
                    var verifiedReservation = verifyResponse.getHotelReservations().get(0);
                    RespHotelReservation.RespHotelReservationBuilder builder = verifiedReservation.toBuilder()
                            .reservationNumber(mockedNumber)
                            .status(BookingStatus.PENDING);
                    if (increasePrice) {
                        var returnedTotal = verifyResponse.getHotelReservations().get(0).getTotal();
                        var patchedTotal = returnedTotal.toBuilder()
                                .priceAfterTax(returnedTotal.getPriceAfterTax() * 1.1)
                                .priceBeforeTax(returnedTotal.getPriceBeforeTax() * 1.1)
                                .build();
                        builder.total(patchedTotal);
                    }
                    HotelReservationResponse res = new HotelReservationResponse();
                    res.setHotelReservations(List.of(builder.build()));
                    return res;
                });
    }

    class LRUMap extends LinkedHashMap<String, Tuple2<ForcedTestScenario, RespHotelReservation>> {
        @Override
        protected boolean removeEldestEntry(Map.Entry<String, Tuple2<ForcedTestScenario, RespHotelReservation>> eldest) {
            return this.size() >= MAX_MOCKED_REQUESTS;
        }
    }
}
