package ru.yandex.travel.hotels.common.partners.travelline.model.dto;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import lombok.Getter;

import ru.yandex.travel.hotels.common.partners.travelline.model.AgeGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.Service;

public class HotelInfoMappings {
    private final String hotelCode;
    private final Map<String, RoomType> roomTypesByCode;
    private final Map<String, RatePlan> ratePlansByCode;
    private final Map<String, Service> servicesByCode;
    private final Map<String, AgeGroup> ageGroupsByCode;

    @Getter
    private final Map<Integer, AgeGroup> ageGroupsByAge;

    public HotelInfoMappings(HotelInfo hotelInfo) {
        this.hotelCode = hotelInfo.getHotel().getCode();
        this.roomTypesByCode =
                ImmutableMap.copyOf(
                        hotelInfo.getHotel().getRoomTypes().stream()
                                .collect(Collectors.toMap(RoomType::getCode, Function.identity())));
        this.ratePlansByCode =
                ImmutableMap.copyOf(
                        hotelInfo.getHotel().getRatePlans().stream()
                                .collect(Collectors.toMap(RatePlan::getCode, Function.identity())));
        this.servicesByCode = ImmutableMap.copyOf(
                hotelInfo.getHotel().getServices().stream()
                        .collect(Collectors.toMap(Service::getCode, Function.identity())));

        Map<Integer, AgeGroup> ageGroupByAge = new HashMap<>();
        Map<String, AgeGroup> ageGroupByCode = new HashMap<>();
        hotelInfo.getHotel().getAgeGroups().forEach(ageGroup -> {
            ageGroupByCode.put(ageGroup.getCode(), ageGroup);
            IntStream.range(ageGroup.getMinAge(), ageGroup.getMaxAge() + 1).forEach(i -> {
                Preconditions.checkArgument(!ageGroupByAge.containsKey(i), "Overlapping age groups in hotel info");
                ageGroupByAge.put(i, ageGroup);
            });
        });
        this.ageGroupsByAge = ImmutableMap.copyOf(ageGroupByAge);
        this.ageGroupsByCode = ImmutableMap.copyOf(ageGroupByCode);
    }

    private <T> T get(Map<String, T> map, String dataType, String key) {
        if (!map.containsKey(key)) {
            throw new MissingHotelDataException(dataType, key);
        } else {
            return map.get(key);
        }
    }


    public RoomType getRoomType(String code) {
        return get(roomTypesByCode, "RoomType", code);
    }

    public RatePlan getRatePlan(String code) {
        return get(ratePlansByCode, "RatePlan", code);
    }

    public Service getServiceByCode(String code) {
        return get(servicesByCode, "Service", code);
    }

    public AgeGroup getAgeGroupsByCode(String code) {
        return get(ageGroupsByCode, "AgeGroup", code);
    }

    public class MissingHotelDataException extends RuntimeException {
        public MissingHotelDataException(String dataType, String dataCode) {
            super(String.format("Could not find %s with code %s for hotel %s", dataType, dataCode, hotelCode));
        }
    }


}
