package ru.yandex.travel.hotels.common.partners.travelline.model.dto;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonPOJOBuilder;
import com.google.common.base.Preconditions;
import lombok.AccessLevel;
import lombok.Builder;
import lombok.Getter;
import lombok.Value;
import lombok.experimental.NonFinal;

import ru.yandex.travel.hotels.common.pansions.PansionUnifier;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.ServiceKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayDates;
import ru.yandex.travel.hotels.common.partners.travelline.placements.PlacementSet;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.proto.EPansionType;

@Value
@Builder(toBuilder = true)
@JsonDeserialize(builder = OfferDto.OfferDtoBuilder.class)
public class OfferDto {
    private HotelDto hotel;
    private RoomType roomType;
    private StayDates stayDates;
    private int quota;
    private RatePlan ratePlan;
    private Map<String, ServiceDto> services;
    private List<PlacementSet> possiblePlacements;


    @NonFinal
    @JsonIgnore
    @Getter(value = AccessLevel.NONE)
    private volatile Integer bestPlacementIndex;

    @NonFinal
    @JsonIgnore
    @Getter(value = AccessLevel.NONE)
    private volatile EPansionType pansionType;


    @JsonIgnore
    public int getBestPlacementIndex() {
        if (bestPlacementIndex == null) {
            Preconditions.checkState(possiblePlacements.size() > 0);
            double bestPrice = possiblePlacements.get(0).getTotalCost();
            int bestIndex = 0;
            for (int i = 1; i < possiblePlacements.size(); i++) {
                if (possiblePlacements.get(i).getTotalCost() < bestPrice) {
                    bestIndex = i;
                }
            }
            bestPlacementIndex = bestIndex;
        }
        return bestPlacementIndex;
    }


    @JsonIgnore
    public EPansionType getPansionType() {
        if (pansionType == null) {
            pansionType = PansionUnifier.combine(
                    services.values().stream()
                            .filter(s -> s.getInfo().getKind() == ServiceKind.MEAL && s.getConditions().isInclusive() && s.getInfo().getMealPlanType() != null)
                            .map(s -> s.getInfo().getMealPlanType().getEPansionType())
                            .collect(Collectors.toList()));
        }
        return pansionType;
    }

    @JsonIgnore
    public double getBestPrice() {
        return possiblePlacements.get(getBestPlacementIndex()).getTotalCost();
    }

    @JsonIgnore
    public RefundRules getRefundRules() {
        return possiblePlacements.get(getBestPlacementIndex()).getRefundRules();
    }

    @JsonPOJOBuilder(withPrefix = "")
    public static class OfferDtoBuilder {
    }
}
