package ru.yandex.travel.hotels.common.partners.travelline.placements;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.travel.hotels.common.partners.travelline.model.AgeGroup;

class Capacity {
    private final Guest[] occupants;
    private final int numPrimary;
    private final int numExtra;
    private final int numNoBed;
    private final List<AgeGroup> noBedPlacementAgeGroups;

    private boolean adultsExist;
    private boolean childrenExist;
    private int takenPrimaries;


    public Capacity(int numPrimary, int numExtra, int numNoBed,  List<AgeGroup> noBedPlacementAgeGroups) {
        this(numPrimary, numExtra, numNoBed, noBedPlacementAgeGroups, new Guest[numPrimary + numExtra + numNoBed], false, false, 0);
    }

    private Capacity(int numPrimary, int numExtra, int numNoBed,  List<AgeGroup> noBedPlacementAgeGroups, Guest[] occupants, boolean adultsExist,
                     boolean childrenExist,
                     int takenPrimaries) {
        this.occupants = occupants;
        this.numPrimary = numPrimary;
        this.numExtra = numExtra;
        this.numNoBed = numNoBed;
        this.noBedPlacementAgeGroups = noBedPlacementAgeGroups;
        this.adultsExist = adultsExist;
        this.childrenExist = childrenExist;
        this.takenPrimaries = takenPrimaries;
    }

    public int getNumPlaces() {
        return numPrimary + numNoBed + numExtra;
    }

    public int getNumUnprocessedPlaces() {
        return (int)Arrays.stream(occupants).filter(Objects::isNull).count();
    }

    protected Integer getFirstFreeSlot() {
        for (int i=0; i<occupants.length; i++) {
            if (occupants[i] == null) {
                return i;
            }
        }
        return null;
    }

    public boolean verify() {
        return adultsExist;
    }

    public Capacity place(Guest occupant) {
        var index = getFirstFreeSlot();
        Preconditions.checkState(index!=null, "No free slots in capacity");
        if (occupant.isEmpty()) {
            return cloneWith(index, occupant);
        } else {
            if (!adultsExist && !occupant.isAdult()) {
                // нельзя селить детей до взрослых
                return null;
            }
            if (childrenExist && occupant.isAdult()) {
                // нельзя селить взрослых после детей
                return null;
            }

            switch (getPlaceType(index)) {
                case Primary:
                    return cloneWith(index, occupant);
                case Extra:
                    if (takenPrimaries < numPrimary) {
                        // Нельзя селить на дополнительные места при свободных основных
                        return null;
                    }
                    return cloneWith(index, occupant);
                case NoBed:
                    if (occupant.isAdult()) {
                        // Взрослых "без места" селить нельзя
                        return null;
                    }
                    if (noBedPlacementAgeGroups.stream().noneMatch(ag -> occupant.getAge() >= ag.getMinAge() && occupant.getAge() <= ag.getMaxAge())) {
                        // возраст ребенка не подпадает ни под одну is age group доступных для
                        // child_band_without_bed
                        return null;
                    }
                    return cloneWith(index, occupant);
                default:
                    throw new AssertionError("We should not be here");
            }
        }
    }


    private Capacity cloneWith(int index, Guest occupant) {
        var occupants = this.occupants.clone();
        occupants[index] = occupant;
        return new Capacity(this.numPrimary, this.numExtra, this.numNoBed, this.noBedPlacementAgeGroups, occupants,
                adultsExist || occupant.isAdult(),
                childrenExist || (!occupant.isAdult() && !occupant.isEmpty()),
                takenPrimaries + ((getPlaceType(index) == PlaceType.Primary && !occupant.isEmpty()) ? 1 : 0));
    }

    private PlaceType getPlaceType(int index) {
        Preconditions.checkArgument(index < getNumPlaces() && index >= 0, "Place index out of range");
        if (index < numPrimary) {
            return PlaceType.Primary;
        } else if (index - numPrimary < numExtra) {
            return PlaceType.Extra;
        } else if (index - (numPrimary + numExtra) < numNoBed) {
            return PlaceType.NoBed;
        }
        throw new AssertionError("We should not be here");
    }

    public Allocation getAllocation() {
        int primaryAdults = 0;
        List<Integer> primaryChildren = new ArrayList<>();
        int extraAdults = 0;
        List<Integer> extraChildren = new ArrayList<>();
        List<Integer> childrenWithNoPlaces = new ArrayList<>();
        for (int i = 0; i < this.getNumPlaces(); i++) {
            if (this.occupants[i].isEmpty()) {
                continue;
            }
            switch (getPlaceType(i)) {
                case Primary:
                    if (occupants[i].isAdult()) {
                        primaryAdults++;
                    } else {
                        primaryChildren.add(occupants[i].getIndex());
                    }
                    break;
                case Extra:
                    if (occupants[i].isAdult()) {
                        extraAdults++;
                    } else {
                        extraChildren.add(occupants[i].getIndex());
                    }
                    break;
                case NoBed:
                    childrenWithNoPlaces.add(occupants[i].getIndex());
                    break;
                default:
                    throw new AssertionError("We should not be here");
            }
        }
        return new Allocation(primaryAdults, primaryChildren, extraAdults, extraChildren, childrenWithNoPlaces);
    }
}
