package ru.yandex.travel.hotels.common.partners.travelline.placements;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.hotels.common.partners.travelline.model.AgeGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.GuestCount;
import ru.yandex.travel.hotels.common.partners.travelline.model.GuestCountInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.GuestPlacementKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.Placement;
import ru.yandex.travel.hotels.common.token.Occupancy;

@Slf4j
public class PlacementGenerator {
    public static Set<Allocation> generateAllocation(Occupancy occupancy, int numPrimary, int numExtra, int numNoBed,
                                                     List<AgeGroup> noBedPlacementAgeGroups) {
        Set<Allocation> res = new HashSet<>();
        Capacity capacity = new Capacity(numPrimary, numExtra, numNoBed, noBedPlacementAgeGroups);
        List<Guest> guests = Guest.fromOccupancy(occupancy);
        if (capacity.getNumPlaces() < guests.size()) {
            log.warn("Required number of guests will never fit in here");
            return res;
        }
        Capacity prefilled = prefill(capacity, guests);
        AtomicInteger iterations = new AtomicInteger(0);
        generateImpl(prefilled, guests.stream().filter(g -> !g.isAdult()).collect(Collectors.toList()), res,
                iterations);
        return res;
    }

    private static Capacity prefill(Capacity capacity, List<Guest> guests) {
        for (Guest g : guests) {
            if (g.isAdult()) {
                var newCapacity = capacity.place(g);
                if (newCapacity == null) {
                    return capacity;
                } else {
                    capacity = newCapacity;
                }
            }
        }
        return capacity;
    }


    private static void generateImpl(Capacity capacity, List<Guest> guests,
                                     Set<Allocation> foundAllocation, AtomicInteger iteration) {
        /*
         * Общая логика у этого метода следующая: на вход приходит номер, в котором кто-то уже (возможно) поселен,
         * список гостей, которых осталось поселить и индекс очередного свободного места в номере.
         * Дальше он пытается каждого гостя попробовать поселить на это свободное место. Это может как получиться, так и
         * не получиться, если очередной гость не подходит к этому месту. Успешное заселение создает копию номера, в
         * который на текущее место доселен гость, после чего этот метод запускается рекурсивно, чтобы сгенерировать все
         * возможные комбинации, которые получатся, если мы займем текущее место этим гостем. В рекурсию передается уже
         * та копия номера, которую вернула операция поселения, уменьшенный на свежеразмещенного гостя список гостей - и
         * новый индекс свободного места, указывающий на следующее.
         * Перепробовав поселить на текущее очередное место всех гостей, метод пробует краевой случай: помечает место
         * как оставшееся свободным ("селит" на него специального, "пустого" гостя) - и точно так же рекурсивно
         * запускает себя дальше, тем самым пытаясь сгенерировать размещения, которые получатся, если на текущее место
         * никого из оставшихся гостей не селить. Этот вариант запускается только в том случае, если количество
         * необработанных мест больше, чем количество нерасселенных гостей, в противном случае пропуск места заведомо
         * бесполезен: мы все равно не сможем расселить всех.
         * Успешный выход из рекурсии происходит после размещения гостя на последнее свободное место: в этой ситуации
         * получается, что все места обработаны: на них либо кто-то размещен, либо они явно оставлены пустыми. Тогда
         * полученный вариант размещения считается подходящим запросу и добавляется в итоговый сет найденных размещений.
         * Еще есть вариант, что ни одного из гостей нельзя поселить на текущее место, и пропускать его тоже нельзя.
         * Тогда происходит выход из рекурсии с неуспехом - никакого варианта доселения не предлагается.
         *
         * Главный недостаток этого метода: комбинаторный взрыв, случающийся при большом количестве гостей и мест.
         * Количество возможных расселений N человек по M местам равно M!/(M-N)!
         * Практика (TRAVELBACK-1126) показала, что попытка расселить 8 человек по 24 местам занимает несколько часов
         * вычислений, выжирает целиком одно ядро CPU и приводит к прочим неприятным последствиям.
         * При этом очевидно, что этот метод излишне универсален: он генерирует комбинации, считая всех взрослых гостей
         * и все места уникальными. Для Travelline это излишне, его логика позволяет расселить всех взрослых гостей
         * сначала на основные места, потом — на дополнительные, а различные комбинации возможны только для расселение
         * детей. Поэтому перед запуском этого метода имеет смысл "предрасселить" всех взрослых гостей на основные и
         * дополнительные места, а комбинации генерить только для детей.
         *
         * В принципе, правильное решение — свести эту задачу к задаче о назначениях и решать через поиск максимального
         * потока минимальной стоимости. Но это оставим до следующей версии.
         * */

        if (capacity.getNumUnprocessedPlaces() == 0) {
            int total = iteration.incrementAndGet();
            if (total >= 1000000) {
                throw new InvalidPlacementAllocationException("Too many allocation options");
            }
            // Прошлись по всем местам — выходим из рекурсии
            Preconditions.checkState(guests.isEmpty(),
                    "Some guests left after all the places have been assigned. This should not happen");
            foundAllocation.add(capacity.getAllocation());
        } else {
            for (Guest guest : guests) {
                Capacity withAddedGuest = capacity.place(guest);
                if (withAddedGuest != null) {
                    generateImpl(withAddedGuest, guests.stream().filter(o -> o != guest).collect(Collectors.toList()),
                            foundAllocation, iteration);
                }
            }
            if (capacity.getNumUnprocessedPlaces() > guests.size()) {
                Capacity occupied = capacity.place(Guest.empty());
                if (occupied != null) {
                    generateImpl(occupied, guests, foundAllocation, iteration);
                }
            }
        }
    }

    public static PlacementSet mapPlacementsForAllocation(List<Placement> placements, Allocation allocation,
                                                          Map<Integer, AgeGroup> ageGroupByAgeMap,
                                                          Occupancy occupancy, boolean sanatorium,
                                                          boolean separateCapacities,
                                                          boolean alwaysAllowChildrenAsAdults) {

        List<Placement> selectedPlacements = new ArrayList<>();
        GuestCountInfo.GuestCountInfoBuilder guestCountInfoBuilder = GuestCountInfo.builder();
        int[] placementIndexes = new int[occupancy.getAdults() + occupancy.getChildren().size()];
        Arrays.fill(placementIndexes, -1);


        // Строим вспомогательные словари
        Map<Tuple2<String, Integer>, Placement> primaryChildPlacementByAgeGroupCodeAndCapacityMap = new HashMap<>();
        Map<Tuple2<String, Integer>, Placement> extraChildPlacementByAgeGroupCodeAndCapacityMap = new HashMap<>();
        Map<String, Placement> noBedChildPlacementByAgeGroupCodeMap = new HashMap<>();
        for (Placement placement : placements) {
            if (placement.getKind() == GuestPlacementKind.CHILD) {
                String ageGroupCode = String.valueOf(placement.getAgeGroup());
                Tuple2<String, Integer> key = Tuple2.tuple(ageGroupCode, placement.getCapacity());
                Preconditions.checkArgument(placement.getCapacity() == 1 || sanatorium, "Unexpected capacity for " +
                        "non-sanatorium");
                Preconditions.checkArgument(!primaryChildPlacementByAgeGroupCodeAndCapacityMap.containsKey(key),
                        "Duplicate age-group-code and capacity for primary children placements");
                primaryChildPlacementByAgeGroupCodeAndCapacityMap.put(key, placement);
            }
            if (placement.getKind() == GuestPlacementKind.EXTRA_CHILD) {
                String ageGroupCode = String.valueOf(placement.getAgeGroup());
                Tuple2<String, Integer> key = Tuple2.tuple(ageGroupCode, placement.getCapacity());
                Preconditions.checkArgument(placement.getCapacity() == 1 || sanatorium, "Unexpected capacity for " +
                        "non-sanatorium");
                Preconditions.checkArgument(!extraChildPlacementByAgeGroupCodeAndCapacityMap.containsKey(key),
                        "Duplicate age-group-code and capacity for extra children placements");
                extraChildPlacementByAgeGroupCodeAndCapacityMap.put(key, placement);
            }
            if (placement.getKind() == GuestPlacementKind.CHILD_BAND_WITHOUT_BED) {
                String ageGroupCode = String.valueOf(placement.getAgeGroup());
                Preconditions.checkArgument(!noBedChildPlacementByAgeGroupCodeMap.containsKey(ageGroupCode),
                        "Duplicate age-group-code for no-bed children placements");
                noBedChildPlacementByAgeGroupCodeMap.put(ageGroupCode, placement);
            }
        }

        if (sanatorium) {
            arrangeBedPlacementsSanatorium(placements, allocation, ageGroupByAgeMap, occupancy, selectedPlacements,
                    guestCountInfoBuilder, placementIndexes, primaryChildPlacementByAgeGroupCodeAndCapacityMap,
                    true, separateCapacities);
            arrangeBedPlacementsSanatorium(placements, allocation, ageGroupByAgeMap, occupancy, selectedPlacements,
                    guestCountInfoBuilder, placementIndexes, extraChildPlacementByAgeGroupCodeAndCapacityMap,
                    false, separateCapacities);
        } else {
            // Размещаем на основные места
            arrangeBedPlacementsNonSanatorium(placements, allocation, ageGroupByAgeMap, occupancy, selectedPlacements,
                    guestCountInfoBuilder, placementIndexes, primaryChildPlacementByAgeGroupCodeAndCapacityMap, true,
                    alwaysAllowChildrenAsAdults);
            // Размещаем на дополнительные
            arrangeBedPlacementsNonSanatorium(placements, allocation, ageGroupByAgeMap, occupancy, selectedPlacements,
                    guestCountInfoBuilder, placementIndexes,
                    extraChildPlacementByAgeGroupCodeAndCapacityMap, false, alwaysAllowChildrenAsAdults);
        }
        // Размещаем на без-места
        arrangeNoBedPlacements(allocation, ageGroupByAgeMap, occupancy, selectedPlacements, guestCountInfoBuilder,
                placementIndexes, noBedChildPlacementByAgeGroupCodeMap);
        if (!Arrays.stream(placementIndexes).allMatch(i -> i != -1)) {
            throw new InvalidPlacementAllocationException("Some guests are not placed");
        }
        selectedPlacements.sort(Comparator.comparingInt(Placement::getIndex));

        return PlacementSet.builder()
                .placements(selectedPlacements)
                .guestCountInfo(guestCountInfoBuilder.build())
                .guestPlacementIndexes(Arrays.stream(placementIndexes).boxed().collect(Collectors.toList()))
                .build();
    }


    private static void arrangeNoBedPlacements(Allocation allocation, Map<Integer, AgeGroup> ageGroupByAgeMap,
                                               Occupancy occupancy, List<Placement> resultPlacements,
                                               GuestCountInfo.GuestCountInfoBuilder guestCountInfoBuilder,
                                               int[] placementIndexes,
                                               Map<String, Placement> noBedChildPlacementByAgeGroupCodeMap) {
        List<Tuple2<Placement, Integer>> noBedChildren = new ArrayList<>();
        for (int index : allocation.getChildrenWithNoPlaceIndexes()) {
            int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
            Placement placement = null;
            AgeGroup ageGroup = ageGroupByAgeMap.get(childAge);
            if (ageGroup != null) {
                placement = noBedChildPlacementByAgeGroupCodeMap.get(ageGroup.getCode());
            }
            if (placement != null) {
                // Для ребенка есть размещение-без-места подходищее по возрасту
                noBedChildren.add(Tuple2.tuple(placement, index));
            } else {
                throw new InvalidPlacementAllocationException("Unable to place child-without-bed: incompatible age " + childAge);
            }
        }

        for (Tuple2<Placement, Integer> noBedChild : noBedChildren) {
            int index = noBedChild.get2();
            int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
            Placement childPlacement = noBedChild.get1().toBuilder().index(resultPlacements.size()).build();
            resultPlacements.add(childPlacement);
            guestCountInfoBuilder.guestCount(GuestCount.builder()
                    .age(childAge)
                    .count(1)
                    .ageQualifyingCode("child")
                    .placementIndex(childPlacement.getIndex())
                    .build());
            Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
            placementIndexes[index] = childPlacement.getIndex();
        }
    }

    private static boolean arrangeBedPlacementsSanatorium(List<Placement> placements,
                                                          Allocation allocation,
                                                          Map<Integer, AgeGroup> ageGroupByAgeMap,
                                                          Occupancy occupancy,
                                                          List<Placement> resultPlacements,
                                                          GuestCountInfo.GuestCountInfoBuilder guestCountInfoBuilder,
                                                          int[] placementIndexes,
                                                          Map<Tuple2<String, Integer>, Placement> childPlacementByAgeGroupCodeAndCapacityMap,
                                                          boolean primary, boolean separateCapacities) {

        GuestPlacementKind adultKind;
        int adults;
        int adultCapacity;
        int childCapacity;
        int startIndex;
        List<Integer> children;
        if (primary) {
            startIndex = 0;
            adultKind = GuestPlacementKind.ADULT;
            adults = allocation.getPrimaryAdults();
            children = allocation.getPrimaryChildrenIndexes();
        } else {
            startIndex = allocation.getPrimaryAdults() + allocation.getPrimaryChildrenIndexes().size();
            adultKind = GuestPlacementKind.EXTRA_ADULT;
            adults = allocation.getExtraAdults();
            children = allocation.getExtraChildrenIndexes();
        }
        if (separateCapacities) {
            adultCapacity = adults;
            childCapacity = children.size();
        } else {
            adultCapacity = childCapacity = adults + children.size();
        }
        var adultPlacement = placements.stream()
                .filter(p -> p.getKind() == adultKind && (p.getCapacity() == adultCapacity))
                .min(Comparator.comparingInt(Placement::getCapacity))
                .orElse(null);
        for (int i = 0; i < adults; i++) {
            if (adultPlacement == null) {
                return false;
            }
            var placement = adultPlacement.toBuilder()
                    .index(resultPlacements.size())
                    .build();
            resultPlacements.add(placement);
            placementIndexes[startIndex + i] = placement.getIndex();
            guestCountInfoBuilder.guestCount(
                    GuestCount.builder()
                            .ageQualifyingCode("adult")
                            .count(1)
                            .placementIndex(placement.getIndex())
                            .build());
        }
        int j = 0;
        for (int index : children) {
            int childIndex = index - occupancy.getAdults();
            int childAge = occupancy.getChildren().get(childIndex);
            var ag = ageGroupByAgeMap.get(childAge);
            if (ag == null) {
                return false;
            }
            var childPlacement = childPlacementByAgeGroupCodeAndCapacityMap.get(Tuple2.tuple(ag.getCode(),
                    childCapacity));
            if (childPlacement == null) {
                return false;
            }
            var placement = childPlacement.toBuilder()
                    .index(resultPlacements.size())
                    .build();
            placementIndexes[startIndex + adults + j] = placement.getIndex();
            j++;
            guestCountInfoBuilder.guestCount(
                    GuestCount.builder()
                            .ageQualifyingCode("child")
                            .count(1)
                            .age(childAge)
                            .placementIndex(placement.getIndex())
                            .build());
            resultPlacements.add(placement);
        }
        return true;
    }

    private static void arrangeBedPlacementsNonSanatorium(List<Placement> placements,
                                                          Allocation allocation,
                                                          Map<Integer, AgeGroup> ageGroupByAgeMap,
                                                          Occupancy occupancy,
                                                          List<Placement> resultPlacements,
                                                          GuestCountInfo.GuestCountInfoBuilder guestCountInfoBuilder,
                                                          int[] placementIndexes,
                                                          Map<Tuple2<String, Integer>, Placement> childPlacementByAgeGroupCodeAndCapacityMap,
                                                          boolean primary, boolean alwaysAllowChildrenAsAdults) {

        List<Integer> childrenIndexes;
        int numAdults;
        int firstAdultIndex;
        int firstChildIndex = 0;
        GuestPlacementKind adultPlacementKind;
        Integer minAge = ageGroupByAgeMap.keySet().stream().min(Comparator.comparing(Integer::valueOf)).orElse(null);

        if (primary) {
            childrenIndexes = allocation.getPrimaryChildrenIndexes();
            numAdults = allocation.getPrimaryAdults();
            firstAdultIndex = 0;
            adultPlacementKind = GuestPlacementKind.ADULT;

        } else {
            childrenIndexes = allocation.getExtraChildrenIndexes();
            numAdults = allocation.getExtraAdults();
            firstAdultIndex = allocation.getPrimaryAdults();
            adultPlacementKind = GuestPlacementKind.EXTRA_ADULT;
        }

        // Для начала находим детские места для детей и определяем, каких детей будем селить на взрослые места
        List<Tuple2<Placement, Integer>> foundChildren = new ArrayList<>();
        List<Integer> childrenAsAdults = new ArrayList<>();
        for (int index : childrenIndexes) {
            int childIndex = index - occupancy.getAdults();
            int childAge = occupancy.getChildren().get(childIndex);
            Placement placement = null;
            AgeGroup ageGroup = ageGroupByAgeMap.get(childAge);
            if (ageGroup != null) {
                placement = childPlacementByAgeGroupCodeAndCapacityMap.get(Tuple2.tuple(ageGroup.getCode(), 1));
            }
            if (placement != null) {
                // Для ребенка есть детское место подходищее по возрасту
                foundChildren.add(Tuple2.tuple(placement, index));
            } else {
                // Для ребенка нет детских мест или они кончились, или вообще нет такой age-группы - такого ребенка
                // надо селить на взрослое место
                if (minAge == null || minAge <= childAge || alwaysAllowChildrenAsAdults) {
                    // возрастные группы не заданы вообще, или в них есть дети младше указанного
                    childrenAsAdults.add(index);
                } else {
                    // возраст ребенка меньше минимального, указанного отельером, таких на взрослые места не селим
                    throw new InvalidPlacementAllocationException(
                            String.format("Unable to place child of age %d to adult place", childAge));
                }
            }
        }

        // Теперь нам надо найти правильный adult-плейсмент
        Placement adultPlacement;
        if (primary) {
            // Нужно capacity не меньше чем количество взрослых + количество детей которых селим на взрослые места
            int requiredAdultCapacity = numAdults + childrenAsAdults.size();
            adultPlacement = placements.stream()
                    .filter(p -> p.getKind() == adultPlacementKind && (p.getCapacity() >= requiredAdultCapacity))
                    .min(Comparator.comparingInt(Placement::getCapacity))
                    .orElse(null);

            if (adultPlacement == null) {
                throw new InvalidPlacementAllocationException(
                        String.format("Not enough primary adult capacity: %d required", requiredAdultCapacity));
            }
        } else {
            // Цитата от партнера: "В гостиничной интеграции (только она у нас с вами на текущий момент)
            // можно игнорировать capacity у extra мест."
            // Поэтому ищем единственный adult placement
            var allAdultPlacementsIgnoringCapacity = placements.stream().filter(p -> p.getKind() == adultPlacementKind)
                    .collect(Collectors.toList());
            Preconditions.checkState(allAdultPlacementsIgnoringCapacity.size() <= 1,
                    "Too many extra adult placements");
            if (allAdultPlacementsIgnoringCapacity.size() == 0) {
                if (numAdults + childrenAsAdults.size() > 0) {
                    throw new InvalidPlacementAllocationException("No extra adult placements");
                } else {
                    adultPlacement = null;
                }
            } else {
                adultPlacement = allAdultPlacementsIgnoringCapacity.get(0);
            }
        }

        if (primary) {
            // Добавляем placement в результаты
            adultPlacement = adultPlacement.toBuilder().index(resultPlacements.size()).build();
            resultPlacements.add(adultPlacement);
            // Добавляем один GuestCount на всех взрослых на основных местах
            guestCountInfoBuilder.guestCount(
                    GuestCount.builder()
                            .ageQualifyingCode("adult")
                            .count(numAdults)
                            .placementIndex(adultPlacement.getIndex())
                            .build()
            );
            // Фиксируем привязки индекса плейсмента к гостям
            for (int i = 0; i < numAdults; i++) {
                int index = firstAdultIndex + i;
                Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
                placementIndexes[index] = adultPlacement.getIndex();
            }
            // Расселяем детей на основные взрослые места
            for (int index : childrenAsAdults) {
                int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
                guestCountInfoBuilder.guestCount(GuestCount.builder()
                        .age(childAge)
                        .count(1)
                        .ageQualifyingCode("child")
                        .placementIndex(adultPlacement.getIndex())
                        .build());
                Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
                placementIndexes[index] = adultPlacement.getIndex();
            }
        } else {
            // Для каждого взрослого гостя делаем копию плейсмента с новым индексом
            for (int i = 0; i < numAdults; i++) {
                var adultPlacementCopy = adultPlacement.toBuilder().index(resultPlacements.size()).build();
                resultPlacements.add(adultPlacementCopy);
                guestCountInfoBuilder.guestCount(
                        GuestCount.builder()
                                .ageQualifyingCode("adult")
                                .count(1)
                                .placementIndex(adultPlacementCopy.getIndex())
                                .build());
                int index = i + firstAdultIndex;
                Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
                placementIndexes[index] = adultPlacementCopy.getIndex();
            }
            // Расселяем детей на взрослые дополнительные места
            for (int index : childrenAsAdults) {
                var adultPlacementCopy = adultPlacement.toBuilder().index(resultPlacements.size()).build();
                resultPlacements.add(adultPlacementCopy);
                int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
                guestCountInfoBuilder.guestCount(GuestCount.builder()
                        .age(childAge)
                        .count(1)
                        .ageQualifyingCode("child")
                        .placementIndex(adultPlacementCopy.getIndex())
                        .build());
                Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
                placementIndexes[index] = adultPlacementCopy.getIndex();
            }
        }

        if (primary) {
            // "Излишки": бывает так, что взрослые места можно забронировать минимум на N человек, а на взрослых и
            // детей на взрослых местах требуется меньше. Тогда в эти излишки селим тех детей, которых в противном
            // случае селили бы на обычные детские места
            int excessPrimaryPlaces = adultPlacement.getCapacity() - (numAdults + childrenAsAdults.size());
            firstChildIndex = excessPrimaryPlaces;
            for (int i = 0; i < excessPrimaryPlaces; i++) {
                if (i < foundChildren.size()) {
                    int index = foundChildren.get(i).get2();
                    int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
                    guestCountInfoBuilder.guestCount(GuestCount.builder()
                            .age(childAge)
                            .count(1)
                            .ageQualifyingCode("child")
                            .placementIndex(adultPlacement.getIndex())
                            .build());
                    Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
                    placementIndexes[index] = adultPlacement.getIndex();
                }
            }
        }

        // Расселяем оставшихся детей по детским местам
        for (int i = firstChildIndex; i < foundChildren.size(); i++) {
            int index = foundChildren.get(i).get2();
            int childAge = occupancy.getChildren().get(index - occupancy.getAdults());
            Placement childPlacement = foundChildren.get(i).get1();
            var childPlacementCopy = childPlacement.toBuilder().index(resultPlacements.size()).build();
            int placementIndex = childPlacementCopy.getIndex();
            resultPlacements.add(childPlacementCopy);
            guestCountInfoBuilder.guestCount(GuestCount.builder()
                    .age(childAge)
                    .count(1)
                    .ageQualifyingCode("child")
                    .placementIndex(placementIndex)
                    .build());
            Preconditions.checkState(placementIndexes[index] == -1, "Placement index is already set");
            placementIndexes[index] = placementIndex;
        }
    }
}
